<?PHP
#
#   FILE:  Developer.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2012 Edward Almasy and Internet Scout
#   http://scout.wisc.edu
#

class Developer extends Plugin {

    function Register()
    {
        $this->Name = "Developer Support";
        $this->Version = "1.0.0";
        $this->Description = "Provides various conveniences useful during"
                ." CWIS software development.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
                "CWISCore" => "2.3.2");
        $this->EnabledByDefault = FALSE;

        $this->CfgSetup["VariableMonitorEnabled"] = array(
                "Label" => "Variable Monitor",
                "Type" => "Flag",
                "Default" => TRUE,
                "OnLabel" => "Enabled",
                "OffLabel" => "Disabled",
                "Help" => "When enabled, the Variable Monitor displays the"
                        ." global (G_), page (H_), and form (F_) variables"
                        ." available to the HTML file for the current page",
                );
        $this->CfgSetup["VariableDisplayThreshold"] = array(
                "Type" => "Number",
                "Label" => "Variable Display Threshold",
                "Default" => 300,
                "Help" => "The Variable Monitor will not attempt to display"
                        ." values where the var_dump() output for the"
                        ." value is more than this number of characters.",
                );
        $this->CfgSetup["ErrorReportingFlags"] = array(
                "Label" => "PHP Error Reporting Flags",
                "Type" => "Option",
                "AllowMultiple" => TRUE,
                "Help" => "Flags controlling PHP error reporting, as defined in the"
                        ." PHP documentation.",
                # (the indexes for "Options" are intentionally strings rather
                #       than the predefined PHP error level constants because
                #       the value of those constants can change from PHP version
                #       to version)
                "Options" => array(
                        "E_WARNING" => "Warning",
                        "E_NOTICE" => "Notice",
                        "E_STRICT" => "Strict",
                        ),
                );

        # add in E_DEPRECATED option if PHP version supports it
        if (version_compare(PHP_VERSION, "5.3.0") >= 0)
        {
            $this->CfgSetup["ErrorReportingFlags"]["Options"]["E_DEPRECATED"]
                    = "Deprecated";
        }
    }

    function HookEvents()
    {
        $Hooks = array();
        if ($this->ConfigSetting("VariableMonitorEnabled"))
        {
            $Hooks["EVENT_IN_HTML_HEADER"] = "AddVariableMonitorStyles";
            $Hooks["EVENT_PHP_FILE_LOAD_COMPLETE"] = "DisplayVariableMonitor";
        }
        return $Hooks;
    }

    function Initialize()
    {
        # set the PHP error reporting level
        $ErrorFlags = $this->ConfigSetting("ErrorReportingFlags");
        if (count($ErrorFlags))
        {
            $CurrentFlags = error_reporting();
            foreach ($ErrorFlags as $Flag)
            {
                switch ($Flag)
                {
                    case "E_WARNING":  $CurrentFlags |= E_WARNING;
                    case "E_NOTICE":  $CurrentFlags |= E_NOTICE;
                    case "E_STRICT":  $CurrentFlags |= E_STRICT;
                    case "E_DEPRECATED":  $CurrentFlags |= E_DEPRECATED;
                }
            }
            error_reporting($CurrentFlags);
        }

        # report that initialization was successful
        return NULL;
    }

    function AddVariableMonitorStyles()
    {
        ?><style type="text/css">
        .VariableMonitor {
            border:         1px solid #999999;
            background:     #E0E0E0;
            font-family:    verdana, arial, helvetica, sans-serif;
            margin-top:     10px;
            width:          100%;
        }
        .VariableMonitor th {
            padding:        5px;
            text-align:     left;
            vertical-align: center;
        }
        .VariableMonitor th span {
            float:          right;
            font-weight:    normal;
            font-style:     italic;
        }
        .VariableMonitor td {
            padding:        10px;
        }
        .VariableMonitor th {
            background:     #D0D0D0;
        }
        .VariableMonitor h2, h3 {
            margin:         0;
        }
        .VariableMonitor div {
            font-family:    Courier New, Courier, monospace;
            color:          #000000;
        }
        .VarMonValue {
            display:        none;
            background:     #F0F0F0;
            border:         1px solid #FFFFFF;
            padding:        0 10px 0 10px;
        }
        </style><?PHP
    }

    function DisplayVariableMonitor($PluginName, $Context)
    {
        # begin Variable Monitor display
        ?><table class="VariableMonitor">
        <tr><th colspan="3">
            <span>(values available at the beginning of HTML file execution)</span>
            <h2>Variable Monitor</h2>
        </th></tr>
        <tr valign="top"><td><?PHP

        # retrieve all variables
        $Vars = $Context["Variables"];

        # list page variables
        $VarIndex = 0;
        ?><h3>Page Variables (H_)</h3><div><?PHP
        $VarFound = FALSE;
        foreach ($Vars as $VarName => $VarValue)
        {
            if (preg_match("/^H_/", $VarName))
            {
                $this->DisplayVariable($VarName, $VarValue, $VarIndex);
                $VarIndex++;
                $VarFound = TRUE;
            }
        }
        if (!$VarFound) {  print "(none)<br/>";  }
        ?></div></td><td><?PHP

        # list form variables
        ?><h3>Form Variables (F_)</h3><div><?PHP
        $VarFound = FALSE;
        foreach ($GLOBALS as $VarName => $VarValue)
        {
            if (preg_match("/^F_/", $VarName))
            {
                $this->DisplayVariable($VarName, $VarValue, $VarIndex);
                $VarIndex++;
                $VarFound = TRUE;
            }
        }
        if (!$VarFound) {  print "(none)<br/>";  }
        ?></div></td><td><?PHP

        # list global variables
        ?><h3>Global Variables (G_)</h3><div><?PHP
        $VarFound = FALSE;
        foreach ($Vars as $VarName => $VarValue)
        {
            if (preg_match("/^G_/", $VarName))
            {
                $this->DisplayVariable($VarName, $VarValue, $VarIndex);
                $VarIndex++;
                $VarFound = TRUE;
            }
        }
        if (!$VarFound) {  print "(none)<br/>";  }
        ?></div><?PHP

        # end Variable Monitor display
        ?></td></tr></tables><?PHP
    }

    private function DisplayVariable($VarName, $VarValue, $VarIndex)
    {
        print "<span onclick=\"$('#VMVal".$VarIndex."').toggle()\">"
                .$VarName."</span><br/>\n"
                ."<div id='VMVal".$VarIndex."' class=\"VarMonValue\">";
        ob_start();
        var_dump($VarValue);
        $VarDump = ob_get_contents();
        ob_end_clean();
        if (strlen($VarDump) < $this->ConfigSetting("VariableDisplayThreshold"))
        {
            print $VarDump;
        }
        else
        {
            if (is_object($VarValue))
            {
                print "(<i>".get_class($VarValue)."</i> object)";
            }
            elseif (is_array($VarValue))
            {
                print "(array:".count($VarValue).")";
            }
            elseif (is_string($VarValue))
            {
                print "(string:".strlen($VarValue).")";
            }
            else
            {
                print "(value too long to display - length:".strlen($VarDump).")";
            }
        }
        print "</div>\n";
    }

}
