<?PHP

#
#   FILE:  Folder.php
#
#   Part of the Collection Workflow Information System (CWIS)
#   Copyright 2012 Edward Almasy and Internet Scout
#   http://scout.wisc.edu
#

/**
* Folder object used to create and manage groups of items.  Items are identified
* and manipulated within folders by a positive integer "ID" value.  For folders
* intended to contain multiple types of item types, item types are arbitrary
* string values.
* \nosubgrouping
*/
class Folder {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /** @name Setup/Initialization */ /*@(*/

    /**
    * Object constructor -- load an existing folder.  New folders must be created
    * with FolderFactory::CreateFolder() or FolderFactory::CreateMixedFolder().
    * @param FolderId ID of folder.
    */
    function Folder($FolderId)
    {
        # create our own DB handle
        $this->DB = new Database();

        # store folder ID
        $this->Id = intval($FolderId);

        # attempt to load in folder info
        $this->DB->Query("SELECT * FROM Folders WHERE FolderId = ".$this->Id);
        $Record = $this->DB->FetchRow();

        # if folder was not found
        if ($Record === FALSE)
        {
            # bail out with exception
            throw new Exception("Unknown Folder ID (".$FolderId.").");
        }

        # save folder info
        $this->OwnerId          = $Record["OwnerId"];
        $this->FolderName       = $Record["FolderName"];
        $this->NormalizedName   = $Record["NormalizedName"];
        $this->FolderNote       = $Record["FolderNote"];
        $this->IsShared         = $Record["IsShared"];
        $this->ContentType      = $Record["ContentType"];
        $this->UpdateValueCache = $Record;

        # load list of resources in folder from database
        $this->DB->Query("SELECT ItemId, ItemTypeId, ItemNote FROM FolderItemInts"
                ." WHERE FolderId = ".$this->Id);

        # create internal cache for item notes
        $this->ItemNoteCache = array();
        while ($Record = $this->DB->FetchRow())
        {
            $Index = self::GetCacheIndex($Record["ItemId"], $Record["ItemTypeId"]);
            $this->ItemNoteCache[$Index] = $Record["ItemNote"];
        }

        # load item ordering
        if ($this->ContentType == self::MIXEDCONTENT)
        {
            $this->OrderList = new PersistentDoublyLinkedList(
                    "FolderItemInts", "ItemId", "FolderId = ".$this->Id, "ItemTypeId");
        }
        else
        {
            $this->OrderList = new PersistentDoublyLinkedList(
                    "FolderItemInts", "ItemId", "FolderId = ".$this->Id);
        }
    }

    /**
    * Delete folder.  (Object is no longer usable after calling this method.)
    */
    function Delete()
    {
        # take folder out of global folder order
        $Factory = new FolderFactory();
        $Factory->RemoveItemFromOrder($this->Id);

        # remove resource listings from DB
        $this->DB->Query("DELETE FROM FolderItemInts WHERE FolderId = ".$this->Id);

        # remove folder listing from DB
        $this->DB->Query("DELETE FROM Folders WHERE FolderId = ".$this->Id);
    }

    /*@)*/ /* Setup/Initialization */
    # ------------------------------------------------------------------------
    /** @name Attribute Setting/Retrieval */ /*@(*/

    /**
    * Get folder ID.
    * @return Numerical folder ID.
    */
    function Id()
    {
        return $this->Id;
    }

    /**
    * Get/set folder name.  When used to set the folder name this also
    * generates and sets a new normalized folder name.
    * @return String containing folder name.
    */
    function Name($NewValue = DB_NOVALUE)
    {
        if ($NewValue != DB_NOVALUE)
        {
            $this->NormalizedName(self::NormalizeFolderName($NewValue));
        }
        return $this->UpdateValue("FolderName", $NewValue);
    }

    /**
    * Get/set normalized version of folder name.  This method can be used
    * to override the normalized name autogenerated when the folder name is
    * set with Folder::Name().
    * @param NewValue New normalized version of folder name.  (OPTIONAL)
    * @return Current normalized version of folder name.
    */
    function NormalizedName($NewValue = DB_NOVALUE)
    {
        $Name = $this->UpdateValue("NormalizedName", $NewValue);
        # attempt to generate and set new normalized name if none found
        if (!strlen($Name))
        {
            $Name = $this->UpdateValue("NormalizedName",
                    self::NormalizeFolderName($this->Name()));
        }
        return $Name;
    }

    /**
    * Convert folder name to normalized form (lower-case alphanumeric only).
    * @param Name Folder name to normalize.
    * @return String containing normalized name.
    */
    static function NormalizeFolderName($Name)
    {
        return preg_replace("/[^a-z0-9]/", "", strtolower($Name));
    }

    /**
    * Get/set whether folder is publically-viewable.  (This flag is not used at
    * all by the Folder object, but rather provided for use in interface code.)
    * @param NewValue Boolean value.  (OPTIONAL)
    * @return Boolean flag indicating whether or not folder is publically-viewable
    */
    function IsShared($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("IsShared", $NewValue);
    }

    /**
    * Get/set user ID of folder owner.
    * @param NewValue Numerical user ID.  (OPTIONAL)
    * @return ID of current folder owner.
    */
    function OwnerId($NewValue = DB_NOVALUE)
    {
        if ($NewValue !== DB_NOVALUE) {  unset($this->Owner);  }
        return intval($this->UpdateValue("OwnerId", $NewValue));
    }

    /**
    * Get/set note text for folder.
    * @param NewValue New note text.  (OPTIONAL)
    * @return String containing current note for folder.
    */
    function Note($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("FolderNote", $NewValue);
    }

    /*@)*/ /* Attribute Setting/Retrieval */
    # ------------------------------------------------------------------------
    /** @name Item Operations */ /*@(*/

    /**
    * Insert item into folder before specified item.  If the item is already
    * present in the folder, it is moved to the new location.  If the target
    * item is not found in the folder, the new item is added to the folder as
    * the first item.
    * @param TargetItemOrItemId Item to insert before.
    * @param NewItemOrItemId Item to insert.
    * @param TargetItemType Type of item to insert before.  (OPTIONAL,
    *       for mixed-item-type folders)
    * @param NewItemType Type of item to insert.  (OPTIONAL, for
    *       mixed-item-type folders)
    */
    function InsertItemBefore($TargetItemOrItemId, $NewItemOrItemId,
            $TargetItemType = NULL, $NewItemType = NULL)
    {
        $this->AddItem($NewItemOrItemId, $NewItemType);
        $this->OrderList->InsertBefore($TargetItemOrItemId, $NewItemOrItemId,
                self::GetItemTypeId($TargetItemType),
                self::GetItemTypeId($NewItemType));
    }

    /**
    * Insert item into folder after specified item.  If the item is already
    * present in the folder, it is moved to the new location.  If the target
    * item is not found, the new item is added to the folder as the last item.
    * @param TargetItemOrItemId Item to insert after.
    * @param NewItemOrItemId Item to insert.
    * @param TargetItemType Type of item to insert after.  (OPTIONAL, for
    *       mixed-item-type folders)
    * @param NewItemType Type of item to insert.  (OPTIONAL, for
    *       mixed-item-type folders)
    */
    function InsertItemAfter($TargetItemOrItemId, $NewItemOrItemId,
            $TargetItemType = NULL, $NewItemType = NULL)
    {
        $this->AddItem($NewItemOrItemId, $NewItemType);
        $this->OrderList->InsertAfter($TargetItemOrItemId, $NewItemOrItemId,
                self::GetItemTypeId($TargetItemType),
                self::GetItemTypeId($NewItemType));
    }

    /**
    * Add item to folder as the first item.  If the item is already present
    * in the folder, it is moved to be the first item.
    * @param ItemOrItemId Item to add.
    * @param ItemType Type of item to add.  (OPTIONAL, for mixed-item-type folders)
    */
    function PrependItem($ItemOrItemId, $ItemType = NULL)
    {
        $this->AddItem($ItemOrItemId, $ItemType);
        $this->OrderList->Prepend($ItemOrItemId, self::GetItemTypeId($ItemType));
    }

    /**
    * Add item to folder as the last item.  If the item is already present
    * in the folder, it is moved to be the last item.
    * @param ItemOrItemId Item to add.
    * @param ItemType Type of item to add.  (OPTIONAL, for mixed-item-type folders)
    */
    function AppendItem($ItemOrItemId, $ItemType = NULL)
    {
        $this->AddItem($ItemOrItemId, $ItemType);
        $this->OrderList->Append($ItemOrItemId, self::GetItemTypeId($ItemType));
    }

    /**
    * Retrieve array of IDs of items in folder, in the order that they appear
    * in the folder.
    * @return Array of item IDs or (if mixed-item-type list) item IDs and types.
    *       When returning IDs and types, each element in the returned array is
    *       an associative array, with the indexes "ID" and "Type".
    */
    function GetItemIds()
    {
        # retrieve item ordered list of type IDs
        $ItemIds = $this->OrderList->GetIds();

        # if this is a mixed-item-type folder
        if ($this->ContentType == self::MIXEDCONTENT)
        {
            # convert item type IDs to corresponding type names
            $NewItemIds = array();
            foreach ($ItemIds as $ItemInfo)
            {
                $NewItemIds[] = array(
                        "ID" => $ItemInfo["ID"],
                        "Type" => self::GetItemTypeName($ItemInfo["Type"]),
                        );
            }
            $ItemIds = $NewItemIds;
        }

        # return list of item type IDs (and possibly types) to caller
        return $ItemIds;
    }

    /**
    * Remove item from folder, if present.
    * @param ItemId ID of item to remove.
    * @param ItemType Type of item to be removed.  (OPTIONAL, for
    *       mixed-item-type folders)
    */
    function RemoveItem($ItemId, $ItemType = NULL)
    {
        # if resource is in folder
        if ($this->ContainsItem($ItemId, $ItemType))
        {
            # remove item from item order
            $ItemTypeId = self::GetItemTypeId($ItemType);
            $this->OrderList->Remove($ItemId, $ItemTypeId);

            # remove resource from folder locally
            unset($this->ItemNoteCache[self::GetCacheIndex($ItemId, $ItemTypeId)]);

            # remove resource from folder in DB
            $this->DB->Query("DELETE FROM FolderItemInts"
                    ." WHERE FolderId = ".intval($this->Id)
                    ." AND ItemId = ".intval($ItemId)
                    ." AND ItemTypeId = ".intval($ItemTypeId));
        }
    }

    /**
    * Get/set note text for specific item within folder.
    * @param ItemId ID of item.
    * @param NewValue New note text for item.  (OPTIONAL)
    * @param ItemType Type of item.  (OPTIONAL, for mixed-item-type folders)
    * @return String containing current note text for specified item.
    */
    function NoteForItem($ItemId, $NewValue = DB_NOVALUE, $ItemType = NULL)
    {
        $ItemTypeId = self::GetItemTypeId($ItemType);
        $Index = self::GetCacheIndex($ItemId, $ItemTypeId);
        $DummyCache = array("ItemNote" => $this->ItemNoteCache[$Index]);

        $Value = $this->DB->UpdateValue("FolderItemInts", "ItemNote", $NewValue,
                "FolderId = ".intval($this->Id)
                        ." AND ItemId = ".intval($ItemId)
                        ." AND ItemTypeId = ".intval($ItemTypeId),
                $DummyCache);

        $this->ItemNoteCache[self::GetCacheIndex($ItemId, $ItemTypeId)] = $Value;

        return $Value;
    }

    /**
    * Check whether specified item is contained in folder.
    * @param ItemId ID of item.
    * @param ItemType Type of item.  (OPTIONAL, for use with mixed-item-type folders)
    * @return TRUE if item is in folder, otherwise FALSE.
    */
    function ContainsItem($ItemId, $ItemType = NULL)
    {
        $ItemTypeId = self::GetItemTypeId($ItemType);
        return array_key_exists(self::GetCacheIndex($ItemId, $ItemTypeId),
                $this->ItemNoteCache) ? TRUE : FALSE;
    }

    /*@)*/ /* Item Operations */

    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $DB;

    private $Id;

    # folder attributes - these much match field names in Folders DB table
    private $OwnerId;
    private $FolderName;
    private $NormalizedName;
    private $FolderNote;
    private $IsShared;
    private $ContentType;

    private $ItemNoteCache;
    private $OrderList;
    private $UpdateValueCache;

    # item type IDs (indexed by normalized type name)
    static private $ItemTypeIds;
    # item type names (indexed by type ID)
    static private $ItemTypeNames;

    # content type that indicates folder contains mixed content types
    const MIXEDCONTENT = -1;

    /** @cond */
    # map item type string to numerical value
    # (not private because FolderFactory needs it)
    static function GetItemTypeId($TypeName)
    {
        # return "no type" ID if null passed in
        if ($TypeName === NULL) {  return -1;  }

        # make sure item type map is loaded
        self::LoadItemTypeMap();

        # normalize item type name
        $NormalizedTypeName = strtoupper(
                preg_replace("/[^a-zA-Z0-9]/", "", $TypeName));

        # if name not already mapped
        if (!array_key_exists($NormalizedTypeName, self::$ItemTypeIds))
        {
            # add name to database
            if (!isset($DB)) {  $DB = new Database();  }
            $DB->Query("INSERT INTO FolderContentTypes SET"
                    ." TypeName = '".addslashes($TypeName)."',"
                    ." NormalizedTypeName = '".addslashes($NormalizedTypeName)."'");

            # add name to cached mappings
            $NewTypeId = $DB->LastInsertId("FolderContentTypes");
            self::$ItemTypeIds[$NormalizedTypeName] = $NewTypeId;
            self::$ItemTypeNames[$NewTypeId] = $TypeName;
        }

        # return item type ID to caller
        return self::$ItemTypeIds[$NormalizedTypeName];
    }
    /** @endcond */

    # map item type numerical value to string
    private static function GetItemTypeName($TypeId)
    {
        # make sure item type map is loaded
        self::LoadItemTypeMap();

        # if ID not present in mappings
        if (!array_key_exists($TypeId, self::$ItemTypeNames))
        {
            # return null value
            return NULL;
        }
        else
        {
            # return item type name to caller
            return self::$ItemTypeNames[$TypeId];
        }
    }

    # load item type map from database
    private static function LoadItemTypeMap()
    {
        # if name-to-number item type map not already loaded
        if (!isset(self::$ItemTypeIds))
        {
            # load item type map from database
            $DB = new Database();
            $DB->Query("SELECT * FROM FolderContentTypes");
            self::$ItemTypeIds = array();
            self::$ItemTypeNames = array();
            while ($Row = $DB->FetchRow())
            {
                self::$ItemTypeIds[$Row["NormalizedTypeName"]] = $Row["TypeId"];
                self::$ItemTypeNames[$Row["TypeId"]] = $Row["TypeName"];
            }
        }
    }

    # add resource to folder (does not add to ordered list)
    private function AddItem($ItemOrItemId, $ItemType)
    {
        # convert item to ID if necessary
        $ItemId = is_object($ItemOrItemId)
                ? $ItemOrItemId->Id() : $ItemOrItemId;

        # convert item type to item type ID
        $ItemTypeId = self::GetItemTypeId($ItemType);

        # convert null item type to "no type" value used in database
        if ($ItemTypeId === NULL) {  $ItemTypeId = -1;  }

        # if resource is not already in folder
        if (!$this->ContainsItem($ItemId, $ItemType))
        {
            # add resource to folder locally
            $this->ItemNoteCache[self::GetCacheIndex($ItemId, $ItemTypeId)] = NULL;

            # add resource to folder in DB
            $this->DB->Query("INSERT INTO FolderItemInts SET"
                    ." FolderId = ".intval($this->Id)
                    .", ItemId = ".intval($ItemId)
                    .", ItemTypeId = ".intval($ItemTypeId));
        }
    }

    # get index to be used with item note cache array
    private static function GetCacheIndex($ItemId, $ItemTypeId)
    {
        $ItemTypeId = ($ItemTypeId === NULL) ? -1 : $ItemTypeId;
        return intval($ItemTypeId).":".intval($ItemId);
    }

    # get/set value in database
    private function UpdateValue($FieldName, $NewValue)
    {
        $this->$FieldName = $this->DB->UpdateValue("Folders", $FieldName, $NewValue,
                "FolderId = ".$this->Id, $this->UpdateValueCache);
        return $this->$FieldName;
    }
}


?>
