<?PHP

#
#   FILE:  FolderFactory.php
#
#   Part of the Collection Workflow Information System (CWIS)
#   Copyright 2012 Edward Almasy and Internet Scout
#   http://scout.wisc.edu
#

/**
* Factory object for Folder class, used to retrieve and manage Folders
* and groups of Folders.
*/
class FolderFactory extends ItemFactory {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Constructor for FolderFactory.
    * @param OwnerId ID of owner of folders to be manipulated by factory.
    *       If specified then all operations pertain only to folders with
    *       the specified owner.  (OPTIONAL)
    */
    function FolderFactory($OwnerId = NULL)
    {
        # set up item factory base class
        $this->ItemFactory("Folder", "Folders", "FolderId", "FolderName", NULL, TRUE);

        # set up
        if ($OwnerId !== NULL)
        {
            $this->OwnerId = intval($OwnerId);
            $this->SetOrderOpsCondition("OwnerId = ".$this->OwnerId);
        }
    }

    /**
    * Create new folder that will contain only one type of item.
    * @param ItemType Type of item that folder will contain.
    * @param FolderName String containing name of folder.  (OPTIONAL)
    * @param OwnerId Numerical ID of folder owner.  (OPTIONAL)
    * @return New folder (object).
    */
    function CreateFolder($ItemType, $FolderName = NULL, $OwnerId = NULL)
    {
        # retrieve numerical item type
        $ItemTypeId = ($ItemType === Folder::MIXEDCONTENT)
                ? $ItemType : Folder::GetItemTypeId($ItemType);

        # use default owner if available and none specified
        if (($OwnerId === NULL) & ($this->OwnerId !== NULL))
                {  $OwnerId = $this->OwnerId;  }

        # add new folder to database
        $this->DB->Query("INSERT INTO Folders SET"
                ." ContentType = ".$ItemTypeId
                .($FolderName ? ", FolderName = '".addslashes($FolderName)."'" : "")
                .(($OwnerId !== NULL) ? ", OwnerId = ".intval($OwnerId) : ""));

        # retrieve ID of new folder
        $Id = $this->DB->LastInsertId("Folders");

        # create new folder object and return it to caller
        return new Folder($Id);
    }

    /**
    * Create new folder that can contain multiple types of items.
    * @param FolderName String containing name of folder.  (OPTIONAL)
    * @param OwnerId Numerical ID of folder owner.  (OPTIONAL)
    * @return New folder (object).
    */
    function CreateMixedFolder($FolderName = NULL, $OwnerId = NULL)
    {
        # create new mixed-content folder and return it to caller
        return $this->CreateFolder(Folder::MIXEDCONTENT, $FolderName, $OwnerId);
    }

    /**
    * Get total number of folders currently existing.
    * @return Number of folders.
    */
    function GetFolderCount()
    {
        return $this->GetItemCount(isset($this->OwnerId)
                ? "OwnerId = ".intval($this->OwnerId) : NULL);
    }

    /*
    * Retrieve folder with specified normalized name (as generated by
    * Folder::NormalizeFolderName() method).
    * @param Normalized folder name.
    * @param OwnerId ID of folder owner.  (OPTIONAL)
    * @return Folder object or NULL if no folder found.  If multiple folders
    *       with the specified name are found, the one with the lowest folder
    *       ID is returned.
    */
    function GetFolderByNormalizedName($NormalizedName, $OwnerId = NULL)
    {
        # use default owner if available and none specified
        if (($OwnerId === NULL) & ($this->OwnerId !== NULL))
                {  $OwnerId = $this->OwnerId;  }

        # query database for folder ID
        $FolderId = $this->DB->Query("SELECT FolderId FROM Folders"
                    ." WHERE NormalizedName = '".addslashes($NormalizedName)."'"
                    .(($OwnerId !== NULL) ? " AND OwnerId = ".$this->OwnerId : "")
                    ." ORDER BY FolderId ASC",
                "FolderId");

        # if folder found with specified name and owner
        if ($FolderId !== FALSE)
        {
            # create folder object and return it to caller
            return new Folder($FolderId);
        }
        else
        {
            # return NULL to caller to indicate folder not found
            return NULL;
        }
    }

    /**
    * Retrieve folders containing specified item.
    * @param Item Item object (must have Id() method) or item ID.
    * @param int|string $ItemType the item type
    * @param int $OwnerId optional owner ID to restrict folders to
    * @param SharedFoldersOnly Whether to only return shared folders.
    * @return Array of Folder objects that contain specified item.
    */
    function GetFoldersContainingItem($Item, $ItemType, $OwnerId = NULL, $SharedFoldersOnly = FALSE)
    {
        # assume we won't find any folders
        $Folders = array();

        # retrieve item ID
        $ItemId = is_object($Item) ? $Item->Id() : $Item;

        # retrieve numerical item type
        $ItemTypeId = ($ItemType === Folder::MIXEDCONTENT)
                ? $ItemType : Folder::GetItemTypeId($ItemType);

        # use default owner if available and none specified
        if (($OwnerId === NULL) & ($this->OwnerId !== NULL))
                {  $OwnerId = $this->OwnerId;  }

        # query database for IDs of all folders that contain item
        $this->DB->Query("
            SELECT DISTINCT FolderItemInts.FolderId
            FROM FolderItemInts
            LEFT JOIN Folders
            ON FolderItemInts.FolderId = Folders.FolderId
            WHERE FolderItemInts.ItemId = '".intval($ItemId)."'
            AND (FolderItemInts.ItemTypeId = '".intval($ItemTypeId)."'
                 OR Folders.ContentType = '".intval($ItemTypeId)."')
            ".(($OwnerId !== NULL) ? " AND Folders.OwnerId = ".intval($OwnerId) : "")."
            ".(($SharedFoldersOnly) ? " AND Folders.IsShared = 1" : ""));
        $FolderIds = $this->DB->FetchColumn("FolderId");

        # create array of folders from folder IDs
        foreach ($FolderIds as $Id)
        {
            $Folders[$Id] = new Folder($Id);
        }

        # return folders (if any) to caller
        return $Folders;
    }

    /**
    * Retrieve folders with specified name, owner, or default content type.  If
    * no parameters are specified, all existing folders are returned.  If no owner
    * ID parameter is supplied and an owner ID was specified for
    * FolderFactory::FolderFactory(), then that owner ID is used.
    * @param ItemType String containing type of item to search for as default
    *       content type of folder.  To search for only mixed-content-type
    *       folders specify Folder::MIXEDCONTENT.  (OPTIONAL, defaults to NULL)
    * @param OwnerId Numerical ID of folder owner.  (OPTIONAL, defaults to NULL)
    * @param Name String containing target folder name.  (OPTIONAL, defaults to NULL)
    * @param Offset Zero-based offset into list of folders.  (OPTIONAL)
    * @param Count Number of folders to retrieve beginning at specified
    *       offset.  (OPTIONAL)
    * @return Array of Folder objects that match specified parameters.
    */
    function GetFolders($ItemType = NULL, $OwnerId = NULL, $Name = NULL,
            $Offset = 0, $Count = NULL)
    {
        # retrieve numerical item type
        $ItemTypeId = ($ItemType === Folder::MIXEDCONTENT)
                ? $ItemType : Folder::GetItemTypeId($ItemType);

        # retrieve IDs of all folders that match specified parameters
        $Condition = ($ItemTypeId !== NULL) ? "ContentType = ".intval($ItemTypeId) : NULL;
        if (($OwnerId !== NULL) || ($this->OwnerId !== NULL))
        {
            $Condition .= ($Condition ? " AND " : "")."OwnerId = "
                    .intval(($OwnerId !== NULL) ? $OwnerId : $this->OwnerId);
        }
        if ($Name !== NULL)
        {
            $Condition .= ($Condition ? " AND " : "")."FolderName = '"
                    .addslashes($Name)."'";
        }
        $FolderIds = $this->GetItemIds($Condition);

        # pare down list to requested range
        if ($Offset || $Count)
        {
            $FolderIds = $Count ? array_slice($FolderIds, $Offset, $Count)
                    : array_slice($FolderIds, $Offset);
        }

        # create array of folders based on IDs
        $Folders = array();
        foreach ($FolderIds as $FolderId)
        {
            $Folders[$FolderId] = new Folder($FolderId);
        }

        # return folders (if any) to caller
        return $Folders;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $OwnerId;
}


?>
