<?PHP
#
#   FILE:  MetadataFieldOrdering.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2012-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

# ----- EXPORTED FUNCTIONS ---------------------------------------------------

/**
 * Print the items in the given metadata field ordering.
 * @param MetadataFieldOrder $Order metadata field ordering
 * @param callback $Include callback for filtering fields and groups
 * @return void
 */
function PrintFieldOrdering(MetadataFieldOrder $Order, $Include=NULL)
{
    $Items = $Order->GetItems();
    $Place = array("Order" => $Order);

    # filter items if requested
    if (is_callable($Include))
    {
        $Items = IncludeOrderItems($Items, $Include);
    }

    for ($i = 0; $i < count($Items); $i++)
    {
        $Item = $Items[$i];
        $Place["Enclosure"] = $Order;
        $Place["Previous"] = isset($Items[$i-1]) ? $Items[$i-1] : NULL;
        $Place["Next"] = isset($Items[$i+1]) ? $Items[$i+1] : NULL;

        if ($Item instanceof MetadataField)
        {
            PrintFieldInOrder($Item, $Place);
        }

        else if ($Item instanceof MetadataFieldGroup)
        {
            PrintGroupInOrder($Item, $Place);
        }
    }
}

/**
 * Print the fields in the given metadata field group.
 * @param MetadataFieldGroup $Group metadata field group
 * @param array $Place information about where the group is in the order
 * @param callback $Include callback for filtering fields and groups
 * @return void
 */
function PrintGroupItems(MetadataFieldGroup $Group, array $Place, $Include=NULL)
{
    $Fields = $Group->GetFields();
    $GroupPlace = array(
        "Order" => $Place["Order"],
        "Enclosure" => $Group);

    # filter items if requested
    if (is_callable($Include))
    {
        $Fields = IncludeOrderItems($Fields, $Include);
    }

    for ($i = 0; $i < count($Fields); $i++)
    {
        $Field = $Fields[$i];
        $GroupPlace["Previous"] = isset($Fields[$i-1]) ? $Fields[$i-1] : NULL;
        $GroupPlace["Next"] = isset($Fields[$i+1]) ? $Fields[$i+1] : NULL;

        PrintFieldInOrder($Field, $GroupPlace);
    }
}

# ----- LOCAL FUNCTIONS ------------------------------------------------------

/**
 * Remove items from the given array if the given filter callback specifies to
 * do so, maintaining continuous indices.
 * @param array $Items array of items
 * @param callack $Callback callback to determine which items should be included
 * @return array the array with only items that should be included
 */
function IncludeOrderItems(array $Items, $Callback)
{
    $Items = array_filter($Items, $Callback);

    # maintain continuous indices
    ksort($Items);
    return array_values($Items);
}

# ----- MAIN -----------------------------------------------------------------

global $DisplayOrder;
global $EditOrder;
global $H_Schema;

PageTitle("Metadata Field Ordering");
CheckAuthorization(PRIV_SYSADMIN, PRIV_COLLECTIONADMIN);

# get the schema ID or use the default one if not specified
$SchemaId = GetArrayValue($_GET, "SC", MetadataSchema::SCHEMAID_DEFAULT);

# construct the metadata schema
$H_Schema = new MetadataSchema($SchemaId);

# construct the order objects
$DisplayOrder = $H_Schema->GetDisplayOrder();
$EditOrder = $H_Schema->GetEditOrder();

# in case a bug results in any issues, try to fix them
$DisplayOrder->MendIssues();
$EditOrder->MendIssues();
