<?PHP
#
#   FILE:  AuthorAutofill.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2002-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Allows user information to be automatically copied into designated metadata
* fields in new resource records.
*/
class AuthorAutofill extends Plugin {

    /**
    * Register information about this plugin.
    */
    function Register()
    {
        $this->Name = "Author Autofill";
        $this->Version = "1.0.0";
        $this->Description = "Allows user information to be automatically"
                ." copied into designated metadata fields in new records";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
                "CWISCore" => "2.2.2");
        $this->EnabledByDefault = FALSE;

        $UserInfoCfgTemplate = array(
                "Type" => "Option",
                "Label" => "User Info",
                "Help" => "User info element to copy.",
                "Options" => array(
                        "" => "--",
                        "RealName" => "Real Name",
                        "UserName" => "User Name",
                        "EMail" => "Email",
                        "WebSite" => "Web Site",
                        "AddressLineOne" => "Address Line One",
                        "AddressLineTwo" => "Address Line Two",
                        "City" => "City",
                        "State" => "State",
                        "ZipCode" => "Zip Code",
                        "Country" => "Country",
                        "User" => "User (for User metadata fields)",
                        "Other" => "Other (enter User object value name below)",
                        ),
                );
        $MetadataFieldCfgTemplate = array(
                "Type" => "MetadataField",
                "Label" => "Metadata Field",
                "Help" => "Metadtata field to copy into.",
                "FieldTypes" => MetadataSchema::MDFTYPE_TEXT
                        | MetadataSchema::MDFTYPE_PARAGRAPH
                        | MetadataSchema::MDFTYPE_USER
                        | MetadataSchema::MDFTYPE_CONTROLLEDNAME
                );
        $OtherNameCfgTemplate = array(
                "Type" => "Text",
                "Label" => "Other Value Name",
                "Help" => "Name of User object value to copy value from"
                        ." (when User Info is set to 'Other').",
                );
        $HeadingTemplate = array(
                "Type" => "Heading",
                "Label" => "Autofill Value",
                );
        $this->CfgSetup["Heading1"] = $HeadingTemplate;
        $this->CfgSetup["UserInfo1"] = $UserInfoCfgTemplate;
        $this->CfgSetup["MetadataField1"] = $MetadataFieldCfgTemplate;
        $this->CfgSetup["OtherName1"] = $OtherNameCfgTemplate;
        $this->CfgSetup["Heading2"] = $HeadingTemplate;
        $this->CfgSetup["UserInfo2"] = $UserInfoCfgTemplate;
        $this->CfgSetup["MetadataField2"] = $MetadataFieldCfgTemplate;
        $this->CfgSetup["OtherName2"] = $OtherNameCfgTemplate;
        $this->CfgSetup["Heading3"] = $HeadingTemplate;
        $this->CfgSetup["UserInfo3"] = $UserInfoCfgTemplate;
        $this->CfgSetup["MetadataField3"] = $MetadataFieldCfgTemplate;
        $this->CfgSetup["OtherName3"] = $OtherNameCfgTemplate;
        $this->Instructions = <<<EOT
            The <i>Author Autofill</i> plugin automatically copies user information
            into specified metadata fields in new resources records when they are
            created.
            <br/><br/>
            Up to three values may be automatically copied.  For each value you want
            copied, set <i>User Info</i> to the appropriate value from the user
            signup information and the corresponding <i>Metadata Field</i> to the
            metadata field you want it copied into.
            <br/><br/>
            If you want to copy a non-standard
            user signup value, select <i>Other</i> for the <i>User Info</i> setting
            and enter the user info value name (as passed to <code>CWUser::Set()</code>
            in PHP) into <i>Other Value Name</i>.
EOT;
    }

    /**
    * Hook the events into the application framework.
    * @return Returns an array of events to be hooked into the application
    *      framework.
    */
    function HookEvents()
    {
        return array(
                "EVENT_RESOURCE_CREATE" => "AutofillValues",
                );
    }

    /**
    * Automatically fill the designated fields of the resource with the
    * designated user information.
    * @param Resource $Resource Resource object to automatically fill with user
    *      information.
    */
    function AutofillValues($Resource)
    {
        # ignore the resource if it isn't part of the default schema
        if ($Resource->SchemaId() != MetadataSchema::SCHEMAID_DEFAULT)
        {
            return;
        }

        # for each possible autofill value
        for ($Index = 1;  $Index <= 3;  $Index++)
        {
            # retrieve configuration values
            $UserInfo = $this->ConfigSetting("UserInfo".$Index);
            $FieldId = $this->ConfigSetting("MetadataField".$Index);
            $OtherName = $this->ConfigSetting("OtherName".$Index);

            # if value has metaadata field assigned to it
            if ($FieldId !== NULL)
            {
                # retrieve current user (if we haven't done so yet)
                if (!isset($User)) {  $User = $GLOBALS["G_User"];  }

                # retrieve metadata field
                $Schema = new MetadataSchema();
                $Field = $Schema->GetField($FieldId);

                # if metadata field is valid
                if ($Field !== NULL)
                {
                    # if actual user was specified for user info
                    if ($UserInfo == "User")
                    {
                        # if metadata field is User field
                        if ($Field->Type() == MetadataSchema::MDFTYPE_USER)
                        {
                            # set user value in resource
                            $Resource->Set($Field, $User);
                        }
                    }
                    # else if non-standard user info value was specified
                    elseif ($UserInfo == "Other")
                    {
                        # if other value name was supplied
                        if (strlen(trim($OtherName)))
                        {
                            # retrieve value from user
                            $Value = $User->Get($OtherName);

                            # set user value in resource
                            $Resource->Set($Field, $Value);
                        }
                    }
                    elseif ($UserInfo != "--")
                    {
                        $DataToSet = $User->Get($UserInfo);


                        if ($Field->Type() == MetadataSchema::MDFTYPE_CONTROLLEDNAME)
                        {
                            # Check to see if there is data for this field before trying to create a CName
                            if (strlen($DataToSet)>0)
                            {
                                # Check to see if the desired CName exists, create it if not.
                                $CNameIds = ControlledName::SearchForControlledName($DataToSet, $Field->Id());
                                if (empty($CNameIds))
                                {
                                    $CName = new ControlledName( NULL, $DataToSet, $Field->Id() );
                                }
                                else
                                {
                                    $CNameId = array_pop($CNameIds);
                                    $CName = new ControlledName($CNameId);
                                }

                                $Resource->Set($Field, $CName->Id());
                            }
                        }
                        else
                        {
                            # set value in resource
                            $Resource->Set($Field, $DataToSet);
                        }
                    }
                }
            }
        }
    }

}
