<?PHP
#
#   FILE:  Qualifier.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2012-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

class Qualifier
{

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
     * Status code used for an okay, valid qualifier.
     */
    const STATUS_OK = 0;

    /**
     * Status code used for a non-existent qualifier.
     */
    const STATUS_DOES_NOT_EXIST = 1;

    /**
     * Load the qualifier with the given ID or create a new qualifier if no ID
     * is given.
     * @param int $QualifierId qualifier ID
     */
    public function Qualifier($QualifierId = NULL)
    {
        $this->DB = new Database();

        # assume the qualifier operations will be successful
        $this->Status = self::STATUS_OK;

        # no ID given
        if (func_num_args() == 0)
        {
            # determine next qualifier ID
            $HighestId = $this->DB->Query("
                SELECT QualifierId FROM Qualifiers
                ORDER BY QualifierId DESC
                LIMIT 1",
                "QualifierId");
            $this->Id = $HighestId > 0 ? $HighestId + 1 : 1;

            # add record to database with that ID
            $this->DB->Query("
                INSERT INTO Qualifiers
                SET QualifierId = ".addslashes($this->Id));
        }

        # if ID supplied
        else if (!is_null($QualifierId))
        {
            $this->Id = intval($QualifierId);

            # attempt to load qualifier info from database
            $this->DB->Query("
                SELECT * FROM Qualifiers
                WHERE QualifierId = '".addslashes($this->Id)."'");

            # if the qualifier was found
            if ($this->DB->NumRowsSelected() > 0)
            {
                # set attributes to values returned by database
                $this->DBFields = $this->DB->FetchRow();
            }

            # the qualifier was not found
            else
            {
                $this->Status = self::STATUS_DOES_NOT_EXIST;
            }
        }

        # null given
        else
        {
            $this->Status = self::STATUS_DOES_NOT_EXIST;
        }
    }

    /**
     * Get the status of this qualifier.
     * @return int status code
     */
    public function Status()
    {
        return $this->Status;
    }

    /**
     * Delete the qualifier if in a valid state.
     * @return void
     */
    public function Delete()
    {
        # do not try deleting a qualifier with a bad status
        if ($this->Status != self::STATUS_OK)
        {
            return;
        }

        # delete record from database
        $this->DB->Query("
            DELETE FROM Qualifiers
            WHERE QualifierId = ".addslashes($this->Id));

        # update status
        $this->Status = self::STATUS_DOES_NOT_EXIST;
    }

    /**
     * Get the qualifier ID.
     * @return int qualifier ID
     */
    public function Id()
    {
        return $this->Id;
    }

    /**
     * Get or set the qualifier name.
     * @param string $NewValue optional new qualifier name
     * @return string the current qualifier name
     */
    public function Name($NewValue=DB_NOVALUE)
    {
        return $this->UpdateValue("QualifierName", $NewValue);
    }

    /**
     * Get or set the qualifier namespace.
     * @param string $NewValue optional new qualfier namespace
     * @return string the current qualifier namespace
     */
    public function NSpace($NewValue=DB_NOVALUE)
    {
        return $this->UpdateValue("QualifierNamespace", $NewValue);
    }

    /**
     * Get or set the qualifier URL.
     * @param string $NewValue optional new qualifier URL
     * @return string the current qualifier URL
     */
    public function Url($NewValue=DB_NOVALUE)
    {
        return $this->UpdateValue("QualifierUrl", $NewValue);
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    /**
     * The ID of the qualifier.
     * @var int $Id
     */
    protected $Id;

    /**
     * The status code of the qualifier.
     * @var int $Status
     */
    protected $Status;

    /**
     * The database object.
     * @var Database $DB
     */
    protected $DB;

    /**
     * Cached qualifier data from the database
     * @var array $DBFields
     */
    protected $DBFields;

    /**
     * Convenience method to supply parameters to Database::UpdateValue().
     * @param string $FieldName database field name
     * @param mixed $NewValue new value for the field or DB_NOVALUE
     * @return mixed the current value for the field name
     */
    protected function UpdateValue($FieldName, $NewValue)
    {
        return $this->DB->UpdateValue(
            "Qualifiers",
            $FieldName,
            $NewValue,
            "QualifierId = '".addslashes($this->Id)."'",
            $this->DBFields);
    }

}
