<?PHP

/**
 * Class used to create a namespace for common Folders plugin functions.
 */
class Folders_Common
{

    /**
     * The maximum number of resources allowed per add.
     */
    const MAX_RESOURCES_PER_ADD = PHP_INT_MAX;

    /**
     * Initialize static data members.
     * @return void
     */
    public static function construct()
    {
        self::$Schema = new MetadataSchema();
    }

    /**
     * Setup page completion by suppressing HTML output and/or redirecting to
     * a certain page based on the GET parameters sent.
     * @param string $DefaultReturnTo optional default value to use for redirect
     * @param boolean $DefaultSuppressHtmlOutput option suppression value
     * @return boolean TRUE if page completion should be allowed to continue
     */
    public static function ApiPageCompletion(
        $DefaultReturnTo=NULL, $DefaultSuppressHtmlOutput=FALSE)
    {
        global $AF;

        # first check that the user is logged in
        if (!CheckAuthorization())
        {
            return FALSE;
        }

        $ReturnTo = GetArrayValue($_GET, "ReturnTo", $DefaultReturnTo);
        $SuppressHtmlOutput = GetArrayValue(
            $_GET,
            "SuppressHtmlOutput",
            $DefaultSuppressHtmlOutput);

        # set a redirect if given and it's safe to do so
        if ($ReturnTo && IsSafeRedirectUrl($ReturnTo))
        {
            $AF->SetJumpToPage($ReturnTo);
        }

        if ($SuppressHtmlOutput)
        {
            $AF->SuppressHTMLOutput();

            # don't redirect if suppressing HTML output
            $AF->SetJumpToPage(NULL);
        }

        return TRUE;
    }

    /**
     * Get the safe, i.e., OK to print to HTML, version of the given resource's
     * title.
     * @param Resource $Resource resource object
     * @return string safe resource title
     */
    public static function GetSafeResourceTitle(Resource $Resource)
    {
        return self::EscapeResourceTitle($Resource->GetMapped("Title"));
    }

    /**
     * Get the safe, i.e., OK to print to HTML, version of the given resource
     * title.
     * @param string $Title resource title
     * @return string safe resource title
     */
    public static function EscapeResourceTitle($Title)
    {
        $TitleField = self::$Schema->GetFieldByMappedName("Title");
        $SafeTitle = $Title;

        if (!$TitleField->AllowHTML())
        {
            $SafeTitle = defaulthtmlentities($SafeTitle);
        }

        return $SafeTitle;
    }

    /**
     * Get the share URL for the given folder.
     * @param Folder $Folder folder
     * @return string share URL for the folder
     */
    public static function GetShareUrl(Folder $Folder)
    {
        global $AF;

        $Id = $Folder->Id();
        $ShareUrl = OurUrl() . "?P=P_Folders_ViewFolder&FolderId=" . $Id;

        # make the share URL prettier if .htaccess support exists
        # (folders/folder_id/normalized_folder_name)
        if ($AF->HtaccessSupport())
        {
            $PaddedId = str_pad($Id, 4, "0", STR_PAD_LEFT);
            $NormalizedName = $Folder->NormalizedName();

            $ShareUrl = OurBaseUrl() . "folders/";
            $ShareUrl .= $PaddedId . "/" . $NormalizedName;
        }

        return $ShareUrl;
    }

    /**
     * @var MetadataSchema $Schema metadata schema object
     */
    protected static $Schema;

} Folders_Common::construct();
