<?PHP
#
#   FILE:  SearchParameterSet.php
#
#   Part of the ScoutLib application support library
#   Copyright 2015 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu
#

/**
* Set of parameters used to perform a search.
*/
class SearchParameterSet {

    # ---- SETUP / CONFIGURATION  ---------------------------------------------
    /** @name Setup / Configuration  */ /*@(*/

    /**
    * Class constructor, used to create a new set or reload an existing
    * set from previously-constructed data.
    * @param string $Data Existing search parameter set data, previously
    *       retrieved with SearchParameterSet::Data().  (OPTIONAL)
    * @see SearchParameterSet::Data()
    */
    function __construct($Data = NULL)
    {
        # if set data supplied
        if ($Data !== NULL)
        {
            # set internal values from data
            $this->LoadFromData($Data);
        }
    }

    /**
    * Register function used to retrieve a canonical value for a field.  This
    * This function should accept a single mixed parameter and return a canonical
    * integer value for the field.
    * @param callable $Func Function to call.
    * @throws InvalidArgumentException if function supplied that is not callable.
    */
    static function SetCanonicalFieldFunction($Func)
    {
        if (is_callable($Func))
        {
            self::$CanonicalFieldFunction = $Func;
        }
        else
        {
            throw new InvalidArgumentException("Invalid function supplied.");
        }
    }

    /**
    * Register function used to retrieve a printable value for a field.
    * This function should accept a single mixed parameter and return a
    * human-readable name for the field.
    * @param callable $Func Function to call.
    * @throws InvalidArgumentException if function supplied that is not callable.
    */
    static function SetPrintableFieldFunction($Func)
    {
        if (is_callable($Func))
        {
            self::$PrintableFieldFunction = $Func;
        }
        else
        {
            throw new InvalidArgumentException("Invalid function supplied.");
        }
    }

    /**
    * Get/set URL parameter prefix.
    */


    /*@)*/
    # ---- SET CONSTRUCTION ---------------------------------------------------
    /** @name Set Construction  */ /*@(*/

    /**
    * Add search parameter to set.  If a canonical field function is set,
    * the field to search can be anything accepted by that function, otherwise
    * the $Field argument must be a type usable as an array index (e.g. an
    * integer or a string).
    * @param mixed $SearchStrings String or array of strings to search for.
    * @param mixed $Field Field to search.  (OPTIONAL – defaults to
    *       keyword search if no field specified)
    * @see SearchParameterSet::SetCanonicalFieldFunction()
    */
    function AddParameter($SearchStrings, $Field = NULL)
    {
        # normalize field value if supplied
        if (($Field !== NULL) && isset(self::$CanonicalFieldFunction))
        {
            $Field = call_user_func(self::$CanonicalFieldFunction, $Field);
        }

        # make sure search strings are an array
        if (!is_array($SearchStrings))
                {  $SearchStrings = array($SearchStrings);  }

        # for each search string
        foreach ($SearchStrings as $String)
        {
            # if field specified
            if ($Field !== NULL)
            {
                # add strings to search values for field
                $this->SearchStrings[$Field][] = $String;
            }
            else
            {
                # add strings to keyword search values
                $this->KeywordSearchStrings[] = $String;
            }
        }
    }

    /**
    * Get/set logic for set.
    * @param string $NewValue New setting, either "AND" or "OR".  (OPTIONAL)
    * @return string Current logic setting..
    * @throws InvalidArgumentException if new setting is invalid.
    */
    function Logic($NewValue = NULL)
    {
        # if new value supplied
        if ($NewValue !== NULL)
        {
            # normalize value
            $NormValue = strtoupper($NewValue);

            # error out if value appears invalid
            if (($NormValue !== "AND") && ($NormValue !== "OR"))
            {
                throw new InvalidArgumentException("New logic setting"
                        ." is invalid (".$NewValue.").");
            }

            # save new setting
            $this->Logic = $NormValue;
        }

        # return current logic setting to caller
        return $this->Logic;
    }

    /**
    * Add subgroup of search parameters to set.
    * @param SearchParameterSet $Set Subgroup to add.
    */
    function AddSet(SearchParameterSet $Set)
    {
        # add subgroup to privilege set
        $this->Subgroups[] = $Set;
    }


    /*@)*/
    # ---- DATA TRANSLATION ---------------------------------------------------
    /** @name Data Translation  */ /*@(*/

    /**
    * Get/set search parameter set data, in the form of an opaque string.
    * This method can be used to retrieve an opaque string containing
    * set data, which can then be saved (e.g. to a database) and later used
    * to reload a search parameter set.  (Use instead of serialize() to avoid
    * future issues with internal class changes.)
    * @param string $NewValue New search parameter set data.  (OPTIONAL)
    * @return string Current search parameter set data (opaque value).
    * @throws InvalidArgumentException if incoming set data appears invalid.
    */
    function Data($NewValue = NULL)
    {
        # if new data supplied
        if ($NewValue !== NULL)
        {
            # unpack set data and load
            $this->LoadFromData($NewValue);
        }

        # serialize current data and return to caller
        $Data = array();
        if ($this->Logic !== "AND") {  $Data["Logic"] = $this->Logic;  }
        if (count($this->SearchStrings))
                {  $Data["SearchStrings"] = $this->SearchStrings;  }
        if (count($this->KeywordSearchStrings))
        {
            $Data["KeywordSearchStrings"] = $this->KeywordSearchStrings;
        }
        if (count($this->Subgroups))
        {
            foreach ($this->Subgroups as $Subgroup)
            {
                $Data["Subgroups"][] = $Subgroup->Data();
            }
        }
        return serialize($Data);
    }

    /*
    * Get/set search parameter set, in the form of URL parameters.
    * @param string $NewValue New parameter set in the form of a URL
    *       parameter string.  (OPTIONAL)
    * @return array URL parameter values, with parameter names for the index..
    */
    function UrlParameters($NewValue = NULL)
    {
        # if new value supplied
        if ($NewValue !== NULL)
        {
            # set new parameters
            $this->SetFromUrlParameters($NewValue);
        }

        # get existing search parameters as URL parameters
        $Params = $this->GetAsUrlParameters();

        # sort parameters by parameter name to normalize result
        ksort($Params);

        # return parameters to caller
        return $Params;
    }

    /*
    * Get/set search parameter set, in the form of an URL parameter string.
    * @param string $NewValue New parameter set in the form of a URL
    *       parameter string.  (OPTIONAL)
    * @return string URL parameter string.
    */
    function UrlParameterString($NewValue = NULL)
    {
        # get/set parameters
        $Params = $this->UrlParameters($NewValue);

        # combine values into string
        $ParamString = "";
        $Separator = "";
        foreach ($Params as $Index => $Value)
        {
            $ParamString .= $Separator.$Index."=".urlencode($Value);
            $Separator = "&";
        }

        # return string to caller
        return $ParamString;
    }

    /*
    * Get text description of search parameter set.
    * @param bool $IncludeHtml Whether to include HTML tags for formatting.
    *       (OPTIONAL, defaults to TRUE)
    * @param bool $StartWithBreak Whether to start string with BR tag.
    *       (OPTIONAL, defaults to TRUE)
    * @param int $TruncateLongWordsTo Number of characters to truncate long
    *       words to (use 0 for no truncation).  (OPTIONAL, defaults to 0)
    * @param string $Indent (for internal (recursive) use only)
    * @return string Text description of search parameters.
    */
    function TextDescription($IncludeHtml = TRUE, $StartWithBreak = TRUE,
            $TruncateLongWordsTo = 0, $Indent = "")
    {
        # define list of phrases used to represent logical operators
        $OperatorPhrases = array(
                "=" => "is",
                "==" => "is",
                ">" => "is greater than",
                "<" => "is less than",
                ">=" => "is at least",
                "<=" => "is no more than",
                "!" => "is not",
                "!=" => "is not",
                );

        # set characters used to indicate literal strings
        $LiteralStart = $IncludeHtml ? "<i>" : "\"";
        $LiteralEnd = $IncludeHtml ? "</i>" : "\"";
        $LiteralBreak = $IncludeHtml ? "<br>\n" : "\n";
        $Indent .= $IncludeHtml ? "&nbsp;&nbsp;&nbsp;&nbsp;" : "  ";

        # for each keyword search string
        $Descriptions = array();
        foreach ($this->KeywordSearchStrings as $SearchString)
        {
            # escape search string if appropriate
            if ($IncludeHtml)
            {
                $SearchString = defaulthtmlentities($SearchString);
            }

            # add string to list of descriptions
            $Descriptions[] = $LiteralStart.$SearchString.$LiteralEnd;
        }

        # for each field with search strings
        foreach ($this->SearchStrings as $FieldId => $SearchStrings)
        {
            # retrieve field name
            $FieldName = call_user_func(self::$PrintableFieldFunction, $FieldId);

            # for each search string
            foreach ($SearchStrings as $SearchString)
            {
                # extract operator from search string
                $MatchResult = preg_match("/^([=><!]+)(.+)/",
                        $SearchString, $Matches);

                # determine operator phrase
                if (($MatchResult == 1) && isset($OperatorPhrases[$Matches[1]]))
                {
                    $OpPhrase = $OperatorPhrases[$Matches[1]];
                    $SearchString = $Matches[2];
                }
                else
                {
                    $OpPhrase = "contains";
                }

                # escape field name and search string if appropriate
                if ($IncludeHtml)
                {
                    $FieldName = defaulthtmlentities($FieldName);
                    $SearchString = defaulthtmlentities($SearchString);
                }

                # assemble field and operator and value into description
                $Descriptions[] = $FieldName." ".$OpPhrase." "
                        .$LiteralStart.$SearchString.$LiteralEnd;
            }
        }

        # for each subgroup
        foreach ($this->Subgroups as $Subgroup)
        {
            # retrieve description for subgroup
            $Descriptions[] = "(".$Subgroup->TextDescription($IncludeHtml,
                    $StartWithBreak, $TruncateLongWordsTo, $Indent).")";
        }

        # join descriptions with appropriate conjunction
        $Descrip = join($LiteralBreak.$Indent." ".strtolower($this->Logic)." ",
                $Descriptions);

        # if caller requested that long words be truncated
        if ($TruncateLongWordsTo > 4)
        {
            # break description into words
            $Words = explode(" ", $Descrip);

            # for each word
            $NewDescrip = "";
            foreach ($Words as $Word)
            {
                # if word is longer than specified length
                if (strlen(strip_tags($Word)) > $TruncateLongWordsTo)
                {
                    # truncate word and add ellipsis
                    $Word = NeatlyTruncateString($Word, $TruncateLongWordsTo - 3);
                }

                # add word to new description
                $NewDescrip .= " ".$Word;
            }

            # set description to new description
            $Descrip = $NewDescrip;
        }

        # return description to caller
        return $Descrip;
    }


    /*@)*/
    # ---- BACKWARD COMPATIBILITY ---------------------------------------------
    /** @name Backward Compatibility  */ /*@(*/

    /**
    * Retrieve search parameters in legacy array format.  This method is
    * provided for backward compatibility only, and its use is deprecated.
    * Searches conducted using the legacy format may not return identical
    * results because the legacy format does not support nested search groups.
    * @return array Parameters in legacy array format.
    * @see SearchEngine::GroupedSearch()
    */
    function GetAsLegacyArray()
    {
        # set logic for search group
        $Group["Logic"] = ($this->Logic == "OR")
                ? SearchEngine::LOGIC_OR : SearchEngine::LOGIC_AND;

        # for each set of search strings
        foreach ($this->SearchStrings as $Field => $Strings)
        {
            # get text name of field
            $FieldName = call_user_func(self::$PrintableFieldFunction, $Field);

            # add set to group
            $Group["SearchStrings"][$FieldName] = $Strings;
        }

        # for each keyword search string
        foreach ($this->KeywordSearchStrings as $String)
        {
            # add string to keyword entry in group
            $Group["SearchStrings"]["XXXKeywordXXX"][] = $String;
        }

        # add group to array
        $Legacy[] = $Group;

        # for each subgroup
        foreach ($this->Subgroups as $Subgroup)
        {
            # retrieve legacy array for subgroup
            $SubLegacy = $Subgroup->GetAsLegacyArray();

            # add groups from legacy array to our array
            $Legacy = array_merge($Legacy, $SubLegacy);
        }

        # set logic for whole array
        $Legacy["Logic"] = $Group["Logic"];

        # return array to caller
        return $Legacy;
    }


    /*@)*/
    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $KeywordSearchStrings = array();
    private $Logic = self::DEFAULT_LOGIC;
    private $SearchStrings = array();
    private $Subgroups = array();

    static private $CanonicalFieldFunction;
    static private $PrintableFieldFunction;
    static private $UrlParameterPrefix = "F";

    const DEFAULT_LOGIC = "AND";
    const URL_KEYWORDFREE_RANGE = "A-JL-Z";
    const URL_KEYWORD_INDICATOR = "K";
    const URL_LOGIC_INDICATOR = "00";

    /**
    * Load set from serialized data.
    * @param string $Serialized Set data.
    * @throws InvalidArgumentException if incoming set data appears invalid.
    */
    private function LoadFromData($Serialized)
    {
        # unpack new data
        $Data = unserialize($Serialized);
        if (!is_array($Data))
        {
            throw new InvalidArgumentException("Incoming set data"
                    ." appears invalid.");
        }

        # load logic
        $this->Logic = isset($Data["Logic"]) ? $Data["Logic"] : "AND";

        # load search strings
        $this->SearchStrings = isset($Data["SearchStrings"])
                ? $Data["SearchStrings"] : array();
        $this->KeywordSearchStrings = isset($Data["KeywordSearchStrings"])
                ? $Data["KeywordSearchStrings"] : array();

        # load any subgroups
        $this->Subgroups = array();
        if (isset($Data["Subgroups"]))
        {
            foreach ($Data["Subgroups"] as $SubgroupData)
            {
                $this->Subgroups[] = new SearchParameterSet($SubgroupData);
            }
        }
    }

    /*
    * Get the search set parameter set as an URL parameter string.
    * @param string $SetPrefix Prefix to use after the URL parameter prefix.
    * @return array Array of URL parameters.
    */
    private function GetAsUrlParameters($SetPrefix = "")
    {
        # for each search string group in set
        $Params = array();
        foreach ($this->SearchStrings as $FieldId => $Values)
        {
            # get numeric version of field ID if not already numeric
            if (!is_numeric($FieldId))
            {
                $FieldId = call_user_func(self::$CanonicalFieldFunction, $FieldId);
            }

            # for each search string in group
            $ParamSuffix = "";
            foreach ($Values as $Value)
            {
                # check for too many search strings for this field
                if ($ParamSuffix == "Z")
                {
                    throw new Exception("Maximum search parameter complexity"
                            ." exceeded:  more than 26 search parameters for"
                            ." field ID ".$FieldId.".");
                }

                # add search string to URL
                $Params[self::$UrlParameterPrefix.$SetPrefix
                        .$FieldId.$ParamSuffix] = $Value;
                $ParamSuffix = ($ParamSuffix == "") ? "A"
                        : chr(ord($ParamSuffix) + 1);
            }
        }

        # for each keyword search string
        $ParamSuffix = "";
        foreach ($this->KeywordSearchStrings as $Value)
        {
            # check for too many keyword search strings
            if ($ParamSuffix == "Z")
            {
                throw new Exception("Maximum search parameter complexity"
                        ." exceeded:  more than 26 keyword search parameters.");
            }

            # add search string to URL
            $Params[self::$UrlParameterPrefix.$SetPrefix
                    .self::URL_KEYWORD_INDICATOR.$ParamSuffix] = $Value;
            $ParamSuffix = ($ParamSuffix == "") ? "A"
                    : chr(ord($ParamSuffix) + 1);
        }

        # add logic if not default
        if ($this->Logic != self::DEFAULT_LOGIC)
        {
            $Params[self::$UrlParameterPrefix.$SetPrefix
                    .self::URL_LOGIC_INDICATOR] = $this->Logic;
        }

        # for each search parameter subgroup
        $SetLetter = "A";
        foreach ($this->Subgroups as $Subgroup)
        {
            # check for too many subgroups
            if ($SetLetter == "Z")
            {
                throw new Exception("Maximum search parameter complexity"
                        ." exceeded:  more than 24 search parameter subgroups.");
            }

            # retrieve URL string for subgroup and add it to URL
            $Params = array_merge($Params, $Subgroup->GetAsUrlParameters(
                    $SetPrefix.$SetLetter));

            # move to next set letter
            $SetLetter = ($SetLetter == chr(ord(self::URL_KEYWORD_INDICATOR) - 1))
                    ? chr(ord(self::URL_KEYWORD_INDICATOR) + 1)
                    : chr(ord($SetLetter) + 1);
        }

        # return constructed URL parameter string to caller
        return $Params;
    }

    /*
    * Set search parameter from URL parameters in the same format as
    * produced by GetAsUrlParameters().
    * @param string $UrlParameter URL parameter string or array.
    * @see SearchParameterSet::GetAsUrlParameters()
    */
    private function SetFromUrlParameters($UrlParameters)
    {
        # if string was passed in
        if (is_string($UrlParameters))
        {
            # split string into parameter array
            $Params = explode("&", $UrlParameters);

            # pare down parameter array to search parameter elements
            #       and strip off search parameter prefix
            $NewUrlParameters = array();
            foreach ($Params as $Param)
            {
                if (strpos($Param, self::$UrlParameterPrefix) === 0)
                {
                    list($Index, $Value) = explode("=", $Param);
                    $NewUrlParameters[$Index] = urldecode($Value);
                }
            }
            $UrlParameters = $NewUrlParameters;
        }

        # for each search parameter
        foreach ($UrlParameters as $ParamName => $SearchString)
        {
            # strip off standard search parameter prefix
            $ParamName = substr($ParamName, strlen(self::$UrlParameterPrefix));

            # split parameter into component parts
            $SplitResult = preg_match("/^([".self::URL_KEYWORDFREE_RANGE."]*)"
                    ."([0-9".self::URL_KEYWORD_INDICATOR."]+)([A-Z]*)$/",
                    $ParamName, $Matches);

            # if split was successful
            if ($SplitResult === 1)
            {
                # pull components from split pieces
                $SetPrefix = $Matches[1];
                $FieldId = $Matches[2];
                $ParamSuffix = $Matches[3];

                # if set prefix indicates parameter is part of our set
                if ($SetPrefix == "")
                {
                    switch ($FieldId)
                    {
                        case self::URL_LOGIC_INDICATOR:
                            # set logic
                            $this->Logic($SearchString);
                            break;

                        case self::URL_KEYWORD_INDICATOR:
                            # add string to keyword searches
                            $this->KeywordSearchStrings[] = $SearchString;
                            break;

                        default:
                            # add string to searches for appropriate field
                            $this->SearchStrings[$FieldId][] = $SearchString;
                            break;
                    }
                }
                else
                {
                    # add parameter to array for subgroup
                    $SubgroupIndex = $SetPrefix[0];
                    $SubgroupPrefix = (strlen($SetPrefix) > 1)
                            ? substr($SetPrefix, 1) : "";
                    $SubgroupParamIndex = self::$UrlParameterPrefix
                            .$SubgroupPrefix.$FieldId.$ParamSuffix;
                    $SubgroupParameters[$SubgroupIndex][$SubgroupParamIndex]
                            = $SearchString;
                }
            }
        }

        # if subgroups were found
        if (isset($SubgroupParameters))
        {
            # for each identified subgroup
            foreach ($SubgroupParameters as $SubgroupIndex => $Parameters)
            {
                # create subgroup and set parameters
                $Subgroup = new SearchParameterSet();
                $Subgroup->SetFromUrlParameters($Parameters);

                # add subgroup to our set
                $this->Subgroups[] = $Subgroup;
            }
        }
    }
}
