<?php
#
#   FILE:  SystemConfiguration.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

class SystemConfiguration {

    /**
     * This value is used to determine if a value is changed. Uses DB_NOVALUE
     * because of its pervasive use within this class.
     */
    const UNCHANGED_VALUE = DB_NOVALUE;

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Constructs a new SystemConfiguration object.
    */
    function __construct()
    {
        $this->DB = new Database();
    }

    /**
    * Gets or sets the top link string system configuration setting.
    * Used by the BrowseResources page to print classification links.
    * @param string $NewValue The new top link string system configuration
    *       setting.  (OPTIONAL)
    * @return The top link string system configuration setting.
    */
    function TopLinkString($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("TopLinkString", $NewValue);
    }

    /**
    * Gets or sets the top link begin letter system configuration setting.
    * Used by the BrowseResources page to print classification links.
    * @param string $NewValue The new top link begin letter system configuration
    *       setting.  (OPTIONAL)
    * @return The top link begin letter system configuration setting.
    */
    function TopLinkBeginLetter($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("TopLinkBeginLetter", $NewValue);
    }

    /**
    * Gets or sets the top link end letter system configuration setting.
    * Used by the BrowseResources page to print classification links.
    * @param string $NewValue The new top link end letter system configuration
    *       setting.  (OPTIONAL)
    * @return The top link end letter system configuration setting.
    */
    function TopLinkEndLetter($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("TopLinkEndLetter", $NewValue);
    }

    /**
    * Gets or sets the admin email system configuration setting.
    * @param string $NewValue The new admin email system configuration
    *       setting.  (OPTIONAL)
    * @return The admin email system configuration setting.
    */
    function AdminEmail($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("AdminEmail", $NewValue);
    }

    /**
    * Gets or sets the password mail subject system configuration setting.
    * @param string $NewValue The new password mail subject system configuration
    *       setting.  (OPTIONAL)
    * @return The password mail subject system configuration setting.
    */
    function PasswordMailSubject($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("PasswordMailSubject", $NewValue);
    }

    /**
    * Gets or sets the password mail body system configuration setting.
    * @param string $NewValue The new password mail body system configuration
    *       setting.  (OPTIONAL)
    * @return The password mail body system configuration setting.
    */
    function PasswordMailBody($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("PasswordMailBody", $NewValue);
    }

    /**
    * Gets or sets the password reset mail subject system configuration setting.
    * @param string $NewValue The new password reset mail subject system
    *       configuration setting.  (OPTIONAL)
    * @return The password reset mail subject system configuration setting.
    */
    function PasswordResetMailSubject($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("PasswordResetMailSubject", $NewValue);
    }

    /**
    * Gets or sets the password reset mail body system configuration setting.
    * @param string $NewValue The new password reset mail body system
    *       configuration setting.  (OPTIONAL)
    * @return The password reset mail body system configuration setting.
    */
    function PasswordResetMailBody($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("PasswordResetMailBody", $NewValue);
    }

    /**
    * Gets or sets the mail change mail subject system configuration setting.
    * @param string $NewValue The new mail change mail subject system
    *       configuration setting.  (OPTIONAL)
    * @return The mail change mail subject system configuration setting.
    */
    function MailChangeMailSubject($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("MailChangeMailSubject", $NewValue);
    }

    /**
    * Gets or sets the mail change mail body system configuration setting.
    * @param string $NewValue The new mail change mail body system configuration
    *       setting.  (OPTIONAL)
    * @return The mail change mail body system configuration setting.
    */
    function MailChangeMailBody($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("MailChangeMailBody", $NewValue);
    }

    /**
    * Gets or sets the portal name system configuration setting.
    * @param string $NewValue The new portal name system configuration
    *       setting.  (OPTIONAL)
    * @return The portal name system configuration setting.
    */
    function PortalName($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("PortalName", $NewValue);
    }

    /**
    * Gets or sets the default active UI system configuration setting.
    * @param string $NewValue The new default active UI system configuration
    *       setting.  (OPTIONAL)
    * @return The default active UI system configuration setting.
    */
    function DefaultActiveUI($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("DefaultActiveUI", $NewValue);
    }

    /**
    * Gets or sets the current theme system configuration setting.
    * @param string $NewValue The new current theme system configuration
    *       setting.  (OPTIONAL)
    * @return The current theme system configuration setting.
    */
    function CurrentTheme($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("CurrentTheme", $NewValue);
    }

    /**
    * Gets or sets the default character set system configuration setting.
    * @param string $NewValue The new default character set system configuration
    *       setting.  (OPTIONAL)
    * @return The default character set system configuration setting.
    */
    function DefaultCharacterSet($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("DefaultCharacterSet", $NewValue);
    }

    /**
    * Gets or sets the search terms required system configuration setting.
    * Used to determine whether the search logic is AND or OR.
    * @param int $NewValue The new search terms required system configuration
    *       setting.  (OPTIONAL)
    * @return The search terms required system configuration setting.
    */
    function SearchTermsRequired($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("SearchTermsRequired", $NewValue);
    }

    /**
    * Gets or sets the show number of resources enabled system configuration
    * setting.  Used on the Home page to display the total number of resources.
    * @param int $NewValue The new show number of resources enabled system
    *       configuration setting.  (OPTIONAL)
    * @return The show number of resources enabled system configuration setting.
    */
    function ShowNumResourcesEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ShowNumResourcesEnabled", $NewValue);
    }

    /**
    * Gets or sets the announcements enabled system configuration setting.
    * @param int $NewValue The new announcements enabled system configuration
    *       setting.  (OPTIONAL)
    * @return The announcements enabled system configuration setting.
    */
    function AnnouncementsEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("AnnouncementsEnabled", $NewValue);
    }

    /**
    * Gets or sets the forums enabled system configuration setting.
    * @param int $NewValue The new forums enabled system configuration
    *       setting.  (OPTIONAL)
    * @return The forums enabled system configuration setting.
    */
    function ForumsEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ForumsEnabled", $NewValue);
    }

    /**
    * Gets or sets the forums allow html system configuration setting.
    * @param int $NewValue The new forums allow html system configuration setting.
    *        (OPTIONAL)
    * #return The forums allow html system configuration setting.
    */
    function ForumsAllowHTML($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ForumsAllowHTML", $NewValue);
    }

    /**
    * Gets or sets the forums use WYSIWYG editor setting.
    * @param int $NewValue The new Forums use WYSIWYG system configuration setting.
    *        (OPTIONAL)
    * @return The forums use WYSIWYG editor setting.
    */
    function ForumsUseWYSIWYG($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ForumsUseWYSIWYG", $NewValue);
    }

    /**
    * Gets or sets the allow multiple UIs enabled system configuration setting.
    * Used to determine whether users can use a UI other than the default.
    * @param int $NewValue The new allow multiple UIs enabled system
    *       configuration setting.  (OPTIONAL)
    * @return The allow multiple UIs enabled system configuration setting.
    */
    function AllowMultipleUIsEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("AllowMultipleUIsEnabled", $NewValue);
    }

    /**
    * Gets or sets the resource launches new window enabled system configuration
    * setting. Used by the FullRecorde page to determine whether the URL opens
    * in a new window.
    * @param int $NewValue The new resource launches new window enabled system
    *       configuration setting.  (OPTIONAL)
    * @return The resource launches new window enabled system configuration
    * setting.
    */
    function ResourceLaunchesNewWindowEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ResourceLaunchesNewWindowEnabled", $NewValue);
    }

    /**
    * Gets or sets the user agents enabled system configuration setting.
    * Used for saved search emails.
    * @param int $NewValue The new user agents enabled system configuration
    *       setting.  (OPTIONAL)
    * @return The user agents enabled system configuration setting.
    */
    function UserAgentsEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("UserAgentsEnabled", $NewValue);
    }

    /**
    * Gets or sets the resource ratings enabled system configuration setting.
    * @param int $NewValue The new resource ratings enabled system configuration
    *       setting.  (OPTIONAL)
    * @return The resource ratings enabled system configuration setting.
    */
    function ResourceRatingsEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ResourceRatingsEnabled", $NewValue);
    }

    /**
    * Gets or sets the resource comments enabled system configuration setting.
    * @param int $NewValue The new resource comments enabled system
    *       configuration setting.  (OPTIONAL)
    * @return The resource comments enabled system configuration setting.
    */
    function ResourceCommentsEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ResourceCommentsEnabled", $NewValue);
    }

    /**
    * Gets or sets the accessibility wizard enabled system configuration
    * setting.  (Vestigial, maintained for backward compatibility with old
    * interfaces -- DO NOT USE.)
    * @param int $NewValue The new accessibility wizard enabled system
    *       configuration setting.  (OPTIONAL)
    * @return The accessibility wizard enabled system configuration setting.
    */
    function AccessibilityWizardEnabled($NewValue = DB_NOVALUE)
    {
        return FALSE;
    }

    /**
    * Gets or sets the incremental keyword search enabled system configuration
    * setting. Used for determining whether searching shows results as you type.
    * @param int $NewValue The new incremental keyword search enabled system
    *       configuration setting.  (OPTIONAL)
    * @return The incremental keyword search enabled system configuration
    *       setting.
    */
    function IncrementalKeywordSearchEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("IncrementalKeywordSearchEnabled", $NewValue);
    }

    /**
    * Gets or sets the release flag approved system configuration setting.
    * Used to determine whether a resource can be flagged for release.
    * @param int $NewValue The new release flag approved enabled system
    *       configuration setting.  (OPTIONAL)
    * @return The release flag approved enabled system configuration setting.
    */
    function ReleaseFlagApproved($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ReleaseFlagApproved", $NewValue);
    }

    /**
    * Gets or sets the search database enabled system configuration setting.
    * @param int $NewValue The new search database enabled system configuration
    *       setting.  (OPTIONAL)
    * @return The search database enabled system configuration setting.
    */
    function SearchDBEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("SearchDBEnabled", $NewValue);
    }

    /**
    * Gets or sets the recommender database enabled system configuration
    * setting.
    * @param int $NewValue The new recommender database enabled system
    *       configuration setting.  (OPTIONAL)
    * @return The recommender database enabled system configuration setting.
    */
    function RecommenderDBEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("RecommenderDBEnabled", $NewValue);
    }

    /**
    * Gets or sets OAISQ enabled system configuration setting.
    * @param int $NewValue The new OAISQ enabled system configuration
    *       setting.  (OPTIONAL)
    * @return The OAISQ enabled system configuration setting.
    */
    function OAISQEnabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("OAISQEnabled", $NewValue);
    }

    /**
    * Gets or sets the number of announcements on homepage system configuration
    * setting.
    * @param int $NewValue The new number of announcements on homepage system
    *       configuration setting.  (OPTIONAL)
    * @return The number of announcements on homepage system configuration
    *       setting.
    */
    function NumAnnounceOnHomePage($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("NumAnnounceOnHomePage", $NewValue);
    }

    /**
    * Gets or sets the number of resources on homepage system configuration
    * setting.
    * @param int $NewValue The new number of resources on homepage system
    *       configuration setting.  (OPTIONAL)
    * @return The number of resources on homepage system configuration setting.
    */
    function NumResourcesOnHomePage($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("NumResourcesOnHomePage", $NewValue);
    }

    /**
    * Gets or sets the legal notice system configuration setting.
    * @param string $NewValue The new legal notice set system configuration
    *       setting.  (OPTIONAL)
    * @return The legal notice system configuration setting.
    */
    function LegalNotice($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("LegalNotice", $NewValue);
    }

    /**
    * Gets or sets the default records per page system configuration setting.
    * @param int $NewValue The new default records per page system configuration
    *       setting.  (OPTIONAL)
    * @return The default records per page system configuration setting.
    */
    function DefaultRecordsPerPage($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("DefaultRecordsPerPage", $NewValue);
    }

    /**
    * Gets or sets the number of resources considered to build search
    * facets.
    * @param int $NewValue Desired setting for the number of
    *        resources to use in computing search facets. (OPTIONAL)
    * @return The number of resources used in computing search facets.
    */
    function NumResourcesForSearchFacets($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("NumResourcesForSearchFacets", $NewValue);
    }

    /**
    * Gets or sets the number of facets considered excessive, above which we
    * won't display the facets.
    * @param int $NewValue Desired setting for the number of
    *        resources to use in computing search facets. (OPTIONAL)
    * @return The maximum number of facets per field to display.
    */
    function MaxFacetsPerField($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("MaxFacetsPerField", $NewValue);
    }

    /**
    * Gets or sets the number of classes per browse page system configuration
    * setting.
    * @param int $NewValue The new number of classes per browse page system
    *       configuration setting.  (OPTIONAL)
    * @return The number of classes per browse page system configuration
    *       setting.
    */
    function NumClassesPerBrowsePage($NewValue = DB_NOVALUE)
    {
        # if new classes per browse page has been specified
        if ($NewValue !== DB_NOVALUE &&
                $NewValue != $this->UpdateValue("NumClassesPerBrowsePage", DB_NOVALUE))
        {
            # clear all classification links
            $this->TopLinkString("");
            $this->DB->Query("UPDATE Classifications SET LinkString=''");
        }
        return $this->UpdateValue("NumClassesPerBrowsePage", $NewValue);
    }

    /**
    * Gets or sets the number of columns per browse page system configuration
    * setting.
    * @param int $NewValue The new number of columns per browse page system
    *       configuration setting.  (OPTIONAL)
    * @return The number of columns per browse page system configuration
    *       setting.
    */
    function NumColumnsPerBrowsePage($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("NumColumnsPerBrowsePage", $NewValue);
    }

    /**
    * Gets or sets the browsing field ID system configuration setting.
    * Used by the BrowseResources page to determine the default browsing option.
    * @param int $NewValue The new browsing field ID system configuration
    *       setting.  (OPTIONAL)
    * @return The browsing field ID system configuration setting.
    */
    function BrowsingFieldId($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("BrowsingFieldId", $NewValue);
    }

    /**
    * Gets or sets the OAI ID domain system configuration setting.
    * @param string $NewValue The new OAI ID domain system configuration
    *       setting.  (OPTIONAL)
    * @return The OAI ID domain system configuration setting.
    */
    function OaiIdDomain($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("OaiIdDomain", $NewValue);
    }

    /**
    * Gets or sets the OAI ID prefix system configuration setting.
    * @param string $NewValue The new OAI ID prefix system configuration
    *       setting.  (OPTIONAL)
    * @return The OAI ID prefix system configuration setting.
    */
    function OaiIdPrefix($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("OaiIdPrefix", $NewValue);
    }

    /**
    * Gets or sets the OAI earliest date system configuration setting.
    * @param string $NewValue The new OAI earliest date system configuration
    *       setting.  (OPTIONAL)
    * @return The OAI earliest date system configuration setting.
    */
    function OaiEarliestDate($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("OaiEarliestDate", $NewValue);
    }

    /**
    * Gets or sets the OAI date granularity system configuration setting.
    * @param string $NewValue The new OAI date granularity system configuration
    *       setting.  (OPTIONAL)
    * @return The OAI date granularity system configuration setting.
    */
    function OaiDateGranularity($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("OaiDateGranularity", $NewValue);
    }

    /**
    * Gets or sets the site keywords system configuration setting.
    * @param string $NewValue The new site keywords system configuration
    *       setting.  (OPTIONAL)
    * @return The site keywords system configuration setting.
    */
    function SiteKeywords($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("SiteKeywords", $NewValue);
    }

    /**
    * Gets or sets the about text system configuration setting.
    * @param string $NewValue The new about text system configuration
    *       setting.  (OPTIONAL)
    * @return The about text system configuration setting.
    */
    function AboutText($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("AboutText", $NewValue);
    }

    /**
    * Gets or sets the contact name system configuration setting.
    * @param string $NewValue The new contact name system configuration
    *       setting.  (OPTIONAL)
    * @return The contact name system configuration setting.
    */
    function ContactName($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ContactName", $NewValue);
    }

    /**
    * Gets or sets the contact email system configuration setting.
    * @param string $NewValue The new contact email system configuration
    *       setting.  (OPTIONAL)
    * @return The contact email system configuration setting.
    */
    function ContactEmail($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ContactEmail", $NewValue);
    }

    /**
    * Gets or sets the organization system configuration setting.
    * @param string $NewValue The new organization system configuration
    *       setting.  (OPTIONAL)
    * @return The organization system configuration setting.
    */
    function Organization($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("Organization", $NewValue);
    }

    /**
    * Gets or sets the organization type system configuration setting.
    * @param string $NewValue The new organization type system configuration
    *       setting.  (OPTIONAL)
    * @return The organization type system configuration setting.
    */
    function OrganizationType($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("OrganizationType", $NewValue);
    }

    /**
    * Gets or sets the site type system configuration setting.
    * @param string $NewValue The new site type system configuration
    *       setting.  (OPTIONAL)
    * @return The site type system configuration setting.
    */
    function SiteType($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("SiteType", $NewValue);
    }

    /**
    * Gets or sets the okay to list site system configuration setting.
    * Used when registering to determine if the site can be included in a public
    * list of CWIS sites.
    * @param int $NewValue The new okay to list site system configuration
    *       setting.  (OPTIONAL)
    * @return The okay to list site system configuration setting.
    */
    function OkayToListSite($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("OkayToListSite", $NewValue);
    }

    /**
    * Gets or sets the last registered on system configuration setting.
    * @param string $NewValue The new last registered on system configuration
    *       setting.  (OPTIONAL)
    * @return The last registered on system configuration setting.
    */
    function LastRegisteredOn($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("LastRegisteredOn", $NewValue);
    }

    /**
    * Gets or sets the add AW stats script system configuration setting.
    * @param int $NewValue The new add AW stats script system configuration
    *       setting.  (OPTIONAL)
    * @return The add AW stats script system configuration setting.
    */
    function AddAWStatsScript($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("AddAWStatsScript", $NewValue);
    }

    /**
    * Gets or sets the display limits by default system configuration setting.
    * Used by the Advanced search page to determine whether search limits should
    * be displayed or hidden.
    * @param int $NewValue The new display limits by default system
    *       configuration setting.  (OPTIONAL)
    * @return The display limits by default system configuration setting.
    */
    function DisplayLimitsByDefault($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("DisplayLimitsByDefault", $NewValue);
    }

    /**
    * Gets or sets the default sort field system configuration setting.
    * Used to determine the default field to use when sorting search results.
    * @param int $NewValue The new default sort field system configuration
    *       setting.  (OPTIONAL)
    * @return The default sort field system configuration setting.
    */
    function DefaultSortField($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("DefaultSortField", $NewValue);
    }

    /**
    * Gets or sets the field display folder system configuration setting.
    * Used to get the sort order of metadata fields for display.
    * @param int $NewValue The new field display folder system configuration
    *       setting.  (OPTIONAL)
    * @return The field display folder system configuration setting.
    */
    function FieldDisplayFolder($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("FieldDisplayFolder", $NewValue);
    }

    /**
    * Gets or sets the field edit folder system configuration setting.
    * Used to get the sort order of metadata fields for editing.
    * @param int $NewValue The new field edit folder system configuration
    *       setting.  (OPTIONAL)
    * @return The field edit folder system configuration setting.
    */
    function FieldEditFolder($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("FieldEditFolder", $NewValue);
    }

    /**
    * Gets or sets the collapse metadata field groups system configuration
    * setting. Used by the MetadataFieldOrdering page to determine whether
    * metadata field groups should be collapsed.
    * @param int $NewValue The new collapse metadata field groups system
    *       configuration setting.  (OPTIONAL)
    * @return The collapse metadata field groups system configuration setting.
    */
    function CollapseMetadataFieldGroups($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("CollapseMetadataFieldGroups", $NewValue);
    }

    /**
    * Gets or sets the require email with feedback system configuration setting.
    * @param int $NewValue The new require email with feedback system
    *       configuration setting.  (OPTIONAL)
    * @return The require email with feedback system configuration setting.
    */
    function RequireEmailWithFeedback($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("RequireEmailWithFeedback", $NewValue);
    }

    /**
    * Gets or sets the search engine update priority system configuration
    *       setting.
    * @param int $NewValue The new search engine update priority system
    *       configuration setting.  (OPTIONAL)
    * @return The search engine update priority system configuration setting.
    */
    function SearchEngineUpdatePriority($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("SearchEngineUpdatePriority", $NewValue);
    }

    /**
    * Gets or sets the recommender engine update priority system configuration
    *       setting.
    * @param int $NewValue The new recommender engine update priority system
    *       configuration setting.  (OPTIONAL)
    * @return The recommender engine update priority system configuration
    *       setting.
    */
    function RecommenderEngineUpdatePriority($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("RecommenderEngineUpdatePriority", $NewValue);
    }

    /**
    * Gets or sets the logo file name system configuration setting.
    * @param string $NewValue The new logo file name system configuration
    *       setting.  (OPTIONAL)
    * @return The logo file name system configuration setting.
    */
    function LogoFileName()
    {
        return $this->UpdateValue("LogoFileName", DB_NOVALUE, "UIStyleConfiguration");
    }

    /**
    * Gets or sets the logo alt text system configuration setting.
    * @param string $NewValue The new logo alt text system configuration
    *       setting.  (OPTIONAL)
    * @return The logo alt text system configuration setting.
    */
    function LogoAltText()
    {
        return $this->UpdateValue("LogoAltText", DB_NOVALUE, "UIStyleConfiguration");
    }

    /**
    * Gets or sets the logo width system configuration setting.
    * @param string $NewValue The new logo width system configuration
    *       setting.  (OPTIONAL)
    * @return The logo width system configuration setting.
    */
    function LogoWidth()
    {
        return $this->UpdateValue("LogoWidth", DB_NOVALUE, "UIStyleConfiguration");
    }

    /**
    * Gets or sets the logo height system configuration setting.
    * @param string $NewValue The new logo height system configuration
    *       setting.  (OPTIONAL)
    * @return The logo height system configuration setting.
    */
    function LogoHeight()
    {
        return $this->UpdateValue("LogoHeight", DB_NOVALUE, "UIStyleConfiguration");
    }

    /**
    * Gets or sets the has registered system configuration setting.
    * @param int $NewValue The new has registered system configuration
    *       setting.  (OPTIONAL)
    * @return The has registered system configuration setting.
    */
    function HasRegistered()
    {
        return strlen($this->UpdateValue("LastRegisteredOn", DB_NOVALUE))
                ? TRUE : FALSE;
    }

    /**
    * Gets or sets the title field system configuration setting.
    * @param int $NewValue The new title field system configuration
    *       setting.  (OPTIONAL)
    * @return The title field system configuration setting.
    */
    function TitleField($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("TitleField", $NewValue);
    }

    /**
    * Gets or sets the description field system configuration setting.
    * @param int $NewValue The new description field system configuration
    *       setting.  (OPTIONAL)
    * @return The description field system configuration setting.
    */
    function DescriptionField($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("DescriptionField", $NewValue);
    }

    /**
    * Gets or sets the url field system configuration setting.
    * @param int $NewValue The new url field system configuration
    *       setting.  (OPTIONAL)
    * @return The url field system configuration setting.
    */
    function UrlField($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("UrlField", $NewValue);
    }

    /**
    * Gets or sets the screenshot field system configuration setting.
    * @param int $NewValue The new screenshot field system configuration
    *       setting.  (OPTIONAL)
    * @return The screenshot field system configuration setting.
    */
    function ScreenshotField($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ScreenshotField", $NewValue);
    }

    /**
    * Gets or sets the secure login system configuration setting.
    * @param int $NewValue The new secure login system configuration
    *       setting.  (OPTIONAL)
    * @return The secure login system configuration setting.
    */
    function SecureLogin($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("SecureLogin", $NewValue);
    }

    /**
    * Gets or sets the email line ending system configuration setting.
    * @param string $NewValue The new email line ending system configuration
    *       setting.  (OPTIONAL)
    * @return The email line ending system configuration setting.
    */
    function EmailLineEnding($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("EmailLineEnding", $NewValue);
    }

    /**
    * Gets or sets the saved search mail template system configuration setting.
    * @param int $NewValue The new saved search mail template system
    *       configuration setting.  (OPTIONAL
    * @return The saved search mail template system configuration setting.
    */
    function SavedSearchMailTemplate($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("SavedSearchMailTemplate", $NewValue);
    }

    /**
    * Gets or sets the default user privileges system configuration setting.
    * @param string $NewValue The new default user privileges system
    *       configuration setting.  (OPTIONAL)
    * @return The default user privileges system configuration setting.
    */
    function DefaultUserPrivs($NewValue = DB_NOVALUE)
    {
        if ($NewValue !== DB_NOVALUE)
        {
            $NewValue = serialize($NewValue);
        }

        $Result = unserialize($this->UpdateValue("DefaultUserPrivs", $NewValue));

        if (!is_array($Result))
        {
            $Result = $this->DefaultUserPrivs(array());
        }

        return $Result;
    }

    /**
    * Gets or sets the root URL override system configuration setting.
    * @param string $NewValue The new root URL override system
    *       configuration setting.  (OPTIONAL)
    * @return The root URL override system configuration setting.
    */
    function RootUrlOverride($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("RootUrlOverride", $NewValue);
    }

    /**
    * Gets or sets the prefer HTTP_HOST system configuration setting.
    * @param string $NewValue The new prefer HTTP_HOST system
    *       configuration setting.  (OPTIONAL)
    * @return The prefer HTTP_HOST system configuration setting.
    */
    function PreferHttpHost($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("PreferHttpHost", $NewValue);
    }

    /**
    * Gets or changes a value in the database.
    * @param string $FieldName Name of database field.
    * @param mixed $NewValue New value to set.  (OPTIONAL)
    * @return Requested value.
    */
    function Value($FieldName, $NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue($FieldName, $NewValue);
    }

    /**
    * Method retained for backward compatibility with old interfaces.
    * Its use should be replaced by the Google Analytics plugin.
    */
    function AddGoogleAnalytics($NewValue = DB_NOVALUE)
    {
        return FALSE;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $DB;
    private $DBFields;

    /**
    * Gets or changes a value in the database.
    * @param string $FieldName Name of database field.
    * @param mixed $NewValue The new value to set.  Use DB_NOVALUE to not set
    *       a new value (i.e. when just getting a value).
    * @param string $Table Name of database table.  (OPTIONAL)
    * @return Requested value.
    */
    private function UpdateValue($FieldName, $NewValue, $Table="SystemConfiguration")
    {
        if ($NewValue != DB_NOVALUE)
        {
            $SignalResult = $GLOBALS["AF"]->SignalEvent(
                    "EVENT_SET_SYSCONFIG_VALUE", array(
                            "FieldName" => $FieldName,
                            "NewValue" => $NewValue));
            $NewValue = $SignalResult["NewValue"];
        }
        $Value = $this->DB->UpdateValue($Table, $FieldName,
                                      $NewValue, NULL, $this->DBFields);
        $SignalResult = $GLOBALS["AF"]->SignalEvent(
                "EVENT_GET_SYSCONFIG_VALUE", array(
                        "FieldName" => $FieldName,
                        "Value" => $Value));
        $Value = $SignalResult["Value"];
        return $Value;
    }
}
