<?PHP
#
#   FILE:  iCalendar.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2013-2015 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Class to generate a simple iCalendar document.
*/
class iCalendar
{

    /**
    * Construct a basic iCalendar document.
    * @param string $ID Event ID used when generating the UID.
    * @param string $StartDate Event start date parsable by strtotime().
    * @param string $EndDate Event end date parsable by strtotime().
    * @param bool $AllDay Flag to specify if the event takes place throughout
    *      the day instead of during specific times.
    * @param string $TimeZoneID Optional time zone ID, e.g., "America/New_York".
    */
    public function __construct($ID, $StartDate, $EndDate, $AllDay, $TimeZoneID=NULL)
    {
        # generate the UID and add it to the document
        $this->AddProperty("VEVENT", "UID", $this->GenerateUID($ID, $StartDate));

        # need to use the time zone parameter if a time zone ID is given
        $DateParameters = is_null($TimeZoneID) ? array() : array("TZID" => $TimeZoneID);

        if ($AllDay)
        {
            # need to offset the end date by one day so that the range spans the
            # entire 24 hours of the last day
            $EndDate = date("Y-m-d", strtotime($EndDate)+86400);

            $this->AddDateProperty("VEVENT", "DTSTART", $StartDate, $DateParameters);
            $this->AddDateProperty("VEVENT", "DTEND", $EndDate, $DateParameters);
        }

        else
        {
            $this->AddDateTimeProperty("VEVENT", "DTSTART", $StartDate, $DateParameters);
            $this->AddDateTimeProperty("VEVENT", "DTEND", $EndDate, $DateParameters);
        }
    }

    /**
    * Add the created property to the iCalendar document. An existing created
    * property will be overwritten.
    * @param string $Value The date and time the event was created.
    */
    public function AddCreated($Value)
    {
        $this->AddTextProperty(
            "VEVENT", "CREATED", $this->GenerateUTCDateTimeString($Value));
    }

    /**
    * Add the summary property to the iCalendar document. An existing summary
    * property will be overwritten.
    * @param string $Value The body of the summary.
    */
    public function AddSummary($Value)
    {
        # add the property
        $this->AddTextProperty("VEVENT", "SUMMARY", $Value);

        # save the summary for use in generating the file name
        $this->Summary = $Value;
    }

    /**
    * Add the description property to the iCalendar document. An existing
    * description property will be overwritten.
    * @param string $Value The body of the description.
    */
    public function AddDescription($Value)
    {
        $this->AddTextProperty("VEVENT", "DESCRIPTION", $Value);
    }

    /**
    * Add the categories property to the iCalendar document. An existing
    * categories property will be overwritten.
    * @param array $Categories A list of categories.
    */
    public function AddCategories(array $Categories)
    {
        # don't add the property if there are no categories to add
        if (!count($Categories))
        {
            return;
        }

        $this->AddProperty(
            "VEVENT",
            "CATEGORIES",
            implode(",", array_map(array($this, "EscapeTextValue"), $Categories)));
    }

    /**
    * Add the URL property to the iCalendar document. An existing URL property
    * will be overwritten.
    * @param string $Value The URL to add.
    */
    public function AddURL($Value)
    {
        # don't add a blank URL
        if (!strlen($Value))
        {
            return;
        }

        $this->AddProperty("VEVENT", "URL", $Value);
    }

    /**
    * Add the geographic position property to the iCalendar document. An
    * existing geographic position property will be overwritten.
    * @param float $Latitude Latitude value.
    * @param float $Longitude Longitude value.
    */
    public function AddGeographicPosition($Latitude, $Longitude)
    {
        # construct the value for the property
        $Value = floatval($Latitude) . ";" . floatval($Longitude);

        # add the property to the list
        $this->AddProperty("VEVENT", "GEO", $Value);
    }

    /**
    * Add the location property to the iCalendar document. An existing location
    * property will be overwritten.
    * @param string $Value The location.
    */
    public function AddLocation($Value)
    {
        $this->AddTextProperty("VEVENT", "LOCATION", $Value);
    }

    /**
    * Generate the iCalendar document based on the current list of properties.
    * @return Returns the iCalendar document as a string.
    */
    public function GenerateDocument()
    {
        # generate a timestamp and add it
        $Timestamp = $this->GenerateUTCDateTimeString(date("Y-m-d H:i:s"));
        $this->AddProperty("VEVENT", "DTSTAMP", $Timestamp);

        # start the iCalendar definition
        $Document = "BEGIN:VCALENDAR\r\n";

        # add basic headers
        $Document .= "CALSCALE:GREGORIAN\r\n";
        $Document .= "PRODID:-//Internet Scout//CWIS//EN\r\n";
        $Document .= "VERSION:2.0\r\n";

        # add each component
        foreach ($this->Properties as $Component => $Properties)
        {
            # don't add empty components
            if (!count($Properties))
            {
                continue;
            }

            # begin the component definition
            $Document .= "BEGIN:" . $Component . "\r\n";

            # add each property line
            foreach ($Properties as $Property => $PropertyLine)
            {
                $Document .= $PropertyLine;
            }

            # end the component definition
            $Document .= "END:" . $Component . "\r\n";
        }

        # end the iCalendar definition
        $Document .= "END:VCALENDAR\r\n";

        # return the generated document
        return $Document;
    }

    /**
    * Generate a file name for the iCalendar document. The file name will be the
    * summary property if set and the current date/time if not. The generated
    * file name is safe to use in the "filename" property of the HTTP
    * "Content-Disposition" header when the value is quoted.
    * @return Returns the generated file name.
    */
    public function GenerateFileName()
    {
        return self::GenerateFileNameFromSummary($this->Summary);
    }

    /**
    * Create a file name for an iCalendar document using a given summary. The
    * fiel name will be the current date/time if the summary is not given. The
    * generated file name is safe to use in the "filename" property of the HTTP
    * "Content-Disposition" header when the value is quoted.
    * @param string $Summary Optional summary to use in the name.
    * @return Returns the generated file name.
    */
    public static function GenerateFileNameFromSummary($Summary=NULL)
    {
        # just use the date/time if the summary isn't given
        if (!$Summary)
        {
            return date("Ymd-His") . ".ics";
        }

        # remove any HTML from the summary
        $Name = strip_tags($Summary);

        # replace problematic characters for most filesystems
        $Name = str_replace(
            array("/", "?", "<", ">", "\\", ":", "*", "|", '"', "^"),
            "-",
            $Name);

        # remove whitespace at the beginning and end
        $Name = trim($Name);

        # make sure the name isn't too long because it can cause problems for
        # some browsers and file systems
        $Name = substr($Name, 0, 75);

        # return the name plus extension
        return $Name . ".ics";
    }

    /**
    * Helper method to transform an HTML string to plain text.
    * @param string $HTML HTML string to transform.
    * @return Returns the HTML string transformed to plain text.
    */
    public static function TransformHTMLToPlainText($HTML)
    {
        # remove HTML tags
        $HTML = strip_tags($HTML);

        # handle a few replacements separately because they aren't handled by
        # html_entity_decode() or are replaced by a character that isn't ideal.
        # string to replace => replacement
        $Replace = array(
            "&nbsp;" => " ",
            "&ndash;" => "-",
            "&mdash;" => "--",
            "&ldquo;" => '"',
            "&rdquo;" => '"',
            "&lsquo;" => "'",
            "&rsquo;" => "'");

        # do the first pass of replacements
        $HTML = str_replace(array_keys($Replace), array_values($Replace), $HTML);

        # do the final pass of replacements and return
        return html_entity_decode($HTML);
    }

    /**
    * Add a generic property, i.e., one whose value is already in the proper
    * form.
    * @param string $Component The iCalendar component the property belongs to.
    * @param string $Property The name of the property.
    * @param string $Value The property value.
    * @param array $Parameters Optional parameters for the property. These
    *      should already be properly escaped.
    * @see AddTextProperty()
    * @see AddDateProperty()
    * @see AddDateTimeProperty()
    */
    protected function AddProperty($Component, $Property, $Value,
            array $Parameters=array())
    {
        # construct the property line
        $Line = $this->GeneratePropertyString($Property, $Parameters) . $Value;

        # fold the line if necessary and add the line ending sequence
        $Line = $this->FoldString($Line) . "\r\n";

        # add the property line to the list of properties
        $this->Properties[$Component][$Property] = $Line;
    }

    /**
    * Add a text property to the list.
    * @param string $Component The iCalendar component the property belongs to.
    * @param string $Property The name of the property.
    * @param string $Value The property value.
    * @param array $Parameters Optional parameters for the property. These
    *      should already be properly escaped.
    * @see AddProperty()
    * @see AddDateProperty()
    * @see AddDateTimeProperty()
    * @see http://tools.ietf.org/html/rfc5545#section-3.3.11
    */
    protected function AddTextProperty($Component, $Property, $Value,
            array $Parameters=array())
    {
        # don't add empty properties
        if (!strlen($Value))
        {
            return;
        }

        $this->AddProperty(
            $Component,
            $Property,
            $this->EscapeTextValue($Value),
            $Parameters);
    }

    /**
    * Add a date property to the list.
    * @param string $Component The iCalendar component the property belongs to.
    * @param string $Property The name of the property.
    * @param string $Value The property value.
    * @param array $Parameters Optional parameters for the property. These
    *      should already be properly escaped.
    * @see AddProperty()
    * @see AddTextProperty()
    * @see AddDateTimeProperty()
    * @see http://tools.ietf.org/html/rfc5545#section-3.3.4
    */
    protected function AddDateProperty($Component, $Property, $Value,
            array $Parameters=array())
    {
        $this->AddProperty(
            $Component,
            $Property,
            $this->GenerateDateString($Value),
            array("VALUE" => "DATE") + $Parameters);
    }

    /**
    * Add a date/time property to the list.
    * @param string $Component The iCalendar component the property belongs to.
    * @param string $Property The name of the property.
    * @param string $Value The property value.
    * @param array $Parameters Optional parameters for the property. These
    *      should already be properly escaped.
    * @see AddProperty()
    * @see AddTextProperty()
    * @see AddDateProperty()
    * @see http://tools.ietf.org/html/rfc5545#section-3.3.5
    */
    protected function AddDateTimeProperty($Component, $Property, $Value,
            array $Parameters=array())
    {
        $this->AddProperty(
            $Component,
            $Property,
            $this->GenerateDateTimeString($Value),
            $Parameters);
    }

    /**
    * Escape a text value for inserting into a property line.
    * @param string $Value The text value to escape.
    * @return Returns the escaped text value.
    */
    protected function EscapeTextValue($Value)
    {
        # escape most characters
        $Value = preg_replace('/([\\;,])/', "\\\\\\1", $Value);

        # escape newlines
        $Value = preg_replace('/\n/', "\\n", $Value);

        return $Value;
    }

    /**
    * Generate a full UID from an event ID and start date.
    * @param string $ID Event ID.
    * @param string $StartDate The date the event starts.
    * @return Returns a full UID.
    */
    protected function GenerateUID($ID, $StartDate)
    {
        # concatenate the date string, ID, and host name as in the spec
        $UID = $this->GenerateUTCDateTimeString($StartDate);
        $UID .= "-" . $ID;
        $UID .= "@" . gethostname();

        return $UID;
    }

    /**
    * Generate a date string from a date parsable by strtotime().
    * @param string $Date Date from which to generate the date string.
    * @return Returns a date string.
    */
    protected function GenerateDateString($Date)
    {
        return date("Ymd", strtotime($Date));
    }

    /**
    * Generate a date/time string from a date parsable by strtotime().
    * @param string $DateTime Date/Time from which to generate the date string.
    * @return Returns a date/time string.
    */
    protected function GenerateDateTimeString($DateTime)
    {
        return date("Ymd\THis", strtotime($DateTime));
    }

    /**
    * Generate a UTC date/time string from a date parsable by strtotime().
    * @param string $DateTime Date/Time from which to generate the date string.
    * @return Returns a UTC date/time string.
    */
    protected function GenerateUTCDateTimeString($DateTime)
    {
        return gmdate("Ymd\THis\Z", strtotime($DateTime));
    }

    /**
    * Generate a property string (property + parameters + ":").
    * @param string $Property The property name.
    * @param array $Parameters Optional parameters for the property. These
    *      should already be properly escaped.
    * @return Returns the generated property string.
    */
    protected function GeneratePropertyString($Property, array $Parameters=array())
    {
        # start the property string off with the property name
        $String = $Property;

        # add each property parameter, if any
        foreach ($Parameters as $Parameter => $Value)
        {
            $String .= ";" . $Parameter . "=" . $Value;
        }

        # add the colon separator and return
        return $String . ":";
    }

    /**
    * Fold a string so that lines are never longer than 75 characters.
    * @param string $String The string to fold.
    * @param string $End Optionally specifieds the line ending sequence.
    * @return Returns the string, folded where necessary.
    */
    protected function FoldString($String, $End="\r\n ")
    {
        # split the line into chunks
        $FoldedString = chunk_split($String, 75, $End);

        # chunk_split() unnecessarily adds the line ending sequence to the end
        # of the string, so remove it
        $FoldedString = substr($FoldedString, 0, -strlen($End));

        return $FoldedString;
    }

    /**
    * The list of components and properties. Not all of the components may be
    * used.
    */
    protected $Properties = array(
        "VEVENT" => array(),
        "VTODO" => array(),
        "VJOURNAL" => array(),
        "VFREEBUSY" => array(),
        "VTIMEZONE" => array(),
        "VALARM" => array());

    /**
    * The summary property for the iCalendar document. Used to generate the file
    * name when set.
    */
    protected $Summary;
}
