<?php
#
#   FILE:  RSSImportUI.php (RSSImport plugin)
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Class representing UI options for the RSSImport plugin.
*/
class RSSImportUI
{

    # ----- PUBLIC INTERFACE -------------------------------------------------

    /**
    * Print a list of imported rss feed items.
    * @param RSSClient $Feed Feed from which items are being printed.
    * @param string $Title Title of the feed being displayed.
    * @param string $Href Link to the source of the RSS feed.
    * @param int $MaxItems Maximum number of feed items to display.
    */
    public static function PrintRssFeedItems(RSSClient $Feed, $Title, $Href, $MaxItems=5)
    {
        $Feed->AutodetectEncoding();
        $Encoding = $Feed->Encoding();
        $Items = $Feed->GetItems($MaxItems);

        $Title = defaulthtmlspecialchars($Title);
        $Href = defaulthtmlspecialchars($Href);

        $Target = ($GLOBALS["G_SysConfig"]->ResourceLaunchesNewWindowEnabled())
            ? "_blank" : "_self";
        ?>
        <div class="cw-section cw-section-simple cw-html5-section">
            <div class="cw-section-header cw-html5-header">
                <b><a href="<?PHP print($Href); ?>" target="<?PHP print($Target);
                        ?>"><?PHP print($Title); ?></a></b>
            </div>
            <div class="cw-section-body">
                <ul class="cw-list cw-list-noindent cw-list-dotlist">
                <?PHP
                foreach ($Items as $Item)
                {
                    self::PrintRssFeedItem($Item, $Target, $Encoding);
                }
                ?></ul>
            </div>
        </div>
        <?PHP
    }

    # ----- PRIVATE INTERFACE ------------------------------------------------

    /**
    * Print a single imported rss feed item.
    * @param array $Item A single RSS feed item.
    * @param string $Target Location of the new window to be opened.
    * @param string $Charset Format of the RSS feed item values.
    */
    private static function PrintRssFeedItem(array $Item, $Target, $Charset)
    {
        $DefaultCharset = $GLOBALS["G_SysConfig"]->DefaultCharacterSet();
        $Map = array("Href" => "link", "Title" => "description", "Inner" => "title");

        foreach ($Map as $VarName => $KeyName)
        {
            $$VarName = (isset($Item[$KeyName])) ? strip_tags($Item[$KeyName]) : NULL;
            $$VarName = self::EscapeRssValue($$VarName, $DefaultCharset, $Charset);
        }

        ?><li><a href="<?PHP print($Href); ?>" target="<?PHP print($Target);
                ?>" title="<?PHP print($Title); ?>"><?PHP print($Inner); ?></a></li>
        <?PHP
    }

    /**
    * Convert rss feed values into either UTF-8 or ISO-8859-1 format if necessary.
    * @param string $Value RSS feed value.
    * @param string $OutputCharset Desired output format.
    * @param string $FeedCharset Feed value input format.
    * @return string RSS value in the desired format.
    */
    private static function EscapeRssValue($Value, $OutputCharset, $FeedCharset)
    {
        # escape special characters
        $Result = safehtmlspecialchars($Value, ENT_QUOTES, $FeedCharset, FALSE);

        # translate IS0-8859-1 to UTF-8
        if (strcasecmp($OutputCharset, "UTF-8") === 0
                && strcasecmp($FeedCharset, "ISO-8859-1") === 0)
        {
            $Result = utf8_encode($Value);
        }
        # translate UTF-8 to IS0-8859-1
        elseif (strcasecmp($OutputCharset, "ISO-8859-1") === 0
                && strcasecmp($FeedCharset, "UTF-8") === 0)
        {
            $Result = utf8_decode($Value);
        }

        return $Result;
    }
}