<?PHP
#
#   FILE:  StdLib.php
#
#   Part of the ScoutLib application support library
#   Copyright 2016 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu
#

/**
* Standard utility library.
* \nosubgrouping
*/
class StdLib
{

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Get info about call to current function.
    * @param string $Element Name of element to return.  (OPTIONAL, defaults
    *       to returning all elements)
    * @return array Array with the element names "FileName", "FullFileName",
    *       "RelativeFileName", and "LineNumber" for the index.
    */
    public static function GetCallerInfo($Element = NULL)
    {
        $Trace = version_compare(PHP_VERSION, "5.4.0", ">=")
                ? debug_backtrace(FALSE, 2) : debug_backtrace(FALSE);
        $FullFileName = $Trace[1]["file"];
        $Info = [
                "FileName" => basename($FullFileName),
                "RelativeFileName" => str_replace(getcwd()."/", "", $FullFileName),
                "FullFileName" => $FullFileName,
                "LineNumber" => $Trace[1]["line"],
                ];
        return ($Element === NULL) ? $Info : $Info[$Element];
    }

    /**
    * Get string with file and line number for call to current function.
    * @return string String with caller info in the form "FILE:LINE".
    */
    public static function GetMyCaller()
    {
        $Trace = version_compare(PHP_VERSION, "5.4.0", ">=")
                ? debug_backtrace(FALSE, 2) : debug_backtrace(FALSE);
        $FileName = $Trace[1]["file"];
        $Caller = basename($FileName).":".$Trace[1]["line"];
        return $Caller;
    }

    /**
    * Check the caller of the current function.  In the desired caller
    * parameter, if a file name is specified it should include the ".php"
    * extension but should not have a leading path.  In the exception
    * message parameter, the following strings can be used and the
    * appropriate values will be substituted in:  %FILE% (no leading path),
    * %LINE%, %FULLFILE% (includes leading path), %CLASS%, %FUNCTION%,
    * and %METHOD% (equivalent to "%CLASS%::%FUNCTION%").
    * @param string $DesiredCaller String describing desired caller, in
    *       the form "Class", "Class::Method", "Function", "File", or
    *       "File:Line".
    * @param string $ExceptionMsg If specified and the caller was not the
    *       desired caller, an exception will be thrown with this message.
    *       (OPTIONAL)
    * @return bool TRUE if caller matched desired caller, otherwise FALSE.
    */
    public static function CheckMyCaller($DesiredCaller, $ExceptionMsg = NULL)
    {
        # retrieve caller info
        $Trace = version_compare(PHP_VERSION, "5.4.0", ">=")
                ? debug_backtrace(FALSE, 3) : debug_backtrace(FALSE);
        $FullFile = $Trace[1]["file"];
        $File = basename($FullFile);
        $Line = $Trace[1]["line"];
        $Class = isset($Trace[2]["class"]) ? $Trace[2]["class"] : "";
        $Function = isset($Trace[2]["function"]) ? $Trace[2]["function"] : "";

        # if caller does not match desired caller
        if (($DesiredCaller != $Class)
                && ($DesiredCaller != $Class."::".$Function)
                && ($DesiredCaller != $Class.$Function)
                && ($DesiredCaller != $File)
                && ($DesiredCaller != $File.":".$Line))
        {
            # if exception message supplied
            if ($ExceptionMsg !== NULL)
            {
                # make any needed substitutions in exception message
                $Msg = str_replace(
                        array(
                            "%FILE%",
                            "%LINE%",
                            "%FULLFILE%",
                            "%CLASS%",
                            "%FUNCTION%",
                            "%METHOD%"),
                        array(
                            $File,
                            $Line,
                            $FullFile,
                            $Class,
                            $Function,
                            $Class."::".$Function),
                        $ExceptionMsg);

                # throw exception
                throw new Exception($Msg);
            }
            else
            {
                # report to our caller that their caller was not the desired one
                return FALSE;
            }
        }

        # report to our caller that their caller was not the desired one
        return TRUE;
    }

    /**
    * Get backtrace as a string.
    * @param bool $IncludeArgs If TRUE, arguments will be included in function
    *       call information.  (OPTIONAL, defaults to TRUE)
    * @return string Backtrace info string.
    */
    public static function GetBacktraceAsString($IncludeArgs = TRUE)
    {
        # get backtrace text
        ob_start();
        $TraceOpts = $IncludeArgs ? 0 : DEBUG_BACKTRACE_IGNORE_ARGS;
        debug_print_backtrace($TraceOpts);
        $TraceString = ob_get_contents();
        ob_end_clean();

        # remove this function from backtrace entries
        $TraceString = preg_replace(
                "/^#0\s+".__METHOD__."[^\n]*\n/", "", $TraceString, 1);

        # renumber backtrace entries
        $TraceString = preg_replace_callback("/^#(\d+)/m", function($Matches)
        {
            return "#".($Matches[1] - 1);
        },
        $TraceString);

        # strip leading path off files names
        $HomeDir = dirname($_SERVER["SCRIPT_FILENAME"]);
        $TraceString = preg_replace("%".preg_quote($HomeDir, "%")."/%",
                "", $TraceString);

        # return backtrace string to caller
        return $TraceString;
    }

    /**
    * Pluralize an English word.
    * @param string $Word Word to make plural.
    * @return string Word in plural form.
    */
    public static function Pluralize($Word)
    {
        # return word unchanged if singular and plural are the same
        if (in_array(strtolower($Word), self::$UncountableWords))
        {
            return $Word;
        }

        # check for irregular singular forms
        foreach (self::$IrregularWords as $Pattern => $Result)
        {
            $Pattern = '/'.$Pattern.'$/i';
            if (preg_match($Pattern, $Word))
            {
                return preg_replace($Pattern, $Result, $Word);
            }
        }

        # check for matches using regular expressions
        foreach (self::$PluralizePatterns as $Pattern => $Result)
        {
            if (preg_match($Pattern, $Word))
            {
                return preg_replace($Pattern, $Result, $Word);
            }
        }

        # return word unchanged if we could not process it
        return $Word;
    }

    /**
    * Singularize an English word.
    * @param string $Word Word to make singular.
    * @return string Word in singular form.
    */
    public static function Singularize($Word)
    {
        # return word unchanged if singular and plural are the same
        if (in_array(strtolower($Word), self::$UncountableWords))
        {
            return $Word;
        }

        # check for irregular plural forms
        foreach (self::$IrregularWords as $Result => $Pattern)
        {
            $Pattern = '/'.$Pattern.'$/i';
            if (preg_match($Pattern, $Word))
            {
                return preg_replace($Pattern, $Result, $Word);
            }
        }

        # check for matches using regular expressions
        foreach (self::$SingularizePatterns as $Pattern => $Result)
        {
            if (preg_match($Pattern, $Word))
            {
                return preg_replace($Pattern, $Result, $Word);
            }
        }

        # return word unchanged if we could not process it
        return $Word;
    }

    /**
    * Attempt to truncate a string as neatly as possible with respect to word
    * breaks, punctuation, and HTML tags.
    * @param string $String String to truncate
    * @param int $MaxLength The maximum length of the truncated string
    * @param bool $BreakAnywhere TRUE to break exactly at the maximum length
    * @return the (possibly) truncated string
    */
    public static function NeatlyTruncateString($String, $MaxLength, $BreakAnywhere=FALSE)
    {
        $TagStrippedString = strip_tags(html_entity_decode($String));

        # if the string contained no HTML tags, we can just treat it as text
        if ($String == $TagStrippedString)
        {
            $Length = self::strlen($String);

            # if string was short enough, we need not do anything
            if ($Length <= $MaxLength)
            {
                return $String;
            }

            # if BreakAnywhere is set, just chop at the max length
            if ($BreakAnywhere)
            {
                $BreakPos = $MaxLength;
            }
            # otherwise look for an acceptable breakpoint
            else
            {
                $BreakPos = self::strrpos($String, " ", -($Length - $MaxLength));

                # if we couldn't find a breakpoint, just chop at max length
                if ($BreakPos === FALSE)
                {
                    $BreakPos = $MaxLength;
                }
            }

            $Result = self::substr($String, 0, $BreakPos);

            # tack on the ellipsis
            $Result .= "...";
        }
        # otherwise, we're in an HTML string and we have to account for
        # how many characters are actually displayed when the string will be
        # rendered
        else
        {
            # if there aren't enough non-whitespace displayed characters to
            # exceed the max length, bail because we don't need to do
            # anything
            if (self::strlen(trim($TagStrippedString)) <= $MaxLength)
            {
                return $String;
            }

            # okay, the hard way -- we have to do some limited parsing
            # of html and attempt to count the number of printing characters
            # as we're doing that.  to accomplish this, we'll build a
            # little state machine and iterate over the characters one at a
            # time

            # split the string into characters (annoyingly, split()/mb_split()
            # cannot do this, so we have to use preg_split() in unicode mode)
            $Tokens = preg_split('%%u', $String, -1, PREG_SPLIT_NO_EMPTY);

            # define our states
            $S_Text = 0;
            $S_MaybeTag = 1;
            $S_MaybeEntity = 2;
            $S_Tag = 3;
            $S_Quote = 4;

            # max length of an HTML Entity
            $MaxEntityLength = 8;

            # track how much we have displayed
            $DisplayedLength = 0;

            $Buffer = "";   # for characters we're not sure about
            $BufLen = 0;    # count of buffered characters
            $Result = "";   # for characters we've included
            $QuoteChar =""; # quote character in use

            # start in the 'text state'
            $State = $S_Text;

            # iterate over all our tokens
            foreach ($Tokens as $Token)
            {
                switch ($State)
                {
                    # text state handles words that will be displayed
                    case $S_Text:
                        switch($Token)
                        {
                            # look for characters that can end a word
                            case "<":
                            case "&":
                            case " ":
                                # if we've buffered up a word
                                if ($BufLen > 0)
                                {
                                    # and if displaying that word exceeds
                                    # our length, then we're done
                                    if ($DisplayedLength + $BufLen > $MaxLength)
                                    {
                                        break 3;
                                    }

                                    # otherwise, add the buffered word to our display
                                    $Result .= $Buffer;
                                    $DisplayedLength += $BufLen;
                                }

                                # buffer this character
                                $Buffer = $Token;
                                $BufLen = 1;

                                # if it could have been the start of a tag or an entity,
                                # change state appropriately
                                if ($Token != " ")
                                {
                                    $State = ($Token == "<") ? $S_MaybeTag :
                                           $S_MaybeEntity;
                                }
                                break;

                                # for characters that can't break a word, just buffer them
                            default:
                                $Buffer .= $Token;
                                $BufLen++;
                                break;
                        }
                        break;

                        # MaybeTag state checks if a < began a tag or not
                    case $S_MaybeTag:
                        # tags start with alpha characters (like <b>)
                        # or a slash (like </b>)
                        if (ctype_alpha($Token) || $Token == "/")
                        {
                            # if this was a tag, output it, output it,
                            # clear our buffer, and move to the Tag state
                            $Result .= $Buffer.$Token;
                            $Buffer = "";
                            $BufLen = 0;
                            $State = $S_Tag;
                        }
                        else
                        {
                            # otherwise, check if displaying this character would
                            # exceed our length.  bail if so
                            if ($DisplayedLength + 1 > $MaxLength)
                            {
                                break 2;
                            }
                            # if not, output the characters, clear our buffer,
                            # move to the Text state
                            $Result .= $Buffer.$Token;
                            $DisplayedLength++;
                            $Buffer = "";
                            $BufLen = 0;
                            $State = $S_Text;
                        }
                        break;

                        # Tag state processes the contents of a tag
                    case $S_Tag:
                        # always output tag contents
                        $Result .= $Token;

                        # check if this is the beginning of a quoted string,
                        # changing state appropriately if so
                        if ($Token == "\"" || $Token == "'")
                        {
                            $QuoteChar = $Token;
                            $State = $S_Quote;
                        }
                        # if this is the end of the tag, go back to Text state
                        elseif ($Token == ">")
                        {
                            $State = $S_Text;
                        }
                        break;

                        # Quote state processes quoted attributes
                    case $S_Quote:
                        # always output quote contents
                        $Result .= $Token;

                        # if we've found the matching quote character,
                        # return to the Tag state
                        if ($Token == $QuoteChar)
                        {
                            $State = $S_Tag;
                        }
                        break;

                        # MaybeEntity decides if we're enjoying an HTML entity
                        # or just an ampersand
                    case $S_MaybeEntity:
                        # buffer this token
                        $Buffer.= $Token;
                        $BufLen++;

                        # if it was a space, then we're not in an entity
                        # as they cannot contain spaces
                        if ($Token == " ")
                        {
                            # check if we should be fone
                            if ($DisplayedLength + $BufLen > $MaxLength)
                            {
                                break 2;
                            }
                            # if not, output the buffer, clear it, and return to Text
                            $Result .= $Buffer;
                            $DisplayedLength += $BufLen;
                            $Buffer = "";
                            $BufLen = 0;
                            $State = $S_Text;
                        }
                        # if we have &xxxx; then count that as a single character entity,
                        # output it, clear the buffer, and return to Text
                        elseif ($Token == ";")
                        {
                            $Result .= $Buffer;
                            $DisplayedLength++;
                            $Buffer = "";
                            $BufLen = 0;
                            $State = $S_Text;
                        }
                        # if this has been too many characters without a ;
                        # for it to be an entity, return to text
                        elseif ($BufLen > 8)
                        {
                            $State = $S_Text;
                        }

                        break;
                }
            }

            # tack on the ellipsis
            $Result .= "...";

            # if our string contained HTML tags that we may need to close
            if (preg_match_all("%<(/?[a-z]+)[^>]*>%", $Result, $Matches))
            {
                # pull out matches for the names of tags
                $Matches = $Matches[1];

                # build up an array of open tags
                $Tags = array();
                while ( ($Tag = array_shift($Matches)) !== NULL )
                {
                    # if this was not a close tag, prepend it to our array
                    if (self::substr($Tag, 0, 1) != "/")
                    {
                        array_unshift($Tags, $Tag);
                    }
                    # if this tag is not self-closing, append it to our list of open tags
                    elseif (self::substr($Tag, -1) != "/")
                    {
                        # if this was a close tag, look to see if this tag was open
                        $Tgt = array_search(self::substr($Tag, 1), $Tags);
                        if ($Tgt !== FALSE)
                        {
                            # if so, remove this tag from our list
                            unset($Tags[$Tgt]);
                        }
                    }
                }

                # iterate over open tags, closing them as we go
                while ( ($Tag = array_shift($Tags)) !== NULL)
                {
                    $Result .= "</".$Tag.">";
                }
            }
        }

        return $Result;
    }

    /**
    * Multibyte-aware (if supported in PHP) version of substr().
    * (Consult PHP documentation for arguments and return value.)
    */
    public static function substr()
    {
        return self::CallMbStringFuncIfAvailable(__FUNCTION__, func_get_args(), 3);
    }

    /**
    * Multibyte-aware (if supported in PHP) version of strpos().
    * (Consult PHP documentation for arguments and return value.)
    */
    public static function strpos()
    {
        return self::CallMbStringFuncIfAvailable(__FUNCTION__, func_get_args(), 3);
    }

    /**
    * Multibyte-aware (if supported in PHP) version of strrpos().
    * (Consult PHP documentation for arguments and return value.)
    */
    public static function strrpos()
    {
        return self::CallMbStringFuncIfAvailable(__FUNCTION__, func_get_args(), 3);
    }

    /**
    * Multibyte-aware (if supported in PHP) version of strlen().
    * (Consult PHP documentation for arguments and return value.)
    */
    public static function strlen()
    {
        return self::CallMbStringFuncIfAvailable(__FUNCTION__, func_get_args(), 1);
    }

    /**
    * Perform compare and return value appropriate for sort function callbacks.
    * @param mixed $A First value to compare.
    * @param mixed $B Second value to compare.
    * @return int 0 if values are equal, -1 if A is less than B, or 1 if B is
    *       greater than A.
    */
    public static function SortCompare($A, $B)
    {
        if ($A == $B)
        {
            return 0;
        }
        else
        {
            return ($A < $B) ? -1 : 1;
        }
    }

    /**
    * Look up the GPS coordinates for a US ZIP code.  Database of GPS
    * coordinates used was drawn from Census 2010. See the "Zip Code
    * Tabulation Areas" section on
    * https://www.census.gov/geo/maps-data/data/gazetteer2010.html for
    * the original source file.  The version used here has been cut
    * down to columns 1, 8, and 9 from that source.
    * @param int $Zip Zip code to look up.
    * @return array Having members "Lat" and "Lng" on successful
    * lookup, FALSE otherwise
    * @throws Exception When coordinates file cannot be opened.
    */
    public static function GetLatLngForZipCode($Zip)
    {
        static $ZipCache = array();

        # if we don't have a cached value for this zip, look one up
        if (!isset($ZipCache[$Zip]))
        {
            # try to open our zip code database
            $FHandle = fopen(dirname(__FILE__)."/StdLib--ZipCodeCoords.txt", "r");

            # if we couldn't open the file, we can't look up a value
            if ($FHandle === FALSE)
            {
                throw new Exception("Unable to open zip code coordinates file");
            }

            # iterate over our database until we find the desired zip
            # or run out of database
            while (($Line = fgetcsv($FHandle, 0, "\t")) !== FALSE)
            {
                if ($Line[0] == $Zip)
                {
                    $ZipCache[$Zip] = array(
                        "Lat" => $Line[1], "Lng" => $Line[2]);
                    break;
                }
            }

            # if we've scanned the entire file and have no coords for
            # this zip, cache a failure
            if (!isset($ZipCache[$Zip]))
            {
                $ZipCache[$Zip] = FALSE;
            }
        }

        # hand back cached value
        return $ZipCache[$Zip];
    }

    /**
    * Compute the distance between two US ZIP codes.
    * @param int $ZipA First zip code.
    * @param int $ZipB Second zip code.
    * @return double Distance in Km between the two zip codes or FALSE
    *     if either zip could not be found
    */
    public static function ZipCodeDistance($ZipA, $ZipB)
    {

        $FirstPoint = self::GetLatLngForZipCode($ZipA);
        $SecondPoint = self::GetLatLngForZipCode($ZipB);

        # if we scanned the whole file and lack data for either of our
        # points, return NULL
        if ($FirstPoint === FALSE || $SecondPoint === FALSE)
        {
            return FALSE;
        }

        return self::ComputeGreatCircleDistance(
            $FirstPoint["Lat"], $FirstPoint["Lng"],
            $SecondPoint["Lat"], $SecondPoint["Lng"]);
    }

    /**
    * Computes the distance in kilometers between two points, assuming a
    * spherical earth.
    * @param int $LatSrc Latitude of the source coordinate.
    * @param int $LonSrc Longitude of the source coordinate.
    * @param int $LatDst Latitude of the destination coordinate.
    * @param int $LonDst Longitude of the destination coordinate.
    * @return distance in miles between the two points.
    */
    public static function ComputeGreatCircleDistance($LatSrc, $LonSrc,
                             $LatDst, $LonDst)
    {
        # See http://en.wikipedia.org/wiki/Great-circle_distance

        # Convert it all to Radians
        $Ps = deg2rad($LatSrc);
        $Ls = deg2rad($LonSrc);
        $Pf = deg2rad($LatDst);
        $Lf = deg2rad($LonDst);

        # Compute the central angle
        return 3958.756 * atan2(
            sqrt( pow(cos($Pf)*sin($Lf-$Ls), 2) +
                  pow(cos($Ps)*sin($Pf) -
                      sin($Ps)*cos($Pf)*cos($Lf-$Ls), 2)),
                  sin($Ps)*sin($Pf)+cos($Ps)*cos($Pf)*cos($Lf-$Ls));

    }

    /**
    * Computes the initial angle on a course connecting two points, assuming a
    * spherical earth.
    * @param int $LatSrc Latitude of the source coordinate.
    * @param int $LonSrc Longitude of the source coordinate.
    * @param int $LatDst Latitude of the destination coordinate.
    * @param int $LonDst Longitude of the destination coordinate.
    * @return initial angle on a course connecting two points.
    */
    public static function ComputeBearing($LatSrc, $LonSrc,
                                   $LatDst, $LonDst)
    {
        # See http://mathforum.org/library/drmath/view/55417.html

        # Convert angles to radians
        $Ps = deg2rad($LatSrc);
        $Ls = deg2rad($LonSrc);
        $Pf = deg2rad($LatDst);
        $Lf = deg2rad($LonDst);

        return rad2deg(atan2(sin($Lf-$Ls)*cos($Pf),
                             cos($Ps)*sin($Pf)-sin($Ps)*cos($Pf)*cos($Lf-$Ls)));
    }

    /**
    * Return all possible permutations of a given array.
    * @param array $Items Array to permutate.
    * @param array $Perms Current set of permutations, used internally for
    *       recursive calls.  (DO NOT USE)
    * @return array Array of arrays of permutations.
    */
    public static function ArrayPermutations($Items, $Perms = array())
    {
        if (empty($Items))
        {
            $Result = array($Perms);
        }
        else
        {
            $Result = array();
            for ($Index = count($Items) - 1;  $Index >= 0;  --$Index)
            {
                $NewItems = $Items;
                $NewPerms = $Perms;
                list($Segment) = array_splice($NewItems, $Index, 1);
                array_unshift($NewPerms, $Segment);
                $Result = array_merge($Result,
                        self::ArrayPermutations($NewItems, $NewPerms));
            }
        }
        return $Result;
    }

    /**
    * Get an array of US state names with their two-letter abbreviations as the
    * index.
    * @return Returns an array of US state names with their two-letter
    *      abbreviations as the index.
    */
    public static function GetUsStatesList()
    {
        return array(
                "AL" => "Alabama",
                "AK" => "Alaska",
                "AZ" => "Arizona",
                "AR" => "Arkansas",
                "CA" => "California",
                "CO" => "Colorado",
                "CT" => "Connecticut",
                "DE" => "Delaware",
                "DC" => "District of Columbia",
                "FL" => "Florida",
                "GA" => "Georgia",
                "HI" => "Hawaii",
                "ID" => "Idaho",
                "IL" => "Illinois",
                "IN" => "Indiana",
                "IA" => "Iowa",
                "KS" => "Kansas",
                "KY" => "Kentucky",
                "LA" => "Louisiana",
                "ME" => "Maine",
                "MD" => "Maryland",
                "MA" => "Massachusetts",
                "MI" => "Michigan",
                "MN" => "Minnesota",
                "MS" => "Mississippi",
                "MO" => "Missouri",
                "MT" => "Montana",
                "NE" => "Nebraska",
                "NV" => "Nevada",
                "NH" => "New Hampshire",
                "NJ" => "New Jersey",
                "NM" => "New Mexico",
                "NY" => "New York",
                "NC" => "North Carolina",
                "ND" => "North Dakota",
                "OH" => "Ohio",
                "OK" => "Oklahoma",
                "OR" => "Oregon",
                "PA" => "Pennsylvania",
                "RI" => "Rhode Island",
                "SC" => "South Carolina",
                "SD" => "South Dakota",
                "TN" => "Tennessee",
                "TX" => "Texas",
                "UT" => "Utah",
                "VT" => "Vermont",
                "VA" => "Virginia",
                "WA" => "Washington",
                "WV" => "West Virginia",
                "WI" => "Wisconsin",
                "WY" => "Wyoming",
                );
    }

    /** Format to feed to date() to get SQL-compatible date/time string. */
    const SQL_DATE_FORMAT = "Y-m-d H:i:s";


    # ---- PRIVATE INTERFACE -------------------------------------------------

    private static $PluralizePatterns = array(
        '/(quiz)$/i'               => "$1zes",
        '/^(ox)$/i'                => "$1en",
        '/([m|l])ouse$/i'          => "$1ice",
        '/(matr|vert|ind)ix|ex$/i' => "$1ices",
        '/(x|ch|ss|sh)$/i'         => "$1es",
        '/([^aeiouy]|qu)y$/i'      => "$1ies",
        '/(hive)$/i'               => "$1s",
        '/(?:([^f])fe|([lr])f)$/i' => "$1$2ves",
        '/(shea|lea|loa|thie)f$/i' => "$1ves",
        '/sis$/i'                  => "ses",
        '/([ti])um$/i'             => "$1a",
        '/(tomat|potat|ech|her|vet)o$/i'=> "$1oes",
        '/(bu)s$/i'                => "$1ses",
        '/(alias)$/i'              => "$1es",
        '/(octop)us$/i'            => "$1i",
        '/(ax|test)is$/i'          => "$1es",
        '/(us)$/i'                 => "$1es",
        '/s$/i'                    => "s",
        '/$/'                      => "s"
    );
    private static $SingularizePatterns = array(
        '/(quiz)zes$/i'             => "$1",
        '/(matr)ices$/i'            => "$1ix",
        '/(vert|ind)ices$/i'        => "$1ex",
        '/^(ox)en$/i'               => "$1",
        '/(alias)es$/i'             => "$1",
        '/(octop|vir)i$/i'          => "$1us",
        '/(cris|ax|test)es$/i'      => "$1is",
        '/(shoe)s$/i'               => "$1",
        '/(o)es$/i'                 => "$1",
        '/(bus)es$/i'               => "$1",
        '/([m|l])ice$/i'            => "$1ouse",
        '/(x|ch|ss|sh)es$/i'        => "$1",
        '/(m)ovies$/i'              => "$1ovie",
        '/(s)eries$/i'              => "$1eries",
        '/([^aeiouy]|qu)ies$/i'     => "$1y",
        '/([lr])ves$/i'             => "$1f",
        '/(tive)s$/i'               => "$1",
        '/(hive)s$/i'               => "$1",
        '/(li|wi|kni)ves$/i'        => "$1fe",
        '/(shea|loa|lea|thie)ves$/i'=> "$1f",
        '/(^analy)ses$/i'           => "$1sis",
        '/((a)naly|(b)a|(d)iagno|(p)arenthe|(p)rogno|(s)ynop|(t)he)ses$/i'  => "$1$2sis",
        '/([ti])a$/i'               => "$1um",
        '/(n)ews$/i'                => "$1ews",
        '/(h|bl)ouses$/i'           => "$1ouse",
        '/(corpse)s$/i'             => "$1",
        '/(us)es$/i'                => "$1",
        '/s$/i'                     => ""
    );
    private static $IrregularWords = array(
        'move'   => 'moves',
        'foot'   => 'feet',
        'goose'  => 'geese',
        'sex'    => 'sexes',
        'child'  => 'children',
        'man'    => 'men',
        'tooth'  => 'teeth',
        'person' => 'people'
    );
    private static $UncountableWords = array(
        'sheep',
        'fish',
        'deer',
        'series',
        'species',
        'money',
        'rice',
        'information',
        'equipment'
    );

    /**
    * Call PHP multibyte string function if available, otherwise call plain
    * version of function.
    * @param string $Func Name of plain function.
    * @param array $Args Argument list to pass to function.
    * @param int $NumPlainArgs Number of arguments to plain version of function.
    */
    private static function CallMbStringFuncIfAvailable($Func, $Args, $NumPlainArgs)
    {
        if (function_exists("mb_".$Func))
        {
            $FuncToCall = "mb_".$Func;
        }
        else
        {
            if (count($Args) > $NumPlainArgs)
            {
                throw new Exception(
                        "Function mb_".$Func."() required but not available.");
            }
            $FuncToCall = $Func;
        }
        return call_user_func_array($FuncToCall, $Args);
    }
}
