<?PHP
#
#   FILE:  VocabularyFactory.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2007-2016 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Factory for manipulating Vocabulary objects.
*/
class VocabularyFactory
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * object constructor
    * @param string $Path Path to directory containing .voc controlled
    *       vocabulary files.
    */
    public function __construct($Path)
    {
        $this->Path = $Path;
    }

    /**
    * load vocabulary objects from files
    * @return Array of Vocabulary objects.
    * @note Vocabularies are returned sorted by name.
    */
    public function GetVocabularies()
    {
        # load vocabularies (if any)
        $Vocabularies = array();
        $VocFileNames = $this->GetFileList();
        foreach ($VocFileNames as $FileName)
        {
            $Vocabularies[] = new Vocabulary($FileName);
        }

        # sort vocabularies by name
        $SortFunction = create_function('$VocA, $VocB', '
                $NameA = $VocA->Name();
                $NameB = $VocB->Name();
                return ($NameA == $NameB) ? 0 : (($NameA < $NameB) ? -1 : 1);
                ');
        usort($Vocabularies, $SortFunction);

        # return array of vocabularies to caller
        return $Vocabularies;
    }

    /**
    * retrieve vocabulary object based on hash string
    * @param string $Hash Hash for Vocabulary (returned by Hash() method).
    * @return Vocabulary object corresponding to hash, or NULL if no matching
    *       vocabulary found.
    */
    public function GetVocabularyByHash($Hash)
    {
        # for each available vocabulary file
        $Vocab = NULL;
        $VocFileNames = $this->GetFileList();
        foreach ($VocFileNames as $FileName)
        {
            # if hash for vocabulary file matches specified hash
            if (Vocabulary::HashForFile($FileName) == $Hash)
            {
                # load vocabulary and stop searching file list
                $Vocab = new Vocabulary($FileName);
                break;
            }
        }

        # return matching vocabulary (if any) to caller
        return $Vocab;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    /**  Path to directory containing vocabulary files. */
    private $Path;

    /**
    * Read in list of available vocabulary files.
    * @return Array with full paths to vocabulary files.
    */
    private function GetFileList()
    {
        $VocFiles = array();
        if (is_dir($this->Path))
        {
            $AllFiles = scandir($this->Path);
            foreach ($AllFiles as $FileName)
            {
                if (preg_match("/\\.voc\$/i", $FileName))
                {
                    $VocFiles[] = realpath($this->Path."/".$FileName);
                }
            }
        }
        return $VocFiles;
    }
}

