<?PHP
#
#   FILE:  BackwardCompatibility.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2012-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Plugin to add measures of backward compatibility with 1.x versions of CWIS.
*/
class BackwardCompatibility extends Plugin
{

    # ---- STANDARD PLUGIN INTERFACE -----------------------------------------

    /**
    * Set plugin attributes.
    */
    public function Register()
    {
        $this->Name = "Backward Compatibility";
        $this->Version = "1.0.4";
        $this->Description = "Improves backwards compatibility with code"
                ." written for older versions of CWIS.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
                "CWISCore" => "2.3.2");
        $this->EnabledByDefault = FALSE;

        $this->CfgSetup["ContextFilterBypass"] = array(
                "Label" => "Context Filter Bypass",
                "Type" => "Flag",
                "Default" => TRUE,
                "Help" => "When the bypass is enabled, no filtering of"
                        ." which variables are passed between the page (PHP),"
                        ." interface (HTML), StdPageStart, StdPageEnd, and"
                        ." other common files will occur."
                );
        $this->CfgSetup["CreateFileStubs"] = array(
                "Label" => "Create Legacy File Stubs",
                "Type" => "Flag",
                "Default" => FALSE,
                "Help" => "When this option is enabled, zero-length versions"
                        ." of some legacy files will be created, to prevent"
                        ." errors in legacy code when attempting to load those"
                        ." files."
                );
    }

    /**
    * Initialize the plugin.  This is called after all plugins have been loaded
    * but before any methods for this plugin (other than Register() or Initialize())
    * have been called.
    * @return NULL if initialization was successful, otherwise a string containing
    *       an error message indicating why initialization failed.
    */
    public function Initialize()
    {
        # set up old configuration references
        $GLOBALS["SPT_DBUserName"] = $GLOBALS["G_Config"]["Database"]["UserName"];
        $GLOBALS["SPT_DBPassword"] = $GLOBALS["G_Config"]["Database"]["Password"];
        $GLOBALS["SPT_DBHost"] =     $GLOBALS["G_Config"]["Database"]["Host"];
        $GLOBALS["SPT_DBName"] =     $GLOBALS["G_Config"]["Database"]["DatabaseName"];
        $GLOBALS["SPT_DefaultUI"] =  $GLOBALS["G_Config"]["UserInterface"]["DefaultUI"];

        # set up possible old UI directories and file names to be included
        #       when searching for UI files
        $GLOBALS["AF"]->AddIncludeDirectories(array(
                "local/interface/%ACTIVEUI%/include/SPT--",
                "local/interface/SPT--%ACTIVEUI%/include/SPT--",
                "local/interface/default/include/SPT--",
                ), FALSE, TRUE);
        $GLOBALS["AF"]->AddIncludeDirectories(array(
                "interface/%ACTIVEUI%/include/SPT--",
                "SPTUI--%ACTIVEUI%/include/",
                "SPTUI--%ACTIVEUI%/include/SPT--",
                "%ACTIVEUI%/include/",
                "%ACTIVEUI%/include/SPT--",
                "interface/default/include/SPT--",
                ), TRUE, TRUE);
        $GLOBALS["AF"]->AddInterfaceDirectories(array(
                "local/interface/%ACTIVEUI%/SPT--",
                "local/interface/SPT--%ACTIVEUI%/SPT--",
                "local/interface/default/SPT--",
                ), FALSE, TRUE);
        $GLOBALS["AF"]->AddInterfaceDirectories(array(
                "interface/%ACTIVEUI%/SPT--",
                "SPTUI--%ACTIVEUI%/",
                "SPTUI--%ACTIVEUI%/SPT--",
                "%ACTIVEUI%/",
                "%ACTIVEUI%/SPT--",
                "interface/default/SPT--",
                ), TRUE, TRUE);
        $GLOBALS["AF"]->AddImageDirectories(array(
                "local/interface/SPT--%ACTIVEUI%/images/",
                ), FALSE, TRUE);
        $GLOBALS["AF"]->AddImageDirectories(array(
                "SPTUI--%ACTIVEUI%/images/",
                "%ACTIVEUI%/images/",
                ), TRUE, TRUE);
        $GLOBALS["AF"]->AddFunctionDirectories(array(
                "local/interface/SPT--%ACTIVEUI%/include/",
                ), FALSE, TRUE);
        $GLOBALS["AF"]->AddFunctionDirectories(array(
                "SPTUI--%ACTIVEUI%/include/",
                "%ACTIVEUI%/include/",
                ), TRUE, TRUE);

        # set up variable filter bypass if requested
        if ($this->ConfigSetting("ContextFilterBypass"))
        {
            $GLOBALS["AF"]->SetContextFilter(
                    ApplicationFramework::CONTEXT_ENV, TRUE);
            $GLOBALS["AF"]->SetContextFilter(
                    ApplicationFramework::CONTEXT_PAGE, TRUE);
            $GLOBALS["AF"]->SetContextFilter(
                    ApplicationFramework::CONTEXT_COMMON, TRUE);
            $GLOBALS["AF"]->SetContextFilter(
                    ApplicationFramework::CONTEXT_INTERFACE, TRUE);
        }

        # create links to files possibly referenced in old (pre-2.0) code.
        if ($this->ConfigSetting("CreateFileStubs"))
        {
            $this->CreateFileStubs();
        }

        # report that initialization was successful
        return NULL;
    }

    /**
    * Perform any work needed when the plugin is uninstalled.
    * @return NULL if uninstall succeeded, otherwise a string containing
    *       an error message indicating why uninstall failed.
    */
    public function Uninstall()
    {
        $this->RemoveFileStubs();
        return NULL;
    }

    /**
    * Create links/stubs to files possibly referenced in old (pre-2.0) code.
    */
    private function CreateFileStubs()
    {
        foreach ($this->NeededFiles as $FileName)
        {
            if (!file_exists($FileName)) {  touch($FileName);  }
        }
        foreach ($this->NeededLinks as $Source => $Destination)
        {
            if (!file_exists($Source)) {  symlink($Destination, $Source);  }
        }
    }

    /**
    * Remove links/stubs to files possibly referenced in old (pre-2.0) code.
    */
    private function RemoveFileStubs()
    {
        foreach ($this->NeededFiles as $FileName)
        {
            if (file_exists($FileName) && (filesize($FileName) == 0))
            {
                unlink($FileName);
            }
        }
        foreach ($this->NeededLinks as $Source => $Destination)
        {
            if (file_exists($Source) && is_link($Source))
            {
                unlink($Source);
            }
        }
    }

    /**
    * Hook the events into the application framework.
    * @return Returns an array of events to be hooked into the application
    *      framework.
    */
    public function HookEvents()
    {
        return array(
            "EVENT_PAGE_OUTPUT_FILTER" => "PageOutputFilter",
            "EVENT_PLUGIN_CONFIG_CHANGE" => "PluginConfigChange",
            );
    }

    /**
    * Handle plugin configuration changes.
    * @param string $PluginName Name of the plugin that has changed.
    * @param string $ConfigSetting Name of the setting that has change.
    * @param mixed $OldValue The old value of the setting.
    * @param mixed $NewValue The new value of the setting.
    */
    public function PluginConfigChange($PluginName, $ConfigSetting, $OldValue, $NewValue)
    {
        # only worried about changes to our settings
        if ($PluginName != $this->Name) {  return;  }

        # add/remove file stubs if that setting has changed
        if ($ConfigSetting == "CreateFileStubs")
        {
            if ($NewValue)
            {
                $this->CreateFileStubs();
            }
            else
            {
                $this->RemoveFileStubs();
            }
        }
    }

    /**
    * Rewrite HTML to deal with files that have been moved.
    * @param string $PageOutput HTML text of page for modification.
    * @return array ("PageOutput" => $PageOutput) with modified page contents.
    */
    public function PageOutputFilter($PageOutput)
    {
        global $AF;

        # define the replacements we want to do
        $Replacements = array(
            "%include/SPT--EmailMunge\.js%" => $AF->GUIFile("SPT--EmailMunge.js"),
            "%include/SPT--FastRating\.js%" => $AF->GUIFile("SPT--FastRating.js"),
            "%include/jsbn/prng4\.js%" => $AF->GUIFile("prng4.js"),
            "%include/jsbn/rng\.js%" => $AF->GUIFile("rng.js"),
            "%include/jsbn/jsbn\.js%" => $AF->GUIFile("jsbn.js"),
            "%include/jsbn/rsa\.js%" => $AF->GUIFile("rsa.js"),
            "%include/jsbn/base64\.js%" => $AF->GUIFile("base64.js"),
            );

        $PageOutput = preg_replace(
            array_keys($Replacements),
            $Replacements,
            $PageOutput
            );

        return array("PageOutput" => $PageOutput);
    }

    private $NeededFiles = array(
            "include/SPT--Classification.php",
            "include/SPT--Common.php",
            "include/SPT--ControlledName.php",
            "include/SPT--ControlledNameFactory.php",
            "include/SPT--ItemFactory.php",
            "include/SPT--MetadataSchema.php",
            "include/SPT--Resource.php",
            "include/SPT--SavedSearch.php",
            "include/SPT--SearchEngine.php",
            "include/SPT--SPTDatabase.php",
            "include/SPT--SPTEventLog.php",
            "include/SPT--SPTImage.php",
            "include/SPT--SPTUser.php",
            "include/DummyFile.php",
            "Axis--User.php",
            "Axis--Date.php",
            "Axis--Session.php",
            );
    private $NeededLinks = array(
            "SPTUI--CWIS" => "interface/default",
            );
}
