<?PHP
#
#   FILE:  MLPHPList.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2014 Internet Scout Project
#   http://scout.wisc.edu/cwis
#

class MLPHPList extends Plugin
{
    /**
    * Register information about this plugin.
    */
    public function Register()
    {
        $this->Name = "PHPList";
        $this->Version = "1.1.0";
        $this->Description = "MailingList backend plugin to support PHPList.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = [
            "CWISCore" => "2.2.3",
            "MailingList" => "1.0.0",
        ];
        $this->EnabledByDefault = FALSE;

        $this->InitializeBefore = ["MailingList"];

        $this->CfgSetup["RESTUrl"] = [
            "Type" => "URL",
            "Label" => "PHPList REST API URL",
            "MaxLength" => 255,
            "Help" => "The URL to use for sending commands to PHPList.",
        ];

        $this->CfgSetup["PHPListUser"] = [
            "Type" => "Text",
            "Label" => "PHPList Username",
            "Help" => "PHPList administrative user to use",
        ];

        $this->CfgSetup["PHPListPass"] = [
            "Type" => "Text",
            "Label" => "PHPList Password",
            "Help" => "Password for the PHPList admin user",
        ];
    }

    /**
    * Perform plugin installation.
    * @return NULL on success, an error string otherwise.
    */
    public function Install()
    {
        return $this->CreateTables($this->SqlTables);
    }

    /**
    * Perform plugin upgrades.
    * @param string $PreviousVersion Previously installed version.
    * @return NULL on success, an error string otherwise.
    */
    public function Upgrade($PreviousVersion)
    {
        if (version_compare($PreviousVersion, "1.1.0", "<"))
        {
            return $this->CreateTables($this->SqlTables);
        }

        return NULL;
    }

    /**
    * Make sure the configuration values are valid
    * @return null|string NULL if there are no errors or an error message
    */
    public function Initialize()
    {
        # verify that all our required parameters have been configured
        $RESTUrl = $this->ConfigSetting("RESTUrl");
        $PHPListUser = $this->ConfigSetting("PHPListUser");
        $PHPListPass = $this->ConfigSetting("PHPListPass");

        $Errors = [];

        # make sure that login creds were specified
        if (strlen($RESTUrl)==0)
        {
            $Errors[]= "No REST Url was specified." ;
        }

        if (filter_var($RESTUrl, FILTER_VALIDATE_URL)===FALSE)
        {
            $Errors[]= "REST Url setting is invalid.";
        }

        if (strlen($PHPListUser)==0)
        {
            $Errors[]= "No PHPList Username was specified." ;
        }

        if (strlen($PHPListPass)==0)
        {
            $Errors[]= "No PHPList Password was specified.";
        }

        if (count($Errors)>0)
        {
            return $Errors;
        }

        # if login creds have changed, verify that we can use them
        $ConfigHash = md5($RESTUrl.$PHPListUser.$PHPListPass);
        if ($this->ConfigSetting("PreviousConfigHash") != $ConfigHash)
        {
            if ($this->GetAllLists() === NULL)
            {
                return "Could not connect to PHPList with the provided credentials.";
            }
            else
            {
                $this->ConfigSetting("PreviousConfigHash", $ConfigHash);
            }
        }

        # register ourselves
        MailingList::RegisterBackend($this->Name, "PHPList Support", $this);

        $this->DB = new Database();
    }

    /**
    * Update the user's mailing list subscriptions when his or her e-mail
    * address has changed.
    * @param string $OldEmail Previous e-mail address for the user.
    * @param string $NewEmail New e-mail address for the user.
    */
    public function UpdateUserEmail($OldEmail, $NewEmail)
    {
        # query remote user with a REST command
        $UserInfo = $this->GetUserInfo($OldEmail);

        # if no user was found, then there's no user to update
        if ($UserInfo===NULL)
        {
            return;
        }

        # otherwise, update user with new email address
        $this->RestCommand([
            "cmd" => "userUpdate",
            "id" => $UserInfo["id"],
            "email" => $NewEmail,
            "confirmed" => $UserInfo["confirmed"],
            "htmlemail" => $UserInfo["htmlemail"],
            "rssfrequency" => $UserInfo["rssfrequency"],
            "password" => $UserInfo["password"],
            "disabled" => $UserInfo["disabled"] ]);

        # and update our locally cached value
        $this->DB->Query(
            "UPDATE MLPHPList_Users "
            ."SET Email='".addslashes($NewEmail)."' "
            ."WHERE Email='".addslashes($OldEmail)."'");
    }

    /**
    * Get all available mailing lists.  Lists are fetched once per
    * object.
    * @return array all available mailing lists, keyed by phplist's listid.
    */
    public function GetAllLists()
    {
        if (!isset($this->Lists))
        {
            $this->Lists = [];

            $Result = $this->RestCommand(["cmd" => "listsGet"]);
            if ($Result !== NULL)
            {
                foreach ($Result as $Item)
                {
                    $this->Lists[$Item["id"]] = $Item["name"];
                }
            }
        }

        return $this->Lists;
    }

    /**
    * Get the mailing lists to which the given user is subscribed.
    * @param string $EMail Address for the user.
    * @return array Names of mailing lists to which the user is subscribed.
    */
    public function GetUserSubscriptions($EMail)
    {
        # grab the users's phplist information
        $UserId = $this->GetRemoteUserId($EMail);

        # no corresponding phplist user -> no subscriptions to get
        if ($UserId === NULL)
        {
            return [];
        }

        # check for a cached set of subscriptions
        $this->DB->Query("SELECT ListId FROM MLPHPList_ListUserInts "
            ."WHERE UserId=".$UserId);

        # if one was found
        if ($this->DB->NumRowsSelected()>0)
        {
            # pull out the ListIds (filtering the NULL that marks this
            # user's cache as valid)
            $ListIds = array_filter($this->DB->FetchColumn("ListId"));

            # get all the lists
            $Lists = $this->GetAllLists();

            # convert PHP ListIds to list names for the user's subscriptions
            $Subscriptions = [];
            foreach ($ListIds as $ListId)
            {
                $Subscriptions[]= $Lists[$ListId];
            }

            return $Subscriptions;
        }
        else
        {
            #  fetch the user's lists via REST command
            $UserLists = $this->RestCommand(
                ["cmd" => "listsUser", "user_id" => $UserId] );

            $this->UpdateUserSubsCache($UserId, $UserLists);

            # iterate over the returned lists
            $Subscriptions = [];
            foreach ($UserLists as $UserList)
            {
                $Subscriptions[]= $UserList["name"];
            }

            return $Subscriptions;
        }
    }

    /**
    * Subscribe the given user to the given mailing list.
    * @param string $Email Address for user to subscribe.
    * @param string $List Mailing list.
    * @return bool TRUE on success.
    */
    public function Subscribe($Email, $List)
    {
        $ListId = array_search($List, $this->GetAllLists());

        # check if phplist has a user corresponding to this CWIS user
        $UserId = $this->GetRemoteUserId($Email);

        # if not, create one
        if ($UserId === NULL)
        {
            $UserInfo = $this->RestCommand([
                "cmd" => "userAdd",
                "email" => $Email,
                "confirmed" => 1,
                "htmlemail" => 1,
                "rssfrequency" => 0,
                "password" => md5( base64_encode( openssl_random_pseudo_bytes(15) )),
                "disabled" => 0]);

            $UserId = $UserInfo["id"];
        }

        # add the user to the specified list
        #  (result is an array of lists to which the user is subscribed)
        $UserLists = $this->RestCommand([
            "cmd" => "listUserAdd",
            "list_id" => $ListId,
            "user_id" => $UserId,
        ]);

        # update their cached subscriptions
        $this->UpdateUserSubsCache($UserId, $UserLists);

        # return TRUE if the user was successfully subscribed to the
        # requested list
        foreach ($UserLists as $UserList)
        {
            if ($UserList["name"] == $List)
            {
                return TRUE;
            }
        }

        # or false if the user was not subscribed
        return FALSE;
    }

    /**
    * Unsubscribe the given user from the given mailing list.
    * @param string $Email Address for user to unsubscribe.
    * @param string $List Mailing list.
    * @return bool TRUE on success.
    */
    public function Unsubscribe($Email, $List)
    {
        $ListId = array_search($List, $this->GetAllLists());

        # pull up the corresponding phplist user
        $UserId = $this->GetRemoteUserId($Email);

        # if no user was found, then we have nothing to do as they
        # couldn't have been subscribed in the first place
        if ($UserId === NULL)
        {
            return FALSE;
        }

        # delete the user from this list
        $Result = $this->RestCommand([
            "cmd" => "listUserDelete",
            "list_id" => $ListId,
            "user_id" => $UserId,
        ]);

        # check that the command was successful
        if ($Result == "User ".$UserId." is unassigned from list ".$ListId)
        {
            $this->DB->Query(
                "DELETE FROM MLPHPList_ListUserInts "
                ."WHERE UserId=".intval($UserId)." AND ListId=".intval($ListId));
            return TRUE;
        }
        else
        {
            return FALSE;
        }
    }

    /**
    * Get the list of subscribers to the given list.
    * @param string $List Mailing list name.
    * @return array Email addresses subscribed to the given list.
    */
    public function GetSubscribers($List)
    {
        # look up the ListId for the given list
        $ListId = array_search($List, $this->GetAllLists());

        # if the provided list is not valid, it will have no
        # subscribers
        if ($ListId === FALSE)
        {
            return [];
        }

        # extract all known users
        $AllUsers = $this->RestCommand(
            ["cmd" => "usersGet", "limit" => PHP_INT_MAX] );

        # if no users were obtained, nobody is subscribed to anything
        if (!is_array($AllUsers))
        {
            return [];
        }

        # get the list of remote userids we have cached emails for
        $this->DB->Query("SELECT RemoteUserId FROM MLPHPList_Users");
        $KnownUIds = array_flip($this->DB->FetchColumn("RemoteUserId"));

        # get list of userids we have cached subscription info for
        $this->DB->Query("SELECT UserId FROM MLPHPList_ListUserInts");
        $KnownUIdSubs = array_flip($this->DB->FetchColumn("UserId"));

        # iterate over them, fetching list info for each as needed
        foreach ($AllUsers as $UserInfo)
        {
            # if we lack a cached email for this user
            if (!isset($KnownUIds[$UserInfo["id"]]))
            {
                # update our cache
                $this->DB->Query(
                    "INSERT INTO MLPHPList_Users (RemoteUserId, Email) "
                    ."VALUES (".intval($UserInfo["id"]).","
                    ."'".addslashes($UserInfo["email"])."')");
            }

            # if we don't know this user's subscriptions
            if (!isset($KnownUIdSubs[$UserInfo["id"]]))
            {
                # fetch user's lists
                $UserLists = $this->RestCommand(
                    ["cmd" => "listsUser", "user_id" => $UserInfo["id"]] );

                # update cached data
                $this->UpdateUserSubsCache($UserInfo["id"], $UserLists);
            }
        }

        # now we know the cache is fully populated, pull our list of
        # subscribers from it
        $this->DB->Query(
            "SELECT Email FROM MLPHPList_Users "
            ."WHERE RemoteUserId IN (SELECT UserId FROM "
            ."MLPHPList_ListUserInts WHERE ListId=".intval($ListId).")");

        return $this->DB->FetchColumn("Email");
    }

    /**
    * Provide the PerformDailyMaintenance function that MailingList
    * calls in its daily task.
    */
    public function PerformDailyMaintenance()
    {
        $this->PerformPeriodicMaintenance();
    }

    /**
    * Perform periodic cleaning of locally cached data from PHPList.
    */
    private function PerformPeriodicMaentenance()
    {
        # delete cached information from the database
        $this->DB->Query("DELETE FROM MLPHPList_Users");
        $this->DB->Query("DELETE FROM MLPHPList_ListUserInts");

        # get all the lists that exist
        $Lists = $this->GetAllLists();
        if (count($Lists)>0)
        {
            # call GetSubscribers on the first list to force a cache
            # refresh
            $this->GetSubscribers(reset($Lists));
        }
    }

    /**
    * Run a REST command against a PHPList instance.
    * Logs us in to phplist on the first function call.
    * @param array $Params POST parameters.
    * @return response from PHPList (format depends on the command
    *   issued) or NULL on command failure.
    */
    private function RestCommand($Params)
    {
        static $Context;

        if (!isset($Context))
        {
            $Context = curl_init();

            # enable cookie handling
            curl_setopt($Context, CURLOPT_COOKIEFILE, '');

            # perform login
            $LoginParams = [
                "cmd" => "login",
                "login" => $this->ConfigSetting("PHPListUser"),
                "password" => $this->ConfigSetting("PHPListPass")];
            $Reply = $this->DoCurlRequest($Context, $LoginParams);

            if ($Reply["status"] != "success")
            {
                return NULL;
            }
        }

        # perform the requested REST call
        $Reply = $this->DoCurlRequest($Context, $Params);
        if ($Reply === NULL)
        {
            return NULL;
        }

        return $Reply["data"];
    }

    /**
    * Send an HTTP POST request to a specified URL with Curl.
    * @param resource $Context Created by curl_init().
    * @param array $Params POST parameters to send.
    * @return array Response data (precise contents depend on the
    * request sent).
    */
    private function DoCurlRequest($Context, $Params)
    {
        # use our configured endpoint
        curl_setopt($Context, CURLOPT_URL,
                    $this->ConfigSetting("RESTUrl"));
        # get results back as a string
        curl_setopt($Context, CURLOPT_RETURNTRANSFER, TRUE);
        # send data in a POST
        curl_setopt($Context, CURLOPT_POST, TRUE);
        # load the POST data
        curl_setopt($Context, CURLOPT_POSTFIELDS,
                     http_build_query($Params));

        $CurlResponse = curl_exec($Context);
        if ($CurlResponse === FALSE)
        {
            $errno = curl_errno($Context);
            $GLOBALS["AF"]->LogMessage(
                ApplicationFramework::LOGLVL_ERROR,
                "MLPHPList: Unable to make CURL request. "
                ."CURL errno: ".$errno);
            return NULL;
        }
        else
        {
            # fetch and decode the data
            $Result = json_decode($CurlResponse, TRUE);
            return $Result;
        }
    }

    /**
    * Get array of remote user information.
    * @param string $UserEmail Email to search for.
    * @return array of userinfo from PHPList or NULL if no user found.
    */
    private function GetRemoteUserInfo($UserEmail)
    {
        # query user info with a REST command
        $UserInfo = $this->RestCommand(
            ["cmd" => "userGetByEmail", "email" => $UserEmail]);

        # if no user was found, return NULL
        if (!is_array($UserInfo) || count($UserInfo)==0)
        {
            return NULL;
        }

        return $UserInfo;
    }

    /**
    * Fetch remote UserId.
    * @param string $UserEmail Email to search for.
    * @return mixed Remote UserId or NULL if none was found.
    */
    private function GetRemoteUserId($UserEmail)
    {
        # check our local cache to see if we know the remote user's id
        $UserId = $this->DB->Query(
            "SELECT RemoteUserId FROM MLPHPList_Users "
            ."WHERE Email='".addslashes($UserEmail)."'", "RemoteUserId");

        # if so, return it
        if ($UserId !== NULL)
        {
            return $UserId;
        }

        # otherwise, query it with a rest command
        $UserInfo = $this->GetRemoteUserInfo($UserEmail);

        # if no user was found, return NULL
        if ($UserInfo === NULL)
        {
            return NULL;
        }

        # otherwise, cache this user's id in the database
        $this->DB->Query(
            "INSERT INTO MLPHPList_Users (RemoteUserId, Email) "
            ."VALUES (".intval($UserInfo["id"]).",'".addslashes($UserEmail)."')");

        # and return it
        return $UserInfo["id"];
    }

    /**
    * Update cached subscription info for a given user.
    * @param int $UserId Remote UserId we're updating.
    * @param array $UserLists Array describing the lists the user is
    *     now on (each element must contain an "id" element giving the
    *     remote list id).
    */
    private function UpdateUserSubsCache($UserId, $UserLists)
    {
        $this->DB->Query(
            "LOCK TABLES MLPHPList_ListUserInts WRITE");

        # clear any cached data currently stored
        $this->DB->Query(
            "DELETE FROM MLPHPList_ListUserInts "
            ."WHERE UserId=".intval($UserId));

        # insert a NULL into ListUserInts to note that we have
        # cached values for them
        $this->DB->Query(
            "INSERT INTO MLPHPList_ListUserInts (UserId, ListId) "
            ."VALUES (".intval($UserId).",NULL)");

        # iterate over their lists, caching those as well
        foreach ($UserLists as $UserList)
        {
            $this->DB->Query(
                "INSERT INTO MLPHPList_ListUserInts (UserId, ListId) "
                ."VALUES (".intval($UserId).",".intval($UserList["id"]).")");
        }

        $this->DB->Query("UNLOCK TABLES");
    }

    /**
    * @var array $Lists mailing list cache
    */
    private $Lists;
    private $DB;

    private $SqlTables = [
        "Users" => "CREATE TABLE IF NOT EXISTS MLPHPList_Users (
            RemoteUserId INT NOT NULL,
            Email TEXT,
            INDEX Index_E (Email(15)),
            UNIQUE UIndex_I (RemoteUserId) );",
        "ListUserInts" => "CREATE TABLE IF NOT EXISTS MLPHPList_ListUserInts (
            ListId INT,
            UserId INT NOT NULL,
            UNIQUE UIndex_LU (ListId, UserId) );",
    ];
}
