<?PHP
#
#   FILE:  Message.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2012-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis
#

/**
* Abstraction for forum messages and resource comments.
* \nosubgrouping
*/
class Message extends Item
{
    const PARENTTYPE_TOPIC = 1;
    const PARENTTYPE_RESOURCE = 2;

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /** @name Setup/Initialization/Destruction */
    /*@{*/

    /**
    * Create an empty message object.
    * @return The message just created.
    */
    public static function Create()
    {
        $DB = new Database();

        # create a new empty message record
        $DB->Query("INSERT INTO Messages (MessageId) VALUES (NULL)");

        # get its message id
        $MessageId = $DB->LastInsertId();

        $Message = new Message($MessageId);
        return $Message;
    }

    /**
    * Delete this message from the underlying database
    */
    public function Delete()
    {
        $this->Destroy();
    }

    /*@}*/

    /** @name Accessors */
    /*@{*/

    /**
    * Get this message's messageId.
    * @return int Message ID.
    */
    public function MessageId()
    {
        return $this->Id();
    }

    /**
    * Get the CWIS username of the most recent poster.
    * @return string CWIS username of the most recent poster.
    */
    public function PosterName()
    {
        try
        {
            $PosterName = new CWUser((int)$this->PosterId());
        }
        catch (Exception $e)
        {
            return "";
        }

        return $PosterName->Get("UserName");
    }

    /**
    * Get the email address of the most recent poster
    * @return string Email address of the most recent poster.
    */
    public function PosterEmail()
    {
        try
        {
            $PosterName = new CWUser((int)$this->PosterId());
        }
        catch (Exception $e)
        {
            return "";
        }

        return $PosterName->Get("EMail");
    }

    /**
    * Get the CWIS user ID of the most recent editor.
    * @param int $NewValue New editor ID.  (OPTIONAL)
    * @return int CWIS user ID of the most recent editor.
    */
    public function EditorId($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("EditorId", $NewValue);
    }

    /**
    * Get or set the ParentId.
    * For forum posts, the ParentId is a TopicId.
    * For resource comments, the ParentId is the ResourceId.
    * @param int $NewValue New value to set (OPTIONAL)
    * @return The current ParentId
    */
    public function ParentId($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ParentId", $NewValue);
    }

    /**
    * Get or set the ParentType.
    * Parent Type = 1 for forum posts and
    * Parent Type = 2 for resource comments
    * @param int $NewValue New parent type.  (OPTIONAL)
    * @return int Current parent type.
    */
    public function ParentType($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ParentType", $NewValue);
    }

    /**
    * Get or set the date posted.
    * @param string $NewValue New posting date.  (OPTIONAL)
    * @return string Posting date.
    */
    public function DatePosted($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("DatePosted", $NewValue);
    }

    /**
    * Get or set the date the message was last edited
    * @param string $NewValue New edit date.  (OPTIONAL)
    * @return string Date the message was last edited.
    */
    public function DateEdited($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("DateEdited", $NewValue);
    }

    /**
    * Get or set the poster id (e.g., the author) for this message.
    * @param int $NewValue New poster ID.  (OPTIONAL)
    * @return int ID number of this message's author.
    */
    public function PosterId($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("PosterId", $NewValue);
    }

    /**
    * Get or set the message subject.
    * @param string $NewValue New subject text.  (OPTIONAL)
    * @return string Message subject.
    */
    public function Subject($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("Subject", $NewValue);
    }

    /**
    * Get or set the message body.
    * @param string $NewValue New body text.  (OPTIONAL)
    * @return string Message body.
    */
    public function Body($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("Body", $NewValue);
    }
    /*@}*/
}
