<?PHP
#
#   FILE:  RSSImport.php (RSSImport plugin)
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2011-2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Plugin that provides support for importing and displaying RSS feeds.
*/
class RSSImport extends Plugin
{
    # ---- STANDARD PLUGIN INTERFACE -----------------------------------------

    /**
    * Register information about this plugin.
    */
    public function Register()
    {
        $this->Name = "RSS Import";
        $this->Version = "1.0.0";
        $this->Description = "Plugin that adds RSS import and display support.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
                "CWISCore" => "4.0.0");
        $this->EnabledByDefault = FALSE;
    }

    /**
    * Perform any work needed when the plugin is first installed (for example,
    * creating database tables).
    * @return NULL if installation succeeded, otherwise a string containing
    *       an error message indicating why installation failed.
    */
    public function Install()
    {
        # create database tables
        $Result = $this->CreateTables($this->FeedTable);
        # if creating feed table fails, bail out and return CreateTables error message
        if ($Result !== NULL)
        {
            # if migrating feed table fails, return SqlToMigrateFeedData error message
            $Result = $this->MigrateFeedTable();
        }
        # otherwise, return NULL
        return $Result;
    }

    /**
    * Perform any work needed when the plugin is uninstalled.
    * @return NULL if uninstall succeeded, otherwise a string containing
    *       an error message indicating why uninstall failed.
    */
    public function Uninstall()
    {
        # drop database tables
        return $this->DropTables($this->FeedTable);
    }

    /**
    * Hook event callbacks into the application framework.
    * @return Returns an array of events to be hooked into the application
    *     framework.
    */
    public function HookEvents()
    {
        return array(
                "EVENT_SYSTEM_ADMINISTRATION_MENU" => "AddSystemAdminMenuEntries",
                "EVENT_REQUEST_SIDEBAR_CONTENT" => "DisplayIncomingRssFeeds",
                );
    }

    /**
    * Migrate data from old RSSFeeds table (if it exists) to new RSSImport_Feeds
    * table then drop RSSFeeds.
    * @return NULL if migration succeeded, otherwise a string containing an error
    *     message indicating why migration failed.
    */
    public function MigrateFeedTable()
    {
        $DB = new Database();
        # if old rss feed table exists
        if ($DB->TableExists("RSSFeeds"))
        {
            # populate new rss feed table with it
            $Result = $DB->Query($this->SqlToMigrateFeedTable);
            if ($Result === FALSE)
            {
                return "Unable to populate RSSImport_Feeds database table."
                        ." (ERROR: ".$DB->QueryErrMsg().")";
            }
            # and drop the old rss feed table
            $Result = $DB->Query("DROP TABLE IF EXISTS RSSFeeds");
            if ($Result === FALSE)
            {
                return "Unable to drop RSSFeeds database table."
                        ." (ERROR: ".$DB->QueryErrMsg().")";
            }

        }
        return NULL;
    }

    # ---- HOOKED METHODS ----------------------------------------------------

    /**
    * Add a link to the list of RSS feeds in the administration menu.
    * @return array The array of links.
    */
    public function AddSystemAdminMenuEntries()
    {
        return array(
            "ListFeeds" => "RSS Feeds (Incoming)",
            );
    }

    /**
    * Displays incoming RSS Feeds on selected pages
    */
    public function DisplayIncomingRssFeeds()
    {
        $DB = new Database();
        $DB->Query("SELECT * FROM RSSImport_Feeds");
        $Rows = $DB->FetchRows();

        $HaveLoadedFeedFromSite = FALSE;
        $DefaultCharset = $GLOBALS["G_SysConfig"]->DefaultCharacterSet();

        $FeedLocations = array(
                "Home" => RSSImport_Feed::DISPLOC_FRONTPAGE,
                "SysAdmin" => RSSImport_Feed::DISPLOC_ADMINPAGE,
                "MDHome" => RSSImport_Feed::DISPLOC_MTOOLPAGE,
                );
        $CurPage = $GLOBALS["AF"]->GetPageName();

        foreach ($Rows as $FeedInfo)
        {
            $DisplayLocs = $FeedInfo["DisplayLocations"];

            if ($FeedInfo["Enabled"])
            {
                # if we are on a page that this feed should be displayed on
                if (($DisplayLocs & RSSImport_Feed::DISPLOC_ALLPAGES)
                    || ((isset($FeedLocations[$CurPage])
                        && $DisplayLocs & $FeedLocations[$CurPage])))
                {
                    $RefreshTime = $HaveLoadedFeedFromSite
                            ? 999999 : $FeedInfo["RefreshRate"];
                    $MaxItems = $FeedInfo["NumberItemsToDisplay"];

                    $Feed = new RSSClient($FeedInfo["URL"], $DB, $RefreshTime);

                    $Title = (strlen($FeedInfo["Title"]))
                        ? $FeedInfo["Title"] : strip_tags($Feed->GetChannelTitle());
                    $Href = strip_tags($Feed->GetChannelLink());

                    RSSImportUI::PrintRssFeedItems($Feed, $Title, $Href, $MaxItems);

                    $HaveLoadedFeedFromSite = !$Feed->UsedCachedData();
                }
            }
        }
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $FeedTable = array(
            "RSSImport_Feeds" => "CREATE TABLE IF NOT EXISTS RSSImport_Feeds (
                FeedId                  INT NOT NULL AUTO_INCREMENT,
                Title                   TEXT DEFAULT NULL,
                URL                     TEXT DEFAULT NULL,
                DisplayLocations        INT NOT NULL,
                NumberItemsToDisplay    INT DEFAULT 0,
                RefreshRate             INT NOT NULL,
                Enabled                 INT DEFAULT 0,
                INDEX                   Index_R (FeedId)
            );"
            );

    private $SqlToMigrateFeedData =
            "INSERT INTO RSSImport_Feeds
                (FeedId, Title, URL, DisplayLocations,
                NumberItemsToDisplay,RefreshRate, Enabled)
            SELECT * FROM RSSFeeds;";
}