<?PHP
#
#   FILE:  PrivilegeSet.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Set of privileges used to access resource information or other parts of
* the system.  A privilege set is a combination of privileges (integers),
* MetadataFields (to check against a specified value), and
* privilege/MetadataField combinations.
*/
class PrivilegeSet
{
    /**
    * Class constructor, used to create a new set or reload an existing
    * set from previously-constructed data.
    * @param mixed $Data Existing privilege set data, previously
    *       retrieved with PrivilegeSet::Data(), array of privilege
    *       values, or a single privilege value.  (OPTIONAL)
    * @throws InvalidArgumentException If data passed in was invalid.
    * @see PrivilegeSet::Data()
    */
    public function __construct($Data = NULL)
    {
        # if privilege data supplied
        if ($Data !== NULL)
        {
            # if data is an array of privileges
            if (is_array($Data))
            {
                # set internal privilege set from array
                $this->Privileges = $Data;
            }
            # else if data is a single privilege
            elseif (is_numeric($Data))
            {
                # set internal privilege set from data
                $this->Privileges = array($Data);
            }
            else
            {
                # set internal values from data
                $this->LoadFromData($Data);
            }
        }
    }

    /**
    * Get/set privilege set data, in the form of an opaque string.  This
    * method can be used to retrieve an opaque string containing privilege
    * set data, which can then be saved (e.g. to a database) and later used
    * to reload a privilege set.  (Use instead of serialize() to avoid
    * future issues with internal class changes.)
    * @param string $NewValue New privilege set data.  (OPTIONAL)
    * @return string Current privilege set data (opaque value).
    * @throws InvalidArgumentException If data passed in was invalid.
    */
    public function Data($NewValue = NULL)
    {
        # if new data supplied
        if ($NewValue !== NULL)
        {
            # unpack privilege data and load
            $this->LoadFromData($NewValue);
        }

        # serialize current data and return to caller
        $Data = array();
        if (count($this->Privileges))
        {
            foreach ($this->Privileges as $Priv)
            {
                $Data["Privileges"][] = is_object($Priv)
                        ? array("SUBSET" => $Priv->Data())
                        : $Priv;
            }
        }
        $Data["Logic"] = $this->Logic;
        return serialize($Data);
    }

    /**
    * Determine if a given user meets the requirements specified by
    * this PrivilegeSet.  Typically used to determine if a user should
    * be allowed access to a particular piece of data.
    * @param object $User CWUser object to use in comparisons.
    * @param object $Resource Resource object to used for comparison, for
    *       sets that include user conditions.  (OPTIONAL)
    * @return bool TRUE if privileges in set are greater than or equal to
    *       privileges in specified set, otherwise FALSE.
    */
    public function MeetsRequirements(CWUser $User, $Resource = self::NO_RESOURCE)
    {
        # when there are no requirements, then every user meets them
        $Satisfied = TRUE;

        # for each privilege requirement
        foreach ($this->Privileges as $Priv)
        {
            # if privilege is actually a privilege subgroup
            if (is_object($Priv))
            {
                # check if the subgroup is satisfied
                $Satisfied = $Priv->MeetsRequirements($User, $Resource);
            }
            # else if privilege is actually a condition
            elseif (is_array($Priv))
            {
                # check if condition is satisfied for the given resource
                $Satisfied = $this->MeetsCondition($Priv, $Resource, $User);
            }
            # else privilege is actually a privilege
            else
            {
                # check if user has the spcified privilege
                $Satisfied = $User->HasPriv( $Priv );
            }

            # for AND logic, we can bail as soon as the first
            # condition is not met
            if ($this->Logic == "AND")
            {
                if (!$Satisfied)
                {
                    break;
                }
            }
            # conversely, for OR logic, we can bail as soon as any
            # condition is met
            else
            {
                if ($Satisfied)
                {
                    break;
                }
            }
        }

        # report result of the test back to caller
        return $Satisfied;
    }

    /**
    * Find all users that meet the requirements for this privilege set.  This
    * is (currently) a very brute-force, inefficient implementation, so it should
    * not be used anywhere long execution times might be an issue.
    * @param array $ResourceIds IDs of resources to use for comparisons, if the
    *       set includes resource-dependent conditions.  (OPTIONAL)
    * @return array Array with IDs of users that meet requirements for the index,
    *       and which resource IDs match for that user for the values.
    */
    public function FindUsersThatMeetRequirements($ResourceIds = array())
    {
        # if there are necessary privileges for this privilege set
        $ReqPrivs = $this->GetPossibleNecessaryPrivileges();
        $UFactory = new CWUserFactory();
        if (count($ReqPrivs))
        {
            # start with only those users who have at least one of those privileges
            $UserIds = array_keys($UFactory->GetUsersWithPrivileges($ReqPrivs));
        }
        else
        {
            # start with all users
            $UserIds = $UFactory->GetUserIds();
        }

        # determine of individual resources will need to be checked
        $NeedToCheckResources =
            (count($ResourceIds) && count($this->FieldsWithUserComparisons()))
            ? TRUE : FALSE ;

        # build up a list of matching users
        $UsersThatMeetReqs = array();

        # iterate over all the users
        foreach ($UserIds as $UserId)
        {
            # load user
            $User = new CWUser($UserId);

            if ($NeedToCheckResources)
            {
                # iterate over all the resources
                foreach ($ResourceIds as $ResourceId)
                {
                    # if we're running low on memory, nuke the resource cache
                    if ($GLOBALS["AF"]->GetFreeMemory() /
                        $GLOBALS["AF"]->GetPhpMemoryLimit() < self::$LowMemoryThresh)
                    {
                        self::$ResourceCache = [];
                    }

                    # load resource
                    if (!isset(self::$ResourceCache[$ResourceId]))
                    {
                        self::$ResourceCache[$ResourceId] = new Resource($ResourceId);
                    }

                    # if user meets requirements for set
                    if ($this->MeetsRequirements(
                        $User, self::$ResourceCache[$ResourceId]))
                    {
                        # add resource to user's list
                        $UsersThatMeetReqs[$UserId][] = $ResourceId;
                    }
                }
            }
            else
            {
                # if user meets requirements for set
                if ($this->MeetsRequirements($User))
                {
                    # add user to list
                    $UsersThatMeetReqs[$UserId] = $ResourceIds;
                }
            }
        }

        # return IDs for users that meet requirements to caller
        return $UsersThatMeetReqs;
    }

    /**
    * Add specified privilege to set.  If specified privilege is already
    * part of the set, no action is taken.
    * @param mixed $Privileges Privilege ID or object (or array of IDs or objects).
    * @see PrivilegeSet::RemovePrivilege()
    */
    public function AddPrivilege($Privileges)
    {
        # convert incoming value to array if needed
        if (!is_array($Privileges))
        {
            $Privileges = array($Privileges);
        }

        # for each privilege passed in
        foreach ($Privileges as $Privilege)
        {
            # add privilege if not currently in set
            if (!$this->IncludesPrivilege($Privilege))
            {
                if (is_object($Privilege)) {  $Privilege = $Privilege->Id();  }
                $this->Privileges[] = $Privilege;
            }
        }
    }

    /**
    * Remove specified privilege from set.  If specified privilege is not
    * currently in the set, no action is taken.
    * @param mixed $Privilege Privilege ID or object to remove from set.
    * @see PrivilegeSet::AddPrivilege()
    */
    public function RemovePrivilege($Privilege)
    {
        # remove privilege if currently in set
        if ($this->IncludesPrivilege($Privilege))
        {
            if (is_object($Privilege)) {  $Privilege = $Privilege->Id();  }
            $Index = array_search($Privilege, $this->Privileges);
            unset($this->Privileges[$Index]);
        }
    }

    /**
    * Check whether this privilege set includes the specified privilege.
    * @param mixed $Privilege Privilege ID or object to check.
    * @return bool TRUE if privilege is included, otherwise FALSE.
    */
    public function IncludesPrivilege($Privilege)
    {
        # check whether privilege is in our list and report to caller
        if (is_object($Privilege)) {  $Privilege = $Privilege->Id();  }
        return $this->IsInPrivilegeData($Privilege) ? TRUE : FALSE;
    }

    /**
    * Get privilege information as an array, with numerical indexes
    * except for the logic, which is contained in a element with the
    * index "Logic".  Values are either an associative array with
    * three elements, "FieldId", "Operator", and "Value", or a
    * PrivilegeSet object (for subsets).
    * @return array Array with privilege information.
    */
    public function GetPrivilegeInfo()
    {
        # grab privilege information and add logic
        $Info = $this->Privileges;
        $Info["Logic"] = $this->Logic;

        # return privilege info array to caller
        return $Info;
    }

    /**
    * Get list of privileges.  (Intended primarily for supporting legacy
    * privilege operations -- list contains privilege IDs only, and does
    * not include conditions.)
    * @return array Array of privilege IDs.
    */
    public function GetPrivilegeList()
    {
        # create list of privileges with conditions stripped out
        $List = array();
        foreach ($this->Privileges as $Priv)
        {
            if (!is_array($Priv)) {  $List[] = $Priv;  }
        }

        # return list of privileges to caller
        return $List;
    }

    /**
    * Add condition to privilege set.  If the condition is already present
    * in the set, no action is taken.
    * @param mixed $Field Metadata field object or ID to test against.
    * @param mixed $Value Value to test against.  User fields expect a
    *       UserId or expect NULL to test against the current user.
    *       Option fields expect a ControlledNameId.  Date and
    *       Timestamp fields expect either a UNIX timestamp or expect
    *       NULL to test against the current time.
    * @param string $Operator String containing operator to used for
    *       condition.  (Standard PHP operators are used.)  (OPTIONAL,
    *       defaults to "==")
    * @return bool TRUE if condition was added, otherwise FALSE.
    * @throws InvalidArgumentException If negative field ID or field with
    *       negative ID supplied.
    */
    public function AddCondition($Field, $Value = NULL, $Operator = "==")
    {
        # get field ID
        $FieldId = is_object($Field) ? $Field->Id() : $Field;

        # make sure we were not passed an invalid field
        if ($FieldId <= -1)
        {
            throw new InvalidArgumentException("Field with negative ID supplied.");
        }

        # set up condition array
        $Condition = array(
                "FieldId" => intval($FieldId),
                "Operator" => trim($Operator),
                "Value" => $Value);

        # if condition is not already in set
        if (!$this->IsInPrivilegeData($Condition))
        {
            # add condition to privilege set
            $this->Privileges[] = $Condition;
            return TRUE;
        }
        return FALSE;
    }

    /**
    * Remove condition from privilege set.  If condition was not present
    * in privilege set, no action is taken.
    * @param mixed $Field Metadata field object or ID to test against.
    * @param mixed $Value Value to test against.  (Specify NULL for User
    *       fields to test against current user.)
    * @param string $Operator String containing operator to used for
    *       condition.  (Standard PHP operators are used.)  (OPTIONAL,
    *       defaults to "==")
    * @param bool $IncludeSubsets TRUE to remove the condition from any
    *       subsets in which it appears as well (OPTIONAL, default FALSE).
    * @return bool TRUE if condition was removed, otherwise FALSE.
    */
    public function RemoveCondition(
        $Field, $Value = NULL, $Operator = "==",
        $IncludeSubsets = FALSE)
    {
        $Result = FALSE;

        # get field ID
        $FieldId = is_object($Field) ? $Field->Id() : $Field;

        # set up condition array
        $Condition = array(
                "FieldId" => intval($FieldId),
                "Operator" => trim($Operator),
                "Value" => $Value);

        # if condition is in set
        if ($this->IsInPrivilegeData($Condition))
        {
            # remove condition from privilege set
            $Index = array_search($Condition, $this->Privileges);
            unset($this->Privileges[$Index]);
            $Result = TRUE;
        }

        if ($IncludeSubsets)
        {
            foreach ($this->Privileges as $Priv)
            {
                if ($Priv instanceof PrivilegeSet)
                {
                    $Result |= $Priv->RemoveCondition(
                        $FieldId, $Value, $Operator, TRUE);
                }
            }
        }

        return $Result;
    }

    /**
    * Add subgroup of privileges/conditions to set.
    * @param PrivilegeSet $Set Subgroup to add.
    */
    public function AddSet(PrivilegeSet $Set)
    {
        # if subgroup is not already in set
        if (!$this->IsInPrivilegeData($Set))
        {
            # add subgroup to privilege set
            $this->Privileges[] = $Set;
        }
    }

    /**
    * Get/set whether all privileges/conditions in set are required (i.e.
    * "AND" logic), or only one privilege/condition needs to be met ("OR").
    * By default only one of the specified privilegs/conditions in a set
    * is required.
    * @param bool $NewValue Specify TRUE if all privileges are required,
    *       otherwise FALSE if only one privilege required.  (OPTIONAL)
    * @return bool TRUE if all privileges required, otherwise FALSE.
    */
    public function AllRequired($NewValue = NULL)
    {
        if ($NewValue !== NULL)
        {
            $this->Logic = $NewValue ? "AND" : "OR";
        }
        return ($this->Logic == "AND") ? TRUE : FALSE;
    }

    /**
    * List which privilege flags (e.g. PRIV_MYRESOURCEADMIN) are examined by
    *  this privset.
    * @return Array of privilege flags checked.
    */
    public function PrivilegeFlagsChecked()
    {
        $Info = $this->GetPrivilegeInfo();
        unset($Info["Logic"]);

        $Result = array();
        foreach ($Info as $Item)
        {
            if (is_object($Item))
            {
                $Result = array_merge($Result, $Item->PrivilegeFlagsChecked() );
            }
            elseif (!is_array($Item))
            {
                $Result[]= $Item;
            }
        }
        return array_unique($Result);
    }

    /**
    * List which fields in this privset are involved in UserIs or UserIsNot
    *  comparisons for this privilege set.
    * @param string $ComparisonType Comparison Type ("==" or "!="). (OPTIONAL,
    *       defaults to both comparisons)
    * @return Array of FieldIds that have a User comparison.
    */
    public function FieldsWithUserComparisons($ComparisonType = NULL)
    {
        $Info = $this->GetPrivilegeInfo();
        unset($Info["Logic"]);

        $Result = array();
        foreach ($Info as $Item)
        {
            if (is_object($Item))
            {
                $Result = array_merge(
                    $Result,
                    $Item->FieldsWithUserComparisons($ComparisonType));
            }
            elseif (is_array($Item))
            {
                if ( (($Item["Operator"] == $ComparisonType)
                      || ($ComparisonType === NULL)) &&
                     MetadataSchema::FieldExistsInAnySchema($Item["FieldId"]))
                {
                    $Field = new MetadataField($Item["FieldId"]);

                    if ($Field->Type() == MetadataSchema::MDFTYPE_USER)
                    {
                        $Result[]= $Item["FieldId"];
                    }
                }
            }
        }

        return array_unique($Result);
    }

    /**
    * Get number of privilege comparisons in set, including those in subgroups.
    * @return int Comparison count.
    */
    public function ComparisonCount()
    {
        $Count = 0;
        foreach ($this->Privileges as $Priv)
        {
            $Count += is_object($Priv) ? $Priv->ComparisonCount() : 1;
        }
        return $Count;
    }

    /**
    * Get all privileges that could be necessary to fulfill privilege set
    * requirements.  Not all of the privileges may be necessary, but a user
    * must have at least one of the listed privileges to qualify.  If there
    * is no set of privileges where at least one is definitely required, an
    * empty array is returned.
    * @return array Privilege IDs.
    */
    public function GetPossibleNecessaryPrivileges()
    {
        # for each privilege requirement
        $NecessaryPrivs = array();
        foreach ($this->Privileges as $Priv)
        {
            # if requirement is comparison
            if (is_array($Priv))
            {
                # if logic is OR
                if ($this->Logic == "OR")
                {
                    # bail out because no privileges are required
                    return array();
                }
            }
            # else if requirement is subgroup
            elseif (is_object($Priv))
            {
                # retrieve possible needed privileges from subgroup
                $SubPrivs = $Priv->GetPossibleNecessaryPrivileges();

                # if no privileges were required by subgroup
                if (!count($SubPrivs))
                {
                    # if logic is OR
                    if ($this->Logic == "OR")
                    {
                        # bail out because no privileges are required
                        return array();
                    }
                }
                else
                {
                    # add subgroup privileges to required list
                    $NecessaryPrivs = array_merge($NecessaryPrivs, $SubPrivs);
                }
            }
            # else requirement is privilege
            else
            {
                # add privilege to required list
                $NecessaryPrivs[] = $Priv;
            }
        }

        # return possible needed privileges to caller
        return $NecessaryPrivs;
    }

    /**
    * Determine if a PrivilegeSet checks values from a specified field.
    * @param int $FieldId FieldId to check.
    * @return bool TRUE if the given field is checked.
    */
    public function ChecksField($FieldId)
    {
        # iterate over all the privs in this privset
        foreach ($this->Privileges as $Priv)
        {
            # if this priv is a field condition that references the
            # provided FieldId, return true
            if (is_array($Priv) && $Priv["FieldId"] == $FieldId)
            {
                return TRUE;
            }
            # otherwise, if this was a privset then call ourself recursively
            elseif ($Priv instanceof PrivilegeSet &&
                    $Priv->ChecksField($FieldId))
            {
                return TRUE;
            }
        }

        # found no references to this field, return FALSE
        return FALSE;
    }

    /**
    * Clear internal caches.  This is primarily intended for situations where
    * memory may have run low.
    */
    public static function ClearCaches()
    {
        self::$MetadataFieldCache = array();
        self::$ResourceCache = array();
        self::$ValueCache = array();
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $RFactories = array();
    private $Privileges = array();
    private $Logic = "OR";

    private static $MetadataFieldCache;
    private static $ResourceCache;
    private static $ValueCache;

    private static $LowMemoryThresh = 0.25;

    const NO_RESOURCE = "XXX NO RESOURCE XXX";

    /**
    * Load privileges from serialized data.
    * @param string $Serialized Privilege data.
    * @throws InvalidArgumentException If data passed in was invalid.
    */
    private function LoadFromData($Serialized)
    {
        # save calling context in case load causes out-of-memory crash
        $GLOBALS["AF"]->RecordContextInCaseOfCrash();

        # unpack new data
        $Data = unserialize($Serialized);
        if ($Data === FALSE)
        {
            throw new InvalidArgumentException(
                    "Invalid serialized data supplied (\"".$Serialized."\").");
        }

        # unpack privilege data (if available) and load
        if (array_key_exists("Privileges", $Data))
        {
            $this->Privileges = array();
            foreach ($Data["Privileges"] as $Priv)
            {
                if (is_array($Priv) && array_key_exists("SUBSET", $Priv))
                {
                    $Subset = new PrivilegeSet();
                    $Subset->LoadFromData($Priv["SUBSET"]);
                    $this->Privileges[] = $Subset;
                }
                else
                {
                    $this->Privileges[] = $Priv;
                }
            }
        }

        # load logic if available
        if (array_key_exists("Logic", $Data))
        {
            $this->Logic = $Data["Logic"];
        }
    }

    /**
    * Check whether this privilege set meets the specified condition.
    * @param array $Condition Condition to check.
    * @param object $Resource Resource to use when checking.
    * @param CWUser $User User to use when checking.
    * @return bool TRUE if condition is met, otherwise FALSE.
    */
    private function MeetsCondition($Condition, $Resource, $User)
    {
        # make sure metadata field is loaded
        $MFieldId = $Condition["FieldId"];
        if (!isset(self::$MetadataFieldCache[$MFieldId]))
        {
            self::$MetadataFieldCache[$MFieldId] =
                    !MetadataSchema::FieldExistsInAnySchema($Condition["FieldId"])
                    ? FALSE
                    : new MetadataField($MFieldId);
        }

        # if the specified field does not exist
        if (self::$MetadataFieldCache[$MFieldId] === FALSE)
        {
            # return a result that in effect ignores the condition
            return ($this->Logic == "AND") ? TRUE : FALSE;
        }

        # pull out provided field
        $Field = self::$MetadataFieldCache[$MFieldId];
        $Operator = $Condition["Operator"];
        $Value = $Condition["Value"];

        # determine if the provided operator is valid for the provided field
        if (!in_array($Operator, $this->ValidOperatorsForFieldType($Field->Type()) ))
        {
            throw new Exception("Operator ".$Operator." not supported for "
                    .$Field->TypeAsName()." fields");
        }

        # if we don't have a specific resource to check, then we want
        # to determine if this condition would be satisfied by any
        # resource
        if ($Resource == self::NO_RESOURCE)
        {
            $Count = $this->CountResourcesThatSatisfyCondition(
                $User, $Field, $Operator, $Value);
            return $Count > 0 ? TRUE : FALSE;
        }
        # else if resource is valid
        elseif ($Resource instanceof Resource)
        {
            # if this field is from a different schema than our resource
            # and also this field is not from the User schema, then there's
            # no comparison for us to do
            if ($Field->SchemaId() != $Resource->SchemaId() &&
                $Field->SchemaId() != MetadataSchema::SCHEMAID_USER)
            {
                # return a result that in effect ignores the condition
                return ($this->Logic == "AND") ? TRUE : FALSE;
            }

            # normalize the incoming value for comparison
            $Value = $this->NormalizeTargetValue($Field->Type(), $User, $Value);
            $FieldValue = $this->GetNormalizedFieldValue($Field, $Resource, $User);

            # perform comparison, returning result
            return $this->CompareNormalizedFieldValues($FieldValue, $Operator, $Value);
        }
        else
        {
            # error out because resource was illegal
            throw new Exception("Invalid Resource passed in for privilege"
                    ." set comparison.");
        }
    }

    /**
    * Determine the valid condition operators for a given field type.
    * @param int $FieldType Field type (one of the
    *     MetadataSchema::MDFTYPE_ constants).
    * @return array of valid operators.
    */
    private function ValidOperatorsForFieldType($FieldType)
    {
        switch ($FieldType)
        {
            case MetadataSchema::MDFTYPE_USER:
                $ValidOps = ["=="];
                break;

            case MetadataSchema::MDFTYPE_DATE:
            case MetadataSchema::MDFTYPE_TIMESTAMP:
            case MetadataSchema::MDFTYPE_NUMBER:
                $ValidOps = ["==", "!=", "<=", "<", ">=", ">"];
                break;

            case MetadataSchema::MDFTYPE_FLAG:
            case MetadataSchema::MDFTYPE_OPTION:
                $ValidOps = ["==", "!="];
                break;

            default:
                $ValidOps = [];
                break;
        }

        return $ValidOps;
    }

    /**
    * Normalize a target value from a privilege set condition for comparison.
    * @param int $FieldType Metadata field type being compared (as a
    *     MetadataSchema::MDFTYPE_ constant).
    * @param CWUser $User User for whom the comparison is being performed.
    * @param mixed $Value Target value for the comparison.
    * @return mixed Normalized value
    */
    private function NormalizeTargetValue($FieldType, $User, $Value)
    {
        switch ($FieldType)
        {
            case MetadataSchema::MDFTYPE_DATE:
            case MetadataSchema::MDFTYPE_TIMESTAMP:
                # "Now" is encoded as NULL for timestamp and date comparisons
                if ($Value === NULL)
                {
                    $Value = time();
                }
                # otherwise, parse the value to get a numeric timestamp
                else
                {
                    $Value = strtotime($Value);
                }
                break;

            case MetadataSchema::MDFTYPE_USER:
                # "Current user" is encoded as NULL for user comparisons
                if ($Value === NULL)
                {
                    $Value = $User->Id();
                }
                break;

            default:
                # no normalization needed for other field types
                break;
        }

        return $Value;
    }

    /**
    * Get a normalized field value from a resource for comparisons.
    * If the provided field is from the User schema, but the resource
    * is not, then the value will be taken from the User doing the
    * comparison rather than the resource (so that conditions like
    * User: ZIp Code = XXX work as expected).
    * @param MetadataField $Field Field to pull values from.
    * @param Resource $Resource Resource to pull values from.
    * @param CWUser $User User performing the comparison.
    * @return mixed Normalized value.  For User and Option fields,
    * this will be an array of Ids.  For Date and Timestamp fields, it
    * will be a UNIX timestamp.  For Number and Flag fields, it will
    * be the literal value stored in the database.
    */
    private function GetNormalizedFieldValue($Field, $Resource, $User)
    {
        # if we have a cached normalized value for this field,
        # use that for comparisons
        $CacheKey = $Resource->Id()."_".$Field->Id();
        if (!isset(self::$ValueCache[$CacheKey]))
        {
            # if the given field comes from the User schema and our
            # resource does not, evaluate this comparison against the
            # provided $User rather than the provided $Resource
            # (this allows conditions like User: Zip Code = XXX to
            # work as expected rather than being skipped)
            if ($Field->SchemaId() != $Resource->SchemaId() &&
                $Field->SchemaId() == MetadataSchema::SCHEMAID_USER)
            {
                $FieldValue = $User->Get($Field);
            }
            else
            {
                # Note: Resource::Get() on a ControlledName with
                # IncludeVariants=TRUE does not return CNIds for
                # array indexes, which will break the normalization
                # below, so do not change this to add $IncludeVariants
                # without revising the normalization code below
                $FieldValue = $Resource->Get($Field);
            }

            # normalize field value for comparison
            switch ($Field->Type())
            {
                case MetadataSchema::MDFTYPE_USER:
                case MetadataSchema::MDFTYPE_OPTION:
                    # get the UserIds or CNIds from this field
                    $FieldValue = array_keys($FieldValue);
                    break;

                case MetadataSchema::MDFTYPE_DATE:
                case MetadataSchema::MDFTYPE_TIMESTAMP:
                    # convert returned value to a numeric timestamp
                    $FieldValue = strtotime($FieldValue);
                    break;

                case MetadataSchema::MDFTYPE_NUMBER:
                case MetadataSchema::MDFTYPE_FLAG:
                    # no conversion needed
                    break;

                default:
                    throw new Exception("Unsupported metadata field type ("
                            .print_r($Field->Type(), TRUE)
                            .") for condition in privilege set with resource.");
                    break;
            }

            # cache the normalized value for subsequent reuse
            self::$ValueCache[$CacheKey] = $FieldValue;
        }

        return self::$ValueCache[$CacheKey];
    }

    /**
    * Compare normalized field and target values with a specified operator.
    * @param mixed $FieldValue Value from the subject resource.
    * @param string $Operator Operator for the comparison.
    * @param mixed $Value Target value of the comparison.
    * @return TRUE if $FieldValue $Operator $Value is satisfied, FALSE
    *     otherwise.
    */
    private function CompareNormalizedFieldValues($FieldValue, $Operator, $Value)
    {
        # compare field value and supplied value using specified operator

        # if this is a multi-value field, be sure that the provided
        # operator makes sense
        if (is_array($FieldValue) && !in_array($Operator, ["==", "!="]) )
        {
            throw new Exception(
                "Multiple-value fields ony support == and != operators");
        }

        switch ($Operator)
        {
            case "==":
                if (is_array($FieldValue))
                {
                    # equality against multi-value fields is
                    # interpreted as 'contains', true if the
                    # target value is one of those set
                    $Result = in_array($Value, $FieldValue);
                }
                else
                {
                    $Result = ($FieldValue == $Value);
                }
                break;

            case "!=":
                if (is_array($FieldValue))
                {
                    # not equal against multi-value fields is
                    # interpreted as 'does not contain', true as long as
                    # the target value is not one of those set
                    $Result = !in_array($Value, $FieldValue);
                }
                else
                {
                    $Result = ($FieldValue != $Value);
                }
                break;

            case "<":
                $Result = ($FieldValue < $Value);
                break;

            case ">":
                $Result = ($FieldValue > $Value);
                break;

            case "<=":
                $Result = ($FieldValue <= $Value);
                break;

            case ">=":
                $Result = ($FieldValue >= $Value);
                break;

            default:
                throw new Exception("Unsupported condition operator ("
                                    .print_r($Operator, TRUE).") in privilege set.");
                break;
        }

        # report to caller whether condition was met
        return $Result ? TRUE : FALSE;
    }

    /**
    * Determine the number of resources in the collection that satisfy
    * a condition.
    * @param CWUser $User User performing the comparisons.
    * @param MetadataField $Field Field being used in the comaprisons.
    * @param string $Operator Operator for comparisons.
    * @param mixed $Value Target value for comparisons.
    * @return number of resources where
    *    Resource->Get($Field) $Operator $Value is TRUE.
    */
    private function CountResourcesThatSatisfyCondition(
        $User, $Field, $Operator, $Value)
    {
        # get the SchemaId for this field
        $ScId = $Field->SchemaId();

        # pull out an RFactory for the field's schema
        if (!isset($this->RFactories[$ScId]))
        {
            $this->RFactories[$ScId] = new ResourceFactory($ScId);
        }

        switch ($Field->Type())
        {
            case MetadataSchema::MDFTYPE_USER:
            case MetadataSchema::MDFTYPE_DATE:
            case MetadataSchema::MDFTYPE_TIMESTAMP:
            case MetadataSchema::MDFTYPE_NUMBER:
            case MetadataSchema::MDFTYPE_FLAG:
                $ValuesToMatch = array(
                    $Field->Id() => $Value,
                );

                $Matches = $this->RFactories[$ScId]->GetMatchingResources(
                    $ValuesToMatch, TRUE, FALSE, $Operator);

                $Count = count($Matches);
                break;

            case MetadataSchema::MDFTYPE_OPTION:
                # find the number of resources associated with this option
                $Count = $this->RFactories[$ScId]->AssociatedVisibleResourceCount(
                    $Value, $User, TRUE);

                # if our Op was !=, then subtract the resources
                # that have the spec'd option out of the total to
                # figure out how many lack the option
                if ($Operator == "!=")
                {
                    $Count = $this->RFactories[$ScId]->GetVisibleResourcesCount(
                        $User) - $Count;
                }

                break;

            default:
                throw new Exception("Unsupported metadata field type ("
                        .print_r($Field->Type(), TRUE)
                        .") for condition in privilege set without resource.");
                break;
        }

        return $Count;
    }

    /**
    * Check whether specified item (privilege, condition, or subgroup) is
    * currently in the list of privileges.  This is necessary instead of just
    * using in_array() because in_array() generates NOTICE messages if the
    * array contains mixed types.
    * @param mixed $Item Item to look for.
    * @return bool TRUE if item found in privilege list, otherwise FALSE.
    */
    private function IsInPrivilegeData($Item)
    {
        # step through privilege data
        foreach ($this->Privileges as $Priv)
        {
            # report to caller if item is found
            if (is_object($Item))
            {
                if (is_object($Priv) && ($Item == $Priv)) {  return TRUE;  }
            }
            elseif (is_array($Item))
            {
                if (is_array($Priv) && ($Item == $Priv)) {  return TRUE;  }
            }
            elseif ($Item == $Priv) {  return TRUE;  }
        }

        # report to caller that item is not in privilege data
        return FALSE;
    }
}
