<?PHP
#
#   FILE:  RSSExport_Feed.php (RSSExport plugin)
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

class RSSExport_Feed
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Constructs a new Feed object. If given an ID, the feed object
    * will be taken from the database. Otherwise, a new feed object will be
    * created with no settings set.
    * @param int $Id The ID of the Feed to construct.  (OPTIONAL)
    * @throws Exception if the feed Id does not exist in the database.
    */
    public function __construct($Id = NULL)
    {
        # create the database
        $this->DB = new Database();

        # if we are retrieving a feed from the database
        if ($Id != NULL)
        {
            # fill in Id
            $this->Id = $Id;

            # look for the specified Id in database
            $this->DB->Query("SELECT * FROM RSSExport_Feeds WHERE FeedId = " . $Id);
            $Row = $this->DB->FetchRow();

            # throw exception if this feed does not actually exist
            if (!$Row)
            {
                throw new Exception("Unable to construct feed: "
                        . "Id does not exist in the database");
            }
            # otherwise cache the feed settings
            else
            {
                $this->DBCache = $Row;
            }
        }
        # otherwise make a new feed
        else
        {
            # get the highest feed ID
            $this->DB->Query("SELECT `FeedId` FROM `RSSExport_Feeds` "
                    . "ORDER BY `FeedId` DESC LIMIT 1");
            $Row = $this->DB->FetchRow();
            $HighestId = $Row["FeedId"];

            # increment the feed ID
            $HighestId++;

            # add a new blank row for this feed
            $this->DB->Query("INSERT INTO `RSSExport_Feeds` "
                    . "(`FeedId`) VALUES (" . $HighestId . ")");

            # save the ID and initialize the chache
            $this->Id = $HighestId;
            $this->DB->Query("SELECT * FROM `RSSExport_Feeds` "
                . "WHERE `FeedId` = " . $this->Id);
            $this->DBCache = $this->DB->FetchRow();
        }
    }

    /**
    * Removes this RSS feed from the database.
    * @throws Exception if the delete failed.
    */
    public function Delete()
    {
        $Deleted = $this->DB->Query("DELETE FROM RSSExport_Feeds WHERE FeedId="
                . $this->Id());
        if (!$Deleted)
        {
            throw new Exception("The RSSExport_Feed could not be deleted.");
        }
    }

    /**
    * Gets the objects unique ID.
    * @return int The ID of the Feed
    */
    public function Id() { return $this->Id; }


    /**
    * Gets or sets the clean URL for the feed.
    * @param string $NewValue The new clean URL for the feed.  (OPTIONAL)
    * @return string The clean URL for the feed.
    */
    public function CleanUrl($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("CleanUrl", $NewValue); }

    /**
    * Gets or sets the number of items displayed for the feed.
    * @param int $NewValue The new number of items for the feed.  (OPTIONAL)
    * @return int The number of items for the feed.
    */
    public function NumItems($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("NumItems", $NewValue); }

    /**
    * Gets or sets whether to show resource images for the feed.
    * @param int $NewValue The new show resource images setting
    *        for the feed.  (OPTIONAL)
    * @return int The show resource images setting for the feed.
    */
    public function ShowResourceImages($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("ShowResourceImages", $NewValue); }

    /**
    * Gets or sets whether to notify browsers of the feed.
    * @param int $NewValue The new notify browser setting
    *        for the feed.  (OPTIONAL)
    * @return int The notify browser setting for the feed.
    */
    public function NotifyBrowser($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("NotifyBrowser", $NewValue); }


    /**
    * Gets or sets the title of the feed.
    * @param string $NewValue The new title of the feed.  (OPTIONAL)
    * @return string The title of the feed.
    */
    public function Title($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("Title", $NewValue); }

    /**
    * Gets the link for the feed. The link is always the CWIS base URL.
    * @return string The link for the feed.
    */
    public function Link() { return OurBaseUrl(); }

    /**
    * Gets or sets the description of the feed.
    * @param string $NewValue The new description of the feed.  (OPTIONAL)
    * @return string The desscription of the feed.
    */
    public function Description($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("Description", $NewValue); }


    /**
    * Gets the language for the feed. The language is always English.
    * @return string The language for the feed.
    */
    public function Language() { return "en-US"; }

    /**
    * Gets or sets the copyright of the feed.
    * @param string $NewValue The new copyright of the feed.  (OPTIONAL)
    * @return string The copyright of the feed.
    */
    public function Copyright($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("Copyright", $NewValue); }

    /**
    * Gets or sets the managing editor of the feed.
    * @param string $NewValue The new managing editor of the feed.  (OPTIONAL)
    * @return string The managing editor of the feed.
    */
    public function ManagingEditor($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("ManagingEditor", $NewValue); }

    /**
    * Gets or sets the web master of the feed.
    * @param string $NewValue The new web master of the feed.  (OPTIONAL)
    * @return string The web master of the feed.
    */
    public function WebMaster($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("WebMaster", $NewValue); }

    /**
    * Gets publication date of the feed. It is the same as the build date.
    * @return string The publication date of the feed.
    */
    public function PubDate() { return $this->LastBuildDate(); }

    /**
    * Gets or sets the last build date of the feed.
    * @param string $NewValue The new last build date of the feed.  (OPTIONAL)
    * @return string The last build date of the feed.
    */
    public function LastBuildDate($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("LastBuildDate", $NewValue); }

    /**
    * Gets or sets the category of the feed.
    * @param string $NewValue The new category of the feed.  (OPTIONAL)
    * @return string The category of the feed.
    */
    public function Category($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("Category", $NewValue); }

    /**
    * Gets or set the search parameters for the feed. The search parameters
    * should be copied and pasted from the URL of a CWIS search.
    * @param string $NewValue The new search parameters from a CWIS URL.  (OPTIONAL)
    * @return string The search parameters for the RSSExport_Feed.
    */
    public function SearchParameters($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("SearchParameters", $NewValue); }

    /**
    * Gets the generator of the feed. The generator is CWIS.
    * @return string The generator of the feed.
    */
    public function Generator()
            { return "CWIS " . $GLOBALS["SPT_SoftwareVersionToDisplay"]; }

    /**
    * Gets the rss documentation.
    * @return string The url of the RSS documenation.
    */
    public function Docs()
            { return "http://www.rssboard.org/rss-specification"; }

    /**
    * Gets or sets the image URL of the feed.
    * @param string $NewValue The new image URL of the feed.  (OPTIONAL)
    * @return string The image URL of the feed.
    */
    public function ImageUrl($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("ImageUrl", $NewValue); }

    /**
    * Gets or sets the image width for the feed.
    * @param int $NewValue The new image width for the feed.  (OPTIONAL)
    * @return int The image width for the feed.
    */
    public function ImageWidth($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("ImageWidth", $NewValue); }

    /**
    * Gets or sets the image height for the feed.
    * @param int $NewValue The new image height for the feed.  (OPTIONAL)
    * @return int The image height for the feed.
    */
    public function ImageHeight($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("ImageHeight", $NewValue); }

    /**
    * Gets or sets the image description for the feed.
    * @param string $NewValue The new image description for the feed. (OPTIONAL)
    * @return string The feed image description.
    */
    public function ImageDescription($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("ImageDescription", $NewValue); }


    /**
    * Gets or sets the metadata schema for the feed.
    * @param int $NewValue The ID of the new metadata schema for the feed.
    *         (OPTIONAL)
    * @return int The ID of the metadata schema for the feed.
    */
    public function SchemaId($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("SchemaId", $NewValue); }


    /**
    * Gets or sets the title metadata field for the feed.
    * @param int $NewValue The ID of the new title metadata field for the feed.
    *         (OPTIONAL)
    * @return int The ID of the title metadata field for the feed.
    */
    public function TitleFieldId($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("TitleFieldId", $NewValue); }

    /**
    * Gets or sets the link metadata field for the feed.
    * @param int $NewValue The ID of the new link metadata field for the feed.
    *         (OPTIONAL)
    * @return int The ID of the link metadata field for the feed.
    */
    public function LinkFieldId($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("LinkFieldId", $NewValue); }

    /**
    * Gets or sets the description metadata field for the feed.
    * @param int $NewValue The ID of the new description metadata field for
    *       the feed.  (OPTIONAL)
    * @return int The ID of the description metadata field for the feed.
    */
    public function DescriptionFieldId($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("DescriptionFieldId", $NewValue); }


    /**
    * Gets or sets the author metadata field for the feed.
    * @param int $NewValue The ID of the new author metadata field for the feed.
    *         (OPTIONAL)
    * @return int The ID of the author metadata field for the feed.
    */
    public function AuthorFieldId($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("AuthorFieldId", $NewValue); }

    /**
    * Gets or sets the category metadata field for the feed.
    * @param int $NewValue The ID of the new category metadata field for the
    *         feed.  (OPTIONAL)
    * @return int The ID of the category metadata field for the feed.
    */
    public function CategoryFieldId($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("CategoryFieldId", $NewValue); }

    /**
    * Gets or sets whether to link to the comments of items in the feed.
    * @param int $NewValue The new comments setting for the feed.
    * @return int The comments setting for the feed.
    */
    public function Comments($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("Comments", $NewValue); }

    /**
    * Gets or sets the enclosure metadata field for the feed.
    * @param int $NewValue The ID of the new enclosure metadata field for the
    *         feed.  (OPTIONAL)
    * @return int The ID of the enclosure metadata field for the feed.
    */
    public function EnclosureFieldId($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("EnclosureFieldId", $NewValue); }

    /**
    * Gets or sets the publication date metadata field for the feed.
    * @param int $NewValue The ID of the new publication date metadata field
    *         for the feed.  (OPTIONAL)
    * @return int The ID of the publication date metadata field for the feed.
    */
    public function PubDateFieldId($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("PubDateFieldId", $NewValue); }

    /**
    * Gets or sets the image metadata field for the feed.
    * @param int $NewValue The ID of the new image metadata field
    *         for the feed.  (OPTIONAL)
    * @return int The ID of the image metadata field for the feed.
    */
    public function ImageFieldId($NewValue = DB_NOVALUE)
            { return $this->UpdateValue("ImageFieldId", $NewValue); }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    /**
    * The unique ID of the Feed.
    */
    private $Id;

    /**
    * The cached row of the database for the feed.
    */
    private $DBCache;

    /**
    * The Database object this feed uses.
    */
    private $DB;

    /**
    * Gets or changes a value in the database.
    * @param string $FieldName Name of database field. (OPTIONAL)
    * @param mixed $NewValue The new value to set.  Use DB_NOVALUE to not set
    *       a new value (i.e. when just getting a value). (OPTIONAL)
    * @param string $Table Name of database table.  (OPTIONAL)
    * @param string $Condition SQL query conditional to use in SELECT or UPDATE
    *       statements (should not include "WHERE").  Use NULL if no conditional
    *       is needed. (OPTIONAL)
    * @return Requested value.
    */
    private function UpdateValue($FieldName = NULL, $NewValue = DB_NOVALUE,
            $Table = "RSSExport_Feeds", $Condition = NULL)
    {
        # create default condition
        if ($Condition == NULL) $Condition = "`FeedId` = " . $this->Id();

        # update and read the value
        $Value = $this->DB->UpdateValue($Table, $FieldName, $NewValue,
                $Condition, $this->DBCache);

        return $Value;
    }

}
