<?PHP
#
#   FILE:  SavedSearchMailings.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Adds periodic emails about new results for saved searches.
*/
class SavedSearchMailings extends Plugin
{
    # ---- STANDARD PLUGIN INTERFACE -----------------------------------------

    /**
    * Set up plugin upon installation.
    * Set defaults for configuration.
    * @return NULL on success
    */
    public function Install()
    {
        $Mailer = $GLOBALS["G_PluginManager"]->GetPlugin("Mailer");
        $MailerTemplates = $Mailer->GetTemplateList();

        # set up the default template if it does not exist
        if (!in_array("Saved Search Default", $MailerTemplates))
        {
            $HtmlBody =
                "<h1>X-PORTALNAME-X</h1>"
                ."<p>Saved Search Results (X-RESOURCECOUNT-X Results found)<br>"
                ."Generated for X-USERREALNAME-X on X-DATE-X at X-TIME-X</p>"
                ."<p>Search Criteria:<br>X-SEARCHCRITERIA-X</p>"
                ."<ul>X-RESOURCELIST-X</ul>"
                ."<p>X-LEGALNOTICE-X</p>";

            $HtmlItem =
                "<li><b>Title</b>: X-STDFIELD:TITLE-X<br>"
                ."<b>URL</b>: X-STDFIELD:URL-X<br>"
                ."<b>Description</b>:<br>X-STDFIELD:DESCRIPTION-X</li>";

            $TextBody =
                "X-PORTALNAME-X\n"
                ."==========\n\n"
                ."Saved Search Results (X-RESOURCECOUNT-X Results found)\n"
                ."Generated for X-USERREALNAME-X on X-DATE-X at X-TIME-X\n\n"
                ."Search Criteria:\nX-SEARCHCRITERIA-X\n\n"
                ."----------\n"
                ."X-RESOURCELIST-X"
                ."----------\n\n"
                ."X-LEGALNOTICE-X";

            $TextItem =
                "      Title:  X-STDFIELD:TITLE-X\n"
                ."        URL:  X-STDFIELD:URL-X\n"
                ."Description:\nX-STDFIELD:DESCRIPTION-X\n\n";

            $Mailer->AddTemplate(
                "Saved Search Default", # name
                "X-PORTALNAME-X <X-ADMINEMAIL-X>", # from
                "X-PORTALNAME-X Search Results", # subject
                $HtmlBody, $HtmlItem,
                $TextBody, $TextItem,
                "", # headers
                FALSE # CollapseBodyMargins
            );
            $MailerTemplates = $Mailer->GetTemplateList();
        }

        # if there is no default template configured, set one up
        if (is_null($this->ConfigSetting("DefaultEmailTemplate")))
        {
            # select the default template
            $DefaultTemplate = array_search("Saved Search Default", $MailerTemplates);
            $this->ConfigSetting("DefaultEmailTemplate", $DefaultTemplate);
        }

        # set up templates for all our schemas
        foreach (MetadataSchema::GetAllSchemas() as $ScId => $Schema)
        {
            $Setting = self::TemplateSettingName($ScId);
            # disable mailings for the User schema, set all others to
            # the default template
            $TgtValue = ($ScId == MetadataSchema::SCHEMAID_USER) ?
                self::TEMPLATE_DISABLED :
                self::TEMPLATE_DEFAULT;

            if (is_null($this->ConfigSetting($Setting)))
            {
                $this->ConfigSetting($Setting, $TgtValue);
            }
        }

        return NULL;
    }

    /**
    * Register information about this plugin.
    */
    public function Register()
    {
        $this->Name = "Saved Search Mailings";
        $this->Version = "1.0.0";
        $this->Description = "Send emails about new results for saved searches.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = [
            "CWISCore" => "4.0.1",
            "Mailer" => "1.2.0"
        ];
        $this->InitializeAfter = array("Mailer");
        $this->EnabledByDefault = FALSE;
    }

    /**
    * Set up plugin configuration options.
    * @return NULL if configuration setup succeeded, otherwise a string with
    *       an error message indicating why config setup failed.
    */
    public function SetUpConfigOptions()
    {
        $MPlugin = $GLOBALS["G_PluginManager"]->GetPlugin("Mailer");

        $this->CfgSetup["DefaultEmailTemplate"] = [
            "Type" => "Option",
            "Label" => "Default Email Template",
            "Help" => "The default Mailer template to use saved search emails.",
            "Options" => $MPlugin->GetTemplateList(),
        ];

        $TemplateList = [
            self::TEMPLATE_DISABLED => "(Disabled)",
            self::TEMPLATE_DEFAULT => "(Use default template)",
        ];
        $TemplateList += $MPlugin->GetTemplateList();

        foreach (MetadataSchema::GetAllSchemas() as $ScId => $Schema)
        {
            $this->CfgSetup[self::TemplateSettingName($ScId)] = [
                "Type" => "Option",
                "Label" => $Schema->Name()." Template",
                "Help" => "The Mailer template to use for ".$Schema->Name()." emails.",
                "Options" => $TemplateList,
            ];
        }

        return NULL;
    }

    /**
    * Hook event callbacks into the application framework.
    * @return Returns an array of events to be hooked into the application
    *      framework.
    */
    public function HookEvents()
    {
        return ["EVENT_PAGE_LOAD" => "PageLoad",
                "EVENT_HOURLY" => "RunSavedSearches",
                "Mailer_EVENT_IS_TEMPLATE_IN_USE" => "ClaimTemplate",
        ];
    }

    # ---- HOOKED METHODS ----------------------------------------------------

    /**
    * Callback for EVENT_PAGE_LOAD that watches for changes to saved
    *   search frequency and saves them.
    * @param string $PageName The name of the page.
    */
    public function PageLoad($PageName)
    {
        if ($PageName == "ListSavedSearches" &&
            isset($_POST["AC"]) && ($_POST["AC"] == "ChangeFrequency"))
        {
            # for each saved search
            $SSFactory = new SavedSearchFactory();
            $Searches = $SSFactory->GetSearchesForUser($GLOBALS["G_User"]->Id());
            foreach ($Searches as $Search)
            {
                # if frequency has changed
                if (isset($_POST["F_Frequency_".$Search->GetSearchId()]) &&
                    ($_POST["F_Frequency_".$Search->GetSearchId()] !=
                         $Search->Frequency()))
                {
                    # update frequency
                    $Search->Frequency($_POST["F_Frequency_".$Search->GetSearchId()]);
                }
            }
        }
        elseif ($PageName == "EditUserComplete" &&
                isset($_SESSION["IdOfUserBeingEdited"]))
        {
            $User = new CWUser($_SESSION["IdOfUserBeingEdited"]);

            $SSFactory = new SavedSearchFactory();
            $Searches = $SSFactory->GetSearchesForUser($User->Id());
            foreach ($Searches as $Search)
            {
                # if frequency has changed
                if (isset($_POST["F_Frequency_".$Search->GetSearchId()]) &&
                    ($_POST["F_Frequency_".$Search->GetSearchId()] !=
                         $Search->Frequency()))
                {
                    # update frequency
                    $Search->Frequency($_POST["F_Frequency_".$Search->GetSearchId()]);
                }
            }
        }
    }

    /**
    * Periodic task to send saved search emails.
    */
    public function RunSavedSearches()
    {
        $MailerPlugin = $GLOBALS["G_PluginManager"]->GetPlugin("Mailer");

        # get searches that need to be run
        $SSFactory = new SavedSearchFactory();
        $Searches = $SSFactory->GetSearchesDueToRun();

        $SearchEngine = new SPTSearchEngine();

        $RFactories = [];

        # for each search
        foreach ($Searches as $Search)
        {
            # retrieve search criteria and target user
            $SearchParameters = $Search->SearchParameters();
            $EndUser = new CWUser($Search->UserId());

            # signal event to allow modification of search parameters
            $SignalResult = $GLOBALS["AF"]->SignalEvent(
                "EVENT_FIELDED_SEARCH", array(
                    "SearchParameters" => $SearchParameters,
                    "User" => $EndUser,
                    "SavedSearch" => $Search));
            $SearchParameters = $SignalResult["SearchParameters"];

            # perform search
            $SearchResults = $SearchEngine->Search($SearchParameters);

            $TotResults = 0;

            # filter results
            foreach ($SearchResults as $ScId => $ScResults)
            {
                # it no template is configured for this schema, then
                # we won't be mailing about it
                if ($this->GetTemplate($ScId) === NULL)
                {
                    $SearchResults[$ScId] = [];
                }
                else
                {
                    # make sure we have a factory for this schema
                    if (!isset($RFactories[$ScId]))
                    {
                        $RFactories[$ScId] = new ResourceFactory($ScId);
                    }

                    # filter non-viewable resources
                    $ViewableResourceIds = $RFactories[$ScId]
                        ->FilterNonViewableResources(
                            array_keys($ScResults), $EndUser);
                    $SearchResults[$ScId] = array_intersect_key(
                        $ScResults, array_flip($ViewableResourceIds));

                }

                $TotResults += count($SearchResults[$ScId]);
            }

            # if any results remain
            if ($TotResults > 0)
            {
                # get the ResourceIds (all schemas) that matched last
                # time we ran this search
                $LastMatches = $Search->LastMatches();

                # merge together all the Ids that match this time
                $CurrentMatchingIds = [];
                foreach ($SearchResults as $ScResults)
                {
                    $CurrentMatchingIds = array_merge(
                        $CurrentMatchingIds, array_keys($ScResults));
                }

                # save these matches for future comparisons and update last run date
                $Search->SaveLastMatches($CurrentMatchingIds);
                $Search->UpdateDateLastRun();

                # iterate over the results for all schemas
                foreach ($SearchResults as $ScId => $ScResults)
                {
                    # get the list of resources that are new since last time
                    $NewMatchingIds = array_diff(
                        array_keys($ScResults), $LastMatches);

                    # if we had any new resources
                    if(count($NewMatchingIds))
                    {
                        # get the email template for this schema
                        $Template = $this->GetTemplate($ScId);

                        # additional values to replace in the template
                        $ExtraValues = [
                            "SEARCHNAME" => $Search->SearchName(),
                            "TOTALNEWMATCHES" => count($NewMatchingIds),
                            "TRUNCATIONMSG" => "",
                            "SEARCHCRITERIA" => $Search->GetSearchGroupsAsTextDescription(
                                NULL, FALSE, FALSE)];

                        # truncate longer lists of results
                        if (count($NewMatchingIds) > 100)
                        {
                            $ExtraValues["TRUNCATIONMSG"] =
                                "Listing the first 100 resources of "
                                .count($NewMatchingIds);
                            $NewMatchingIds = array_slice($NewMatchingIds, 0, 100);
                        }

                        # send the e-mail
                        $MailerPlugin->SendEmail(
                            $Template,
                            $EndUser,
                            $NewMatchingIds,
                            $ExtraValues);
                    }
                }
            }

            # if we're running in the background and are low on time,
            # mark ourselves to be re-queued and stop processing
            if ($GLOBALS["AF"]->IsRunningInBackground() &&
                $GLOBALS["AF"]->GetSecondsBeforeTimeout() < 30)
            {
                $GLOBALS["AF"]->RequeueCurrentTask(TRUE);
                break;
            }
        }
    }

    /**
    * Inform Mailer plugin about templates we are using so that they
    * won't be deleted out from under us.
    * @param int $TemplateId TemplateId being checked.
    * @param array $TemplateUsers Array of template users.
    * @return array of arguments to be used as a Chain event.
    */
    public function ClaimTemplate($TemplateId, $TemplateUsers)
    {
        if ($this->ConfigSetting("DefaultEmailTemplate") == $TemplateId)
        {
            $TemplateUsers[]= $this->Name." (default template)";
        }
        else
        {
            foreach (MetadataSchema::GetAllSchemas() as $ScId => $Schema)
            {
                $SchemaTemplate = $this->GetTemplate($ScId);
                if ($SchemaTemplate !== NULL &&
                    $SchemaTemplate == $TemplateId)
                {
                    $TemplateUsers[]= $this->Name." (".$Schema->Name().")";
                }
            }
        }

        return ["TemplateId" => $TemplateId, "TemplateUsers" => $TemplateUsers];
    }

    # ---- CALLABLE METHODS ----------------------------------------------------

    /**
    * Get search frequency options for a given user in the format
    * needed by HtmlOptionList.
    * @param CWUser $User User.
    * @return Array keyed by option id (a SavedSearch::SEARCHFREQ_
    *   constant in this case) where values give the text description of
    *   the search frequency.
    */
    public static function GetFrequencyOptions($User)
    {
        if ($User->HasPriv(PRIV_RESOURCEADMIN))
        {
            # resource admins get all frequency options
            $FrequencyOptions =  SavedSearch::GetSearchFrequencyList();
        }
        else
        {
            # for all other users, we disallow hourly mailings
            $FrequencyOptions = SavedSearch::GetSearchFrequencyList(
                SavedSearch::SEARCHFREQ_HOURLY);
        }

        return $FrequencyOptions;
    }

    /**
    * Get the email template for a specified schema.
    * @param int $SchemaId SchemaId to look up.
    * @return int Mailer template id or NULL when there is no template
    */
    public function GetTemplate($SchemaId)
    {
        $Setting = $this->ConfigSetting(self::TemplateSettingName($SchemaId));

        if ($Setting == self::TEMPLATE_DISABLED)
        {
            return NULL;
        }
        elseif ($Setting == self::TEMPLATE_DEFAULT)
        {
            return $this->ConfigSetting("DefaultEmailTemplate");
        }
        else
        {
            return $Setting;
        }
    }

    # ---- PRIVATE METHODS ----------------------------------------------------

    /**
    * Construct the setting name used to store the email template for
    * a schema.
    * @param int $SchemaId SchemaId.
    * @return string setting name.
    */
    private static function TemplateSettingName($SchemaId)
    {
        return "EmailTemplate_".$SchemaId;
    }

    # ---- CLASS CONSTANTS ----------------------------------------------------

    const TEMPLATE_DEFAULT = -1;
    const TEMPLATE_DISABLED = -2;
}
