<?PHP

class Classification_Test extends PHPUnit\Framework\TestCase
{
    protected static $TestFieldIds;
    protected static $TestFields;

    /**
    * Prior to running any of the tests, this function is
    * run. It creates all of the test Metadata fields and adds
    * them to class variables $TestFieldIds and $TestFields
    * so each function may use them.
    */
    public static function setUpBeforeClass()
    {
        # construct the schema object
        $Schema = new MetadataSchema(
            MetadataSchema::SCHEMAID_DEFAULT);

        self::$TestFieldIds = array();

        # outline fields to be created
        self::$TestFields = array(
          "ClassificationTestField" => MetadataSchema::MDFTYPE_TREE,
        );

        # create the fields
        foreach (self::$TestFields as $FieldName => $FieldType)
        {
            $TmpField = $Schema->GetItemByName($FieldName);
            if ($TmpField === NULL)
            {
                $TmpField = $Schema->AddField($FieldName, $FieldType);
            }
            $TmpField->IsTempItem(FALSE);
            self::$TestFieldIds[$FieldName] = $TmpField->Id();
        }

    }

    /**
    * After to running the tests, this function is
    * run. It deletes all of the test Metadata fields.
    */
    public static function tearDownAfterClass()
    {
        # construct the schema object
        $Schema = new MetadataSchema(
            MetadataSchema::SCHEMAID_DEFAULT);
        $Database = new Database();

        # drop all of the test fields
        foreach (self::$TestFieldIds as $FieldName => $FieldId)
        {
            $Schema->DropField($FieldId);

            # remove from OAIFieldMappings too
            $Database->Query("
                DELETE FROM OAIFieldMappings
                WHERE SPTFieldId = " . addslashes($FieldId));
        }
    }


    public function testClassification()
    {
        $MyId = self::$TestFieldIds['ClassificationTestField'];

        $TestParent = Classification::Create(
            "TestParent", $MyId);

        $this->assertInstanceOf(
            Classification::class, $TestParent);
        $this->assertEquals(
            $TestParent->FieldId(), $MyId);


        # attempt to create a duplicate w/o setting a parentid
        try
        {
            Classification::Create(
                "TestParent", $MyId);
            $this->fail(
                "Exception not thrown on creation of duplicate Classification");
        }
        catch (Exception $e)
        {
            $this->assertEquals(
                $e->getMessage(),
                "Duplicate name specified for new classification "
                ."(TestParent).");
        }

        # attempt to create a duplicate top-level class
        try
        {
            Classification::Create(
                "TestParent", $MyId, Classification::NOPARENT);
            $this->fail(
                "Exception not thrown on creation of duplicate Classification");
        }
        catch (Exception $e)
        {
            $this->assertEquals(
                $e->getMessage(),
                "Duplicate name specified for new classification "
                ."(TestParent).");
        }

        # attempt to create a child with an invalid parent
        try
        {
            Classification::Create(
                "TestBadChild", $MyId, -10);
            $this->fail(
                "Exception not thrown when creating with invalid ParentId");
        }
        catch (Exception $e)
        {
            $this->assertInstanceOf(
                InvalidArgumentException::class, $e);
            $this->assertEquals(
                $e->getMessage(),
                "Invalid parent ID specified (-10).");
        }

        # attempt to create a child, specifying the parent
        $TestFirstChild = Classification::Create(
            "FirstChild", $MyId, $TestParent->Id() );
        $this->assertInstanceOf(
            Classification::class, $TestFirstChild);
        $this->assertEquals(
            $TestFirstChild->ParentId(),
            $TestParent->Id() );

        # attempt to create a child, not specifying the parent
        $TestSecondChild = Classification::Create(
            "TestParent -- SecondChild", $MyId);
        $this->assertInstanceOf(
            Classification::class, $TestSecondChild);
        $this->assertEquals(
            Classification::SegmentsCreated(), 1);

        # test the various ways to retrieve our name
        $this->assertEquals(
            $TestSecondChild->Name(),
            "TestParent -- SecondChild");

        $this->assertEquals(
            $TestSecondChild->FullName(),
            "TestParent -- SecondChild");

        $this->assertEquals(
            $TestSecondChild->SegmentName(),
            "SecondChild");

        $this->assertNull(
            $TestSecondChild->VariantName());

        # test recalculating
        $IdsUpdated = $TestSecondChild->RecalcResourceCount();

        $Expected = array(
            $TestSecondChild->Id(),
            $TestParent->Id() );

        sort($IdsUpdated);
        sort($Expected);

        $this->assertEquals($IdsUpdated, $Expected);

        # Test initial Depth values
        $this->assertEquals(
            $TestSecondChild->Depth(), 1);
        $this->assertEquals(
            $TestParent->Depth(), 0);


        # test recalculating depth
        $TestParent->RecalcDepthAndFullName();

        # verify that names and depths remain correct after recalc
        $this->assertEquals(
            $TestSecondChild->FullName(),
            "TestParent -- SecondChild");
        $this->assertEquals(
            $TestSecondChild->SegmentName(),
            "SecondChild");

        $this->assertEquals(
            $TestSecondChild->Depth(), 1);
        $this->assertEquals(
            $TestParent->Depth(), 0);

        $this->assertEquals(
            $TestSecondChild->FullResourceCount(), 0);


        # test listing children
        $ChildIds = $TestParent->ChildList();
        $Expected = array(
            $TestFirstChild->Id(),
            $TestSecondChild->Id() );
        sort($ChildIds);
        sort($Expected);
        $this->assertEquals($ChildIds, $Expected);

        # test deleting
        $DelCount = $TestSecondChild->Delete();
        $this->assertEquals(
            $DelCount, 1);

        # test deletions that eat our parents
        $DelCount = $TestFirstChild->Delete(TRUE);
        $this->assertEquals(
            $DelCount, 2);

    }
}
