<?PHP
#
#   FILE:  SysConfig.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

# ----- LOCAL FUNCTIONS ------------------------------------------------------

/**
* Define fields for form.
* @return array Associative array of form field parameters, in the format
*       expected by FormUI.
*/
function DefineFormFields()
{
    # load up possible values for SearchEngineUpdatePriority and
    #       RecommenderEngineUpdatePriority settings
    $PFactory = new PrivilegeFactory();
    $TaskPriorities = array(
            ApplicationFramework::PRIORITY_BACKGROUND => "Background",
            ApplicationFramework::PRIORITY_LOW => "Low",
            ApplicationFramework::PRIORITY_MEDIUM => "Medium",
            ApplicationFramework::PRIORITY_HIGH => "High");

    # load up possible values for DefaultSortField setting
    $ResourceSchema = new MetadataSchema(MetadataSchema::SCHEMAID_DEFAULT);
    $PossibleSortFields = array("R" => "Relevance");
    foreach ($ResourceSchema->GetFields(MetadataSchema::MDFTYPE_TEXT
            | MetadataSchema::MDFTYPE_NUMBER
            | MetadataSchema::MDFTYPE_DATE
            | MetadataSchema::MDFTYPE_TIMESTAMP
            | MetadataSchema::MDFTYPE_URL) as $FieldId => $Field)
    {
        if ($Field->IncludeInSortOptions())
        {
            $PossibleSortFields[$FieldId] = $Field->Name();
        }
    }

    # set up editing form
    $FormFields = array(
            "HEADING-General" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "General",
                    ),
            "PortalName" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "Portal Name",
                    "Required" => TRUE,
                    "Help" => "The name of the CWIS site as displayed in the"
                            ." title bar of the browser window and the page"
                            ." header above the navigation bar.",
                    ),
            "AdminEmail" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "Administrator Email",
                    "Required" => TRUE,
                    "ValidateFunction" => array("FormUI", "ValidateEmail"),
                    "Help" => "The email address of the individual responsible"
                            ." for overall site management. Feedback and other"
                            ." administrative mail is directed to this address.",
                    ),
            "LegalNotice" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "Legal Notice",
                    "Help" => "The legal notice displayed at the bottom of"
                            ." each page in the default interface.",
                    ),
            "SiteKeywords" => array(
                    "Type" => FormUI::FTYPE_PARAGRAPH,
                    "Label" => "Site Keywords",
                    "Help" => "Used by search engines to find your site."
                            ." Separate words and phrases by commas.",
                    ),
            "AboutText" => array(
                    "Type" => FormUI::FTYPE_PARAGRAPH,
                    "Label" => "About Text",
                    "UseWYSIWYG" => TRUE,
                    "Help" => "Intended to hold information that briefly "
                            ."describes the site and its collection. "
                            ."This is the text that is viewable on the "
                            ."About page. HTML is permitted in the about text."
                    ),
            # -------------------------------------------------
            "HEADING-Features" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Features",
                    ),
            "AnnouncementsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Announcements",
                    "Help" => "Whether the <i>Announcements</i> section"
                            ." on the home page is displayed.",
                    ),
            "ForumsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Forums",
                    "Help" => "Whether <i>forums</i> can be created "
                            ."and used within the site."
                    ),
            "ResourceRatingsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Resource Ratings/Recommendations",
                    "Help" => "Whether <i>resource ratings</i> are "
                            ."enabled and displayed on the site."
                    ),
            "ResourceCommentsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Resource Comments",
                    "Help" => "Whether <i>resource comments</i> are "
                            ."enabled and displayed on the site."
                    ),
            "IncrementalKeywordSearchEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Incremental Keyword Search",
                    "Help" => "Whether users see an incremental "
                            ."keyword search, a.k.a. search-as-you-type, "
                            ."interface to interactively show a subset "
                            ."of search results for the current search "
                            ."string when performing a keyword search "
                            ."from the primary search form."
                    ),
            "OAISQEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "OAI-SQ",
                    "Help" => "Whether the OAI-SQ extension to OAI-PMH "
                            ."is enabled."
                    ),
            # -------------------------------------------------
            "HEADING-Interface" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Interface",
                    ),
            "NumAnnounceOnHomePage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Announcements On Home Page",
                    "MinVal" => 1,
                    "Help" => "The maximum number of announcements "
                            ."that will be displayed on the home page "
                            ."if announcements are enabled."
                    ),
            "NumResourcesOnHomePage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Resources On Home Page",
                    "MinVal" => 0,
                    "Help" => "The maximum number of resources that "
                            ."will be displayed in the New Resources "
                            ."section on the home page."
                    ),
            "ShowNumResourcesEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Display Resource Total On Home Page",
                    "Help" => "Determines whether the total number "
                            ."of publicly-viewable resources is "
                            ."displayed on the home page."
                    ),
            "DefaultActiveUI" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Default User Interface",
                    "Options" => $GLOBALS["AF"]->GetUserInterfaces(),
                    "Required" => TRUE,
                    "Help" => "Determines the user interface new "
                            ."members and logged-out users will view. "
                            ."Individual users may control this option "
                            ."for themselves through their preferences "
                            ."options if multiple interfaces are allowed "
                            ."by the site administrator. Selecting the "
                            ."Set All Users to This Interface checkbox "
                            ."will set each user's interface setting to "
                            ."the user interface selected."
                    ),
            "ForceDefaultActiveUI" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Set All Users to Default Interface",
                    "Help" => "When checked, this option will set all user"
                            ." accounts to the above Default User Interface.",
                    "Value" => FALSE,
                    ),
            "AllowMultipleUIsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Allow Multiple User Interfaces",
                    "Help" => "Determines whether users may use a different "
                            ."user interface when logged in by selecting one "
                            ."in their preferences options. System "
                            ."administrators may select different interfaces "
                            ."even if this option is enabled."
                    ),
            "ResourceLaunchesNewWindowEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Resource Launches New Window",
                    "Help" => "Determines whether clicking on a resource title "
                            ."or URL opens a new browser window when loading "
                            ."the resource's full record page. The full record "
                            ."page will open in the same browser window when "
                            ."this option is disabled. <b>Note:</b> launching "
                            ."a new window may interfere with assistive "
                            ."technology devices."
                    ),
            "ShowGroupNamesEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Show Group Names In Full Record Page",
                    "Help" => "Whether group names are showned in full record page."
                    ),
            "ResourceTitleField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource Title Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_TEXT,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    "Required" => TRUE,
                    "Help" => "The metadata field used as the title when "
                            ."displaying resources. Only Text fields can "
                            ."be used as the default title field."
                    ),
            "ResourceDescriptionField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource Description Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_PARAGRAPH,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    "Help" => "The metadata field used as the description "
                            ."field when displaying resources. Only "
                            ."Paragraph fields can be used as the "
                            ."default description field."
                    ),
            "ResourceUrlField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource URL Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_URL,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    "Help" => "The metadata field used as the URL "
                            ."field when displaying resources. "
                            ."Only Url fields can be used as "
                            ."the default URL field."
                    ),
            "ResourceFileField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource File Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_FILE,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    "Help" => "The metadata field used as the "
                            ."file field when displaying resources. "
                            ."Only File fields can be used as the "
                            ."default file field."
                    ),
            "ResourceScreenshotField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource Screenshot Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_IMAGE,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    "Help" => "The metadata field used as the "
                            ."screenshot field when displaying "
                            ."resources. Only Image fields can "
                            ."be used as the default screenshot field."
                    ),
            "PreferredLinkValue" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Preferred Link Value",
                    "Help" => "Used when both <i>Resource URL Field</i> and"
                            ." <i>Resource File Field</i> are set, for records"
                            ." where both fields have values.",
                    "Options" => array("URL" => "URL", "FILE" => "File"),
                    ),
            "TitlesLinkTo" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Titles Link to",
                    "Options" => array("URL" => "URL", "RECORD" => "Full Record"),
                    "Help" => "Determines whether to use the resource's full "
                            ."record page or its URL when displaying links "
                            ."containing its title."
                    ),
            "CollapseMetadataFieldGroups" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Collapse Metadata Field Groups",
                    "Help" => "Determines whether metadata field groups "
                            ."created on the Metadata Field Ordering page "
                            ."should be collapsed by default when "
                            ."editing a resource."
                    ),
            "RequireEmailWithFeedback" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Require Email Address with Feedback",
                    "Help" => "Determines whether users who are not "
                            ."logged in are required to include an "
                            ."e-mail address when submitting feedback."
                    ),
            "ForumsAllowHTML" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Forums Allow HTML",
                    "Help" => "Whether HTML is allowed in forums."
                    ),
            "ForumsUseWYSIWYG" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Forums Use WYSIWYG Editor for HTML",
                    "Help" => "Whether forums should use WYSIWYG Editor "
                            ."for HTML."
                    ),
            # -------------------------------------------------
            "HEADING-Browsing" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Browsing",
                    ),
            "NumClassesPerBrowsePage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Classifications Per Page When Browsing",
                    "MinVal" => 2,
                    "Help" => "The default number of classifications "
                            ."to display on the Browse Resources page "
                            ."before they are split up. System "
                            ."administrators should consider the size "
                            ."of the collection as well as the current "
                            ."state of browser technology as longer "
                            ."numbers of resource entries per page "
                            ."may require lengthy browser load times."
                    ),
            "NumColumnsPerBrowsePage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Columns Per Page When Browsing",
                    "MinVal" => 1,
                    "MaxVal" => 4,
                    "RecVal" => 2,
                    "Help" => "The number of columns in which to "
                            ."display the classifications on the "
                            ."Browse Resources page. (Minimum: "
                            ."<i>1</i>, Maximum: <i>4</i>, "
                            ."Recommended: <i>2</i>)"
                    ),
            "BrowsingFieldId" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Default Browsing Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_TREE,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    "Required" => TRUE,
                    "Help" => "The default field displayed and used "
                            ."as the default browsing option on the "
                            ."Browse Resources page. This may be set "
                            ."to any tree field present in the "
                            ."metadata schema. While the field "
                            ."specified will be the default browsing "
                            ."option, users may choose to browse by "
                            ."any tree field they have "
                            ."permission to browse."
                    ),
            # -------------------------------------------------
            "HEADING-Search" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Search",
                    ),
            "SearchTermsRequired" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Default Search Term Handling",
                    "Options" => array(1 => "AND", 0 => "OR"),
                    "Help" => "Determines whether AND or OR logic "
                            ."is used when more than one search "
                            ."term is specified. Resource records "
                            ."that contain all specified search "
                            ."terms will be retrieved when AND is "
                            ."selected. Resource records that have "
                            ."any of the search terms specified "
                            ."will be retrieved when OR is selected, "
                            ."but those that have two or more "
                            ."will be ranked higher."
                    ),
            "DisplayLimitsByDefault" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Display Search Limits by Default",
                    "Help" => "Determines whether the search "
                            ."limits on the Advanced Search "
                            ."page are displayed or hidden "
                            ."by default."
                    ),
            "DefaultRecordsPerPage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Default Search Results Per Page",
                    "Required" => TRUE,
                    "MinVal" => 5,
                    "RecVal" => 10,
                    "Help" => "Determines the default number "
                            ."of search results displayed per "
                            ."page. Users can override this "
                            ."setting from the User "
                            ."Preferences page."
                    ),
            "MaxFacetsPerField" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Max Facets Per Field",
                    "Help" => "The maximum number of facets to display "
                            ."per field in faceted search.",
                    "Required" => TRUE,
                    "MinVal" => 2,
                    "MaxVal" => 100,
                    "RecVal" => 20,
                    ),
            "DefaultSortField" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Default Sort Field",
                    "Options" => $PossibleSortFields,
                    "Required" => TRUE,
                    "Help" => "Determines the default field used "
                            ."to sort search results."
                    ),
            # -------------------------------------------------
            "HEADING-Metrics" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Metrics",
                    ),
            "AddAWStatsScript" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "AWStats Logging",
                    "Help" => "Whether AWStats logging is performed."
                    ),
            # -------------------------------------------------
            "HEADING-System" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "System",
                    ),
            "MaxSimultaneousTasks" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Maximum Simultaneous Background Tasks",
                    "MinVal" => 1,
                    "MaxVal" => 32,
                    "Value" => $GLOBALS["AF"]->MaxTasks(),
                    "Help" => "The maximum amount of tasks to run in "
                            ."the background per execution cycle."
                    ),
            "SearchEngineUpdatePriority" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Search Engine Update Task Priority",
                    "Options" => $TaskPriorities,
                    "Required" => TRUE,
                    "Help" => "The priority given to the tasks that "
                            ."run search engine updates for resources."
                    ),
            "RecommenderEngineUpdatePriority" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Recommender Engine Update Task Priority",
                    "Options" => $TaskPriorities,
                    "Required" => TRUE,
                    "Help" => "The priority given to the tasks that "
                            ."run recommender engine updates for resources."
                    ),
            # -------------------------------------------------
            "HEADING-UserAccounts" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "User Accounts",
                    ),
            "DefaultUserPrivs" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Default New User Privileges",
                    "AllowMultiple" => TRUE,
                    "Rows" => 15,
                    "Options" => $PFactory->GetPrivileges(TRUE, FALSE),
                    "Help" => "Determines the privilege flags that are "
                            ."given to users after they have "
                            ."registered for an account."
                    ),
            );

    # return form field info to caller
    return $FormFields;
}

/**
* Load values for form fields.
* @param array $FormFields Associative array of form field parameters, in
*       the format expected by FormUI.
* @return array Associative array of form field parameters, in the format
*       expected by FormUI, with field values filled in where available.
*/
function LoadFormValues($FormFields)
{
    $ResourceSchema = new MetadataSchema(MetadataSchema::SCHEMAID_DEFAULT);
    foreach ($FormFields as $FieldName => $FieldParams)
    {
        if ($FieldParams["Type"] == FormUI::FTYPE_HEADING) {  continue;  }
        unset($FieldValue);
        switch ($FieldName)
        {
            case "ResourceTitleField":
            case "ResourceDescriptionField":
            case "ResourceUrlField":
            case "ResourceFileField":
            case "ResourceScreenshotField":
                # (strip off "Resource" and "Field" off of the form field name,
                #       and use the result for the standard metadata field name)
                $MFieldName = str_replace(
                        array("Resource", "Field"), "", $FieldName);
                $FieldValue = $ResourceSchema->StdNameToFieldMapping($MFieldName);
                break;

            case "PreferredLinkValue":
            case "TitlesLinkTo":
            case "CollapseMetadataFieldGroups":
            case "RequireEmailWithFeedback":
                $FieldValue = $GLOBALS["G_SysConfig"]->Value($FieldName);
                break;

            default:
                if (!isset($FormFields[$FieldName]["Value"]))
                {
                    # retrieve field values from SystemConfiguration where available
                    if (method_exists($GLOBALS["G_SysConfig"], $FieldName))
                    {
                        $FieldValue = $GLOBALS["G_SysConfig"]->$FieldName();
                    }
                    else
                    {
                        throw new Exception("Configuration setting for which value"
                                ." is not available (".$FieldName.").");
                    }
                }
                break;
        }
        if (isset($FieldValue))
        {
            $FormFields[$FieldName]["Value"] = $FieldValue;
        }
    }

    return $FormFields;
}

/**
* Save values from form fields.
* @param array $NewSettings Associative array with field names for the index
*       and field values for the values.
*/
function SaveFormValues($NewSettings)
{
    $ResourceSchema = new MetadataSchema(MetadataSchema::SCHEMAID_DEFAULT);
    foreach ($NewSettings as $FieldName => $NewFieldValue)
    {
        switch ($FieldName)
        {
            case "ForceDefaultActiveUI":
                if ($NewFieldValue)
                {
                    # set all users to selected user interface
                    $UFactory = new CWUserFactory();
                    foreach ($UFactory->GetUserIds() as $UserId)
                    {
                        $User = new CWUser($UserId);
                        $User->Set("ActiveUI", $NewSettings["DefaultActiveUI"]);
                    }
                }
                break;

            case "MaxSimultaneousTasks":
                $GLOBALS["AF"]->MaxTasks($NewFieldValue);
                break;

            case "ResourceDescriptionField":
            case "ResourceFileField":
            case "ResourceScreenshotField":
            case "ResourceTitleField":
            case "ResourceUrlField":
                if (!strlen($NewFieldValue))
                {
                    # (NULL is what you pass in to clear a standard field mapping)
                    $NewFieldValue = NULL;
                }
                # (strip off "Resource" and "Field" off of the form field name,
                #       and use the result for the standard metadata field name)
                $MFieldName = str_replace(
                        array("Resource", "Field"), "", $FieldName);
                $ResourceSchema->StdNameToFieldMapping($MFieldName, $NewFieldValue);
                break;

            case "CollapseMetadataFieldGroups":
            case "PreferredLinkValue":
            case "RequireEmailWithFeedback":
            case "TitlesLinkTo":
                # save values that match SystemConfiguration DB table columns
                $GLOBALS["G_SysConfig"]->Value($FieldName, $NewFieldValue);
                break;

            default:
                # save values via SystemConfiguration method if available
                if (method_exists($GLOBALS["G_SysConfig"], $FieldName))
                {
                    $GLOBALS["G_SysConfig"]->$FieldName($NewFieldValue);
                }
                else
                {
                    throw new Exception("New configuration value for which"
                            ." setting is not available (".$FieldName.").");
                }
                break;
        }
    }
}


# ----- MAIN -----------------------------------------------------------------

# check permissions
CheckAuthorization(PRIV_SYSADMIN, PRIV_COLLECTIONADMIN);

# set up form
$FormFields = DefineFormFields();

# load up current values for form fields
$FormFields = LoadFormValues($FormFields);

# instantiate form UI
$H_FormUI = new FormUI($FormFields);

# act on any button push
$ButtonPushed = GetFormValue("Submit");
switch ($ButtonPushed)
{
    case "Save":
        # check values and bail out if any are invalid
        if ($H_FormUI->ValidateFieldInput())
        {
            return;
        }

        # retrieve values from form
        $NewSettings = $H_FormUI->GetNewValuesFromForm();

        # save updated values
        SaveFormValues($NewSettings);

        # return to admin menu page
        $GLOBALS["AF"]->SetJumpToPage("SysAdmin");
        break;

    case "Cancel":
        # return to admin menu page
        $GLOBALS["AF"]->SetJumpToPage("SysAdmin");
        break;
}
