<?PHP
#
#   FILE:  ImportUsersExecute.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2004-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

# ----- LOCAL FUNCTIONS ------------------------------------------------------

# ----- MAIN -----------------------------------------------------------------

# check if current user is authorized
if (!CheckAuthorization(PRIV_SYSADMIN, PRIV_USERADMIN)) {  return;  }

$DB = new Database();

# initialize variables
$FSeek = GetArrayValue($_SESSION, "FSeek", 0);
$TempFile = GetArrayValue($_SESSION, "TempFile", NULL);
$H_UserCount = GetArrayValue($_SESSION, "UserCount", 0);
$H_ErrorMessages = GetArrayValue($_SESSION, "ErrorMessages", array());
$H_DuplicateEntries = GetArrayValue($_SESSION, "DuplicateEntries", array());
$H_PrivNotFound = GetArrayValue($_SESSION, "PrivNotFound", array());
$UsersProcessed = GetArrayValue($_SESSION, "UsersProcessed", array());
$LineCount = GetArrayValue($_SESSION, "LineCount", 0);
$LastUserName = GetArrayValue($_SESSION, "LastUserName", NULL);
$ImportComplete = FALSE;

# check for Cancel button
if (isset($_POST["Submit"]) && $_POST["Submit"] == "Cancel")
{
    $ImportComplete = TRUE;
    $GLOBALS["AF"]->SetJumpToPage("SysAdmin");
    return;
}

if (isset($_FILES['F_FileName']['tmp_name']) &&
    is_uploaded_file($_FILES['F_FileName']['tmp_name']))
{
    # zero length or tiny file
    if ($_FILES['F_FileName']['size'] == 0)
    {
        $ErrorMessage = "Error: File doesn't exist or is empty.";
        $_SESSION["ErrorMessage"] = $ErrorMessage;
        $GLOBALS["AF"]->SetJumpToPage("ImportUsers");
        return;
    }
    # files with illegal extensions
    elseif (preg_match('/\.php/i', $_FILES['F_FileName']['name'])
            || preg_match('/\.js/i', $_FILES['F_FileName']['name'])
            || preg_match('/\.html/i', $_FILES['F_FileName']['name']))
    {
        $ErrorMessage = "Error: File type is not allowed.";
        $_SESSION["ErrorMessage"] = $ErrorMessage;
        $GLOBALS["AF"]->SetJumpToPage("ImportUsers");
        return;
    }

    $ScriptFileName = GetFullScriptFileName();

    $TempFile = dirname($ScriptFileName).
        "/tmp/".
        $_FILES['F_FileName']['name'];

    copy($_FILES['F_FileName']['tmp_name'], $TempFile);

    $fp = fopen($TempFile, 'r');
    unlink($_FILES['F_FileName']['tmp_name']);
}
else if ($FSeek > 0)
{
    $fp = fopen($TempFile, 'r');
}
else
{
    switch($_FILES['F_FileName']['error'])
    {
        # uploaded file exceeds the upload_max_filesize directive in php.ini
        case UPLOAD_ERR_INI_SIZE:
            $ErrorMessage = "The file you are trying to upload is too big.";
            break;

        # uploaded file exceeds the MAX_FILE_SIZE directive that
        # was specified in the html form
        case UPLOAD_ERR_FORM_SIZE:
            $ErrorMessage = "The file you are trying to upload is too big.";
            break;

        # uploaded file was only partially uploaded
        case UPLOAD_ERR_PARTIAL:
            $ErrorMessage =
                "The file you are trying upload was only partially uploaded.";
            break;

        # no file was uploaded
        case UPLOAD_ERR_NO_FILE:
            $ErrorMessage = "You must select a file for upload.";
            break;

        # a default error, just in case!
        default:
            $ErrorMessage =  "There was a problem with your upload.";
            break;
    }
    $_SESSION["ErrorMessage"] = $ErrorMessage;
    $GLOBALS["AF"]->SetJumpToPage("ImportUsers");
    return;
}

# seek to the next line
if ($FSeek > 0)
{
    fseek($fp, $FSeek);
}

# begin import
$Schema = new MetadataSchema();
$PrivDescriptions = (new PrivilegeFactory())->GetPrivileges(TRUE, FALSE);
$LocalLineCount = 0;

while (!feof($fp) && $LocalLineCount < 500 && $ImportComplete == FALSE)
{
    # read in line from import file
    $fline = fgets($fp, 4096);

    # update variables
    $LocalLineCount++;
    $LineCount++;
    $_SESSION["LineCount"] = $LineCount;
    $FSeek += strlen($fline);
    $_SESSION["FSeek"] = $FSeek;

    $Value = NULL;
    # parse line from import file
    $Vars = explode("\t", $fline);

    $NumberOfVars = count($Vars);
    if ($NumberOfVars < 1 || feof($fp))
    {
        $ImportComplete = TRUE;
        break;
    }

    # should be 14 variables per line
    if (count($Vars) != 14)
    {
        $ErrorMessage = "Error: Wrong number of fields on Line ".$LineCount;
        $_SESSION["ErrorMessage"] = $ErrorMessage;

        # clean up file pointer, temp file
        fclose($fp);
        unlink($TempFile);

        # jump back and display error message
        $GLOBALS["AF"]->SetJumpToPage("ImportUsers");
        return;
    }

    # initial the vars
    $UserName = trim(addslashes($Vars[0]));
    $UserPassword = $Vars[1];
    $Email = addslashes($Vars[2]);
    $WebSite = addslashes($Vars[3]);
    $RealName = addslashes($Vars[4]);
    $AddressLineOne = addslashes($Vars[5]);
    $AddressLineTwo = addslashes($Vars[6]);
    $City = addslashes($Vars[7]);
    $State = addslashes($Vars[8]);
    $Country = addslashes($Vars[9]);
    $ZipCode = $Vars[10];
    $ActiveUI = $Vars[11];
    $BrowsingField = trim($Vars[12]);
    $PrivDescription = trim($Vars[13]);

    try
    {
        $Field = $Schema->GetFieldByName($BrowsingField);
    }
    catch (Exception $e)
    {
        $Field = NULL;
    }

    # force FieldId to reasonable value if not set
    if (is_object($Field))
    {
        $FieldId = $Field->Id();
    }
    else
    {
        $FieldId = "NULL";
    }

    # default UserPassword to UserName if blank
    if (empty($UserPassword))
    {
        $UserPassword = $UserName;
    }

    # encrypt the password
    if (isset($_POST["F_PasswordFlag"]) &&
        $_POST["F_PasswordFlag"] == 1)
    {
        $UserPassword = crypt($UserPassword);
    }

    # bail out if line was empty or end of file encountered
    if (is_null($UserName) || feof($fp))
    {
        break;
    }

    # check if this is a privilege line
    if ($LastUserName == $UserName && !empty($PrivDescription))
    {
        # if this privilege line belongs to a duplicate entry,
        #       save this line as a duplicate entry as well
        if (isset($UsersProcessed[$UserName]) && $UsersProcessed[$UserName] === 1)
        {
            $H_DuplicateEntries[$LineCount] = $fline;
        }
        # if this user was not created, its privilege lines should also be skipped
        else if (!array_key_exists($UserName, $H_ErrorMessages))
        {
            $Privilege = array_search($PrivDescription, $PrivDescriptions);

            # if a privilege is not found in the privilege factoroy,
            #       notify admin about this failure
            if ($Privilege === FALSE)
            {
                $H_PrivNotFound[$LineCount] = $fline;
            }
            else
            {
                $ThisUser = new CWUser($UserName);
                $ThisUser->GivePriv($Privilege);
            }
        }
        continue;
    }

    # cache username for privileges
    $LastUserName = $UserName;

    # only process the 1st entry if there are duplicate entries (same username)
    if (array_key_exists($UserName, $UsersProcessed))
    {
        $H_DuplicateEntries[$LineCount] = $fline;
        $UsersProcessed[$UserName] = 1;
        continue;
    }
    else
    {
        $UsersProcessed[$UserName] = 0;
    }

    # attemp to create user
    $UserFactory = new CWUserFactory();
    $User = $UserFactory->CreateNewUser(
            $UserName, $UserPassword, $UserPassword, $Email, $Email);

    # check if user creation succeeded, if failed, save error messages to print
    if (!($User instanceof CWUser))
    {
        $H_ErrorMessages[$UserName]["LineNumber"] = $LineCount;
        foreach ($User as $ErrorCode)
        {
            $H_ErrorMessages[$UserName]["Messages"][] =
                    CWUser::GetStatusMessageForCode($ErrorCode);
        }
        continue;
    }

    $User->Set("WebSite", isset($Website) ? $Website : "");
    $User->Set("AddressLineOne", isset($AddressLineOne) ? $AddressLineOne : "");
    $User->Set("AddressLineTwo", isset($AddressLineTwo) ? $AddressLineTwo : "");
    $User->Set("City", isset($City) ? $City : "");
    $User->Set("State", isset($State) ? $State : "");
    $User->Set("ZipCode", isset($ZipCode) ? $ZipCode : "");
    $User->Set("Country", isset($Country) ? $Country : "");
    $User->Set("RealName", isset($RealName) ? $RealName : "");
    $User->Set("ActiveUI", isset($ActiveUI) ? $ActiveUI
            : $GLOBALS["G_SysConfig"]->DefaultActiveUI());

    $User->IsActivated(TRUE);

    if (!isset($_POST["F_PasswordFlag"]))
    {
        $User->SetEncryptedPassword($UserPassword);
    }
    else if($_POST["F_PasswordFlag"] != 1)
    {
        $User->SetEncryptedPassword($UserPassword);
    }

    # add in privilege if set
    if (!empty($PrivDescription))
    {
        $Privilege = array_search($PrivDescription, $PrivDescriptions);
        $ThisUser = new CWUser($UserName);
        $ThisUser->GivePriv($Privilege);
    }

    # keep track of number of users added
    $H_UserCount++;
    $_SESSION["UserCount"] = $H_UserCount;
}

# end of file reached?
if (feof($fp))
{
    $ImportComplete = TRUE;
}
$_SESSION["UserCount"] = $H_UserCount;
$_SESSION["FSeek"] = $FSeek;
$_SESSION["TempFile"] = $TempFile;
$_SESSION["ErrorMessages"] = $H_ErrorMessages;
$_SESSION["UsersProcessed"] = $UsersProcessed;
$_SESSION["DuplicateEntries"] = $H_DuplicateEntries;
$_SESSION["PrivNotFound"] = $H_PrivNotFound;
$_SESSION["LastUserName"] = $LastUserName;

#  Time to auto-refresh?
if ($ImportComplete == FALSE)
{
    $GLOBALS["AF"]->SetJumpToPage("index.php?P=ImportUsersExecute", 1);
}

PageTitle("Import Users");

# register post-processing function with the application framework
$GLOBALS["AF"]->AddPostProcessingCall("PostProcessingFn",
        $TempFile, $fp, $ImportComplete);

# post-processing call
/**
* Handle post-processing after a pageload.
* Post-processing task: clear import file and session variables.
* @param string $TempFile Temporary file name.
* @param int $fp Active file descriptor.
* @param bool $ImportComplete True when import is complete.
*/
function PostProcessingFn($TempFile, $fp, $ImportComplete)
{
    if ($ImportComplete == TRUE)
    {
        fclose($fp);
        # remove temporary uploaded file
        unlink($TempFile);

        $Variables = array("UserCount", "FSeek", "TempFile",
                "LineCount", "ErrorMessages", "LastUserName",
                "DuplicateEntries", "UsersProcessed", "PrivNotFound");
        foreach ($Variables as $Var)
        {
            unset($_SESSION[$Var]);
        }
    }
}
