<?PHP
#
#   FILE:  MyForumPosts.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2002-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Plugin that adds recently added forum posts from the current user to the
* sidebar.
*/
class MyForumPosts extends Plugin
{

    /**
    * Register information about this plugin.
    */
    public function Register()
    {
        $this->Name = "My Forum Posts";
        $this->Version = "1.0.0";
        $this->Description = "Displays list of user's recent CWIS forum"
                ." posts in a box in the sidebar.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
                "CWISCore" => "2.2.0");
        $this->EnabledByDefault = TRUE;

        $this->CfgSetup["BoxHeader"] = array(
                "Type" => "Text",
                "Label" => "Box Header",
                "Help" => "Title for <i>My Posts</i> box in sidebar.",
                "Size" => 40,
                "MaxLength" => 60,
                );
        $this->CfgSetup["ListLength"] = array(
                "Type" => "Number",
                "Label" => "Number of Posts to Display",
                "Help" => "Number of recent posts to display in box.",
                "MaxVal" => 20,
                );
    }

    /**
    * Initialize default settings.
    */
    public function Install()
    {
        $this->ConfigSetting("BoxHeader", "My Posts");
        $this->ConfigSetting("ListLength", 5);
    }

    /**
    * Hook the events into the application framework.
    * @return Returns an array of events to be hooked into the application
    *      framework.
    */
    public function HookEvents()
    {
        return array(
                "EVENT_REQUEST_SIDEBAR_CONTENT" => "GenerateForumPostBox",
                );
    }

    /**
    * Generate and return the sidebar content for the recently added forum posts
    * of the current user.
    * @return Returns the sidebar content as a string for the recently added
    *      forum posts of the current user.
    */
    public function GenerateForumPostBox()
    {
        # if the user is logged in
        $Box = "";
        if ($GLOBALS["G_User"]->IsLoggedIn())
        {
            # retrieve list of messages recently posted by user
            $MFactory = new MessageFactory();
            $Messages = $MFactory->GetMessagesPostedByUser(
                    $GLOBALS["G_User"]->Id(), $this->ConfigSetting("ListLength"));

            # if there were messages found
            if (count($Messages))
            {
                ob_start();
                ?><div class="cw-section cw-section-simple cw-html5-section">
                    <div class="cw-section-header cw-html5-header">
                        <b><?PHP  print($this->ConfigSetting("BoxHeader"));  ?></b>
                    </div>
                    <div class="cw-section-body">
                        <ul class="cw-list cw-list-noindent cw-list-dotlist">
                            <?PHP
                            # for each message
                            foreach ($Messages as $Msg)
                            {
                                # format message body excerpt for tooltip
                                $Body = defaulthtmlentities(StdLib::NeatlyTruncateString(
                                        $Msg->Body(), 250));

                                # format message subject for message title
                                $Subject = defaulthtmlentities(
                                        StdLib::NeatlyTruncateString(
                                        $Msg->Subject(), 40));

                                # if message was a resource comment
                                if ($Msg->ParentType() == Message::PARENTTYPE_RESOURCE)
                                {
                                    # set link to full record page for resource
                                    $Link = "index.php?P=FullRecord"
                                            ."&amp;ID=".$Msg->ParentId();
                                }
                                # else message was a forum posting
                                else
                                {
                                    # set link to topic page where message was posted
                                    $Link = "index.php?P=TopicMessages"
                                            ."&amp;TI=".$Msg->ParentId();
                                }

                                # add message to list in box
                                ?><li><a href="<?PHP  print($Link);
                                ?>" title="<?PHP print($Body);
                                ?>"><?PHP  print($Subject);  ?></a></li><?PHP
                            }
                        ?></ul>
                    </div>
                </div><?PHP
                $Box = ob_get_contents();
                ob_end_clean();
            }
        }

        # return generated box to caller
        return $Box;
    }
}
