<?PHP
#
#   FILE:  Chart_Base.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Base class for generating and displaying a chart.
*/
abstract class Chart_Base
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Set legend position.
    * @param string $Position Legend position as a BarChart::LEGEND_
    * constant. LEGEND_BOTTOM will place the legend below the chart,
    * LEGEND_RIGHT will place it at the right, LEGEND_INSET will
    * display it as an inset within the graphic, and LEGEND_NONE will
    * disable the legend entirely.
    * @throws Exception If an invalid legend position is supplied.
    */
    public function LegendPosition($Position)
    {
        if (!in_array($Position,
                      [self::LEGEND_BOTTOM, self::LEGEND_RIGHT,
                       self::LEGEND_INSET, self::LEGEND_NONE]))
        {
            throw new Exception("Unsupported legend position: ".$Position);
        }
        $this->LegendPosition = $Position;
    }

    /**
    * Set shortened labels to be used in the legend of the chart.
    * @param array $LegendLabels Array of legend labels to use, with keys
    * matching the keys of the data passed to the constructor. Any
    * labels that do not need shortening may be omitted.
    */
    public function LegendLabels($LegendLabels)
    {
        $this->LegendLabels = $LegendLabels;
    }

    /**
    * Set color palette.
    * @param array $NewValue Array of hex colors (e.g., '#1f77b4'),
    * indexed by the corresponding labels (OPTIONAL). Bars with no
    * color defined will use an autogenerated color.
    */
    public function Colors($NewValue=NULL)
    {
        if (!is_null($NewValue))
        {
            $this->Colors = $NewValue;
        }

        return $this->Colors;
    }

    /**
    * Get/Set height of the chart including the legend.
    * @param int $NewValue New height in pixels (OPTIONAL).
    */
    public function Height($NewValue)
    {
        if (!is_null($NewValue))
        {
            $this->Height = $NewValue;
        }

        return $this->Height;
    }

    /**
    * Get/Set width of the chart including the legend.
    * @param int $NewValue New width in pixels (OPTIONAL).
    */
    public function Width($NewValue)
    {
        if (!is_null($NewValue))
        {
            $this->Width = $NewValue;
        }

        return $this->Width;
    }

    /**
    * Display a chart. These charts are currently generated with
    * c3.js, which outputs SVG that is annotated with CSS classes for
    * styling (C3 is only relevant insofar as it is reflected in the
    * CSS class names). For example, if there is a data item called
    * "Engineering", then the chart element for that item will be in a
    * g.c3-target-Engineering. The text giving the label for that
    * element can be changed to black with CSS like:
    *
    * g.c3-target-Engineering text {
    *   fill: black;
    * }
    *
    * Similarly, the text for the corresponding legend item:
    *
    * g.c3-legend-item-Engineering text {
    *   fill: green;
    * }
    *
    * Note that the foreground color of an SVG text is controlled by
    * 'fill' rather than 'color' because consistency is overrated. A
    * reference to the SVG elements and the CSS properties that they
    * understand can be found at:
    *
    *   https://developer.mozilla.org/en-US/docs/Web/SVG/Element
    *
    * In practice, it's often simplest to use a browser's "Inspect
    * Element" feature to locate SVG elements in order to customize
    * them.
    *
    * @param string $ContainerId Id to use when generating the div to contain this chart.
    */
    public function Display($ContainerId)
    {
        # be sure the necessary js and css are loaded
        $GLOBALS["AF"]->RequireUIFile("d3.js");
        $GLOBALS["AF"]->RequireUIFile("c3.js");
        $GLOBALS["AF"]->RequireUIFile("c3.css");
        $GLOBALS["AF"]->RequireUIFile("Chart_Base.css");

        # declare the chart data that we will give to c3.generate
        # for function callbacks, give the function a name ending with
        # '_fn' and include the function name as a string
        $this->Chart = [
            "bindto" => "#".$ContainerId,
            "size" => [
                "height" => $this->Height,
                "width" => $this->Width,
            ],
            "tooltip" => [
                "format" => [
                    "name" => "tooltip_name_fn",
                ],
            ],
        ];

        # set up legend positioning
        if ($this->LegendPosition == self::LEGEND_NONE)
        {
            $this->Chart["legend"]["show"] = FALSE;
        }
        else
        {
            $this->Chart["legend"]["position"] = $this->LegendPosition;
        }

        # if the user provided a color palette, set that up as will
        if (!is_null($this->Colors()))
        {
            # sort user-provided colors into the correct order
            $Palette = [];
            foreach ($this->Data as $Label => $Value)
            {
                $Palette[]= isset($this->Colors[$Label]) ?
                    $this->Colors[$Label] :
                    $this->GenerateRgbColorString($Label);
            }

            $this->Chart["color"]["pattern"] = $Palette;
        }

        static::PrepareData();


        // @codingStandardsIgnoreStart
        ?><div id="<?= $ContainerId ?>" class="cw-<?= strtolower(get_called_class()) ?>"></div>
        <script type="text/javascript">
        $(document).ready(function(){
            // define state variables for this chart
            var label_lut = <?= json_encode($this->LabelLUT) ?>;
            <?PHP static::DeclareStateVariables(); ?>

            // define helper functions for this chart
            function tooltip_name_fn(name, ratio, id, index) {
                return name in label_lut ? label_lut[name] : name;
            }
            <?PHP static::DeclareHelperFunctions(); ?>

            // get the chart spec data
            var chart_spec = <?= json_encode($this->Chart) ?>;

            // convert any strings that refer to functions into callable function refs
            function eval_fns(obj){
                for (var prop in obj) {
                    if (typeof obj[prop] == "object") {
                        eval_fns(obj[prop]);
                    } else if (typeof obj[prop] == "string" && obj[prop].match(/_fn$/)) {
                        obj[prop] = eval(obj[prop]);
                    }
                }
            }
                eval_fns(chart_spec);

            // generate the chart
            c3.generate(chart_spec);
        });
        </script><?PHP
        // @codingStandardsIgnoreEnd
    }

    # legend position constants
    const LEGEND_BOTTOM = "bottom";
    const LEGEND_RIGHT = "right";
    const LEGEND_INSET = "inset";
    const LEGEND_NONE = "none";

    # ---- PRIVATE INTERFACE --------------------------------------------------

    /**
    * Massage data provided by the user into an appropriate format for
    * plotting and do any necessary tweaks to $this->Chart. Child
    * classes MUST implement this method.
    */
    abstract protected function PrepareData();

    /**
    * Output var declarations for any js state variables needed in
    * this chart's display helper functions.
    */
    protected function DeclareStateVariables()
    {
        return;
    }

    /**
    * Output function definitions for any needed javascript display
    * helper functions. Note that all these functions must have names
    * ending with _fn for the eval_fns() js helper to make them
    * callable.
    */
    protected function DeclareHelperFunctions()
    {
        return;
    }

    /**
    * Get RGB hex color when no color supplied.
    * @param string $DataIndex Index for data for which color will be used.
    * @return string RGB hex color string.
    */
    protected function GenerateRgbColorString($DataIndex)
    {
        return "#".substr(md5($DataIndex), 0, 6);
    }

    /**
    * Merge an array of settings into $this->Chart.
    * @param array $Data Settings to merge.
    */
    protected function AddToChart($Data)
    {
        $this->AddToArray($this->Chart, $Data);
    }

    /**
    * Merge elements from a source array into a dest array.
    * @param array $Tgt Target array.
    * @param array $Data Data to be added.
    */
    protected function AddToArray(&$Tgt, $Data)
    {
        foreach ($Data as $Key => $Val)
        {
            if (isset($Tgt[$Key]) &&
                is_array($Tgt[$Key]) && is_array($Val))
            {
                $this->AddToArray($Tgt[$Key], $Val);
            }
            else
            {
                $Tgt[$Key] = $Val;
            }
        }
    }

    # data provided by caller
    protected $Data = [];

    # chart parameters that can be changed prior to generation
    protected $LegendPosition = self::LEGEND_BOTTOM;
    protected $Colors = NULL;
    protected $LegendLabels = [];
    protected $Height = 600;
    protected $Width = 600;

    # internal variables used to generate the chart
    protected $LabelLUT = [];
    protected $Chart = NULL;
}
