<?PHP
#
#   FILE:  CWUserFactory.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* CWIS-specific user factory class.
*/
class CWUserFactory extends UserFactory
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Construct the user factory object.
    */
    public function __construct()
    {
        parent::__construct();
        $this->ResourceFactory = new ResourceFactory(MetadataSchema::SCHEMAID_USER);
    }

    /**
    * Get a list of users sorted by how many resources they have added or
    * edited, starting with those who have added/edited the most.
    * @param int $Limit The maximum number of users to return.  (OPTIONAL,
    *       defaults to 5.)
    * @return array List of users, with user IDs for the index and CWUser
    *       objects for the values.
    */
    public function GetTopContributors($Limit = 5)
    {
        # assume no users will be found
        $Users = array();

        $Schema = new MetadataSchema();
        $LastModField = $Schema->GetFieldByName("Last Modified By Id");

        # fetch the top contributors
        $this->DB->Query(
            "SELECT UserId FROM ResourceUserInts "
            ." WHERE FieldId = ".$LastModField->Id()
            ." GROUP BY UserId"
            ." ORDER BY COUNT(*) DESC"
            ." LIMIT ".intval($Limit));
        $UserIds = $this->DB->FetchColumn("UserId");

        # for each user id found
        foreach ($UserIds as $UserId)
        {
            $Users[$UserId] = new CWUser($UserId);
        }

        # return the newest users
        return $Users;
    }

    /**
    * Get the users sorted by when they last added or edited a resource
    * starting with those who added/edited a resource most recently.
    * @param int $Limit The maximum number of users to return.  (OPTIONAL,
    *       defaults to 5.)
    * @return array List of users, with user IDs for the index and CWUser
    *       objects for the values.
    */
    public function GetMostRecentContributors($Limit = 5)
    {
        # assume no users will be found
        $Users = array();

        $Schema = new MetadataSchema();
        $LastModField = $Schema->GetFieldByName("Last Modified By Id");

        # fetch the top contributors
        $this->DB->Query(
            "SELECT UserId FROM ResourceUserInts RU, Resources R "
            ." WHERE RU.FieldId = ".$LastModField->Id()
            ." AND R.ResourceId = RU.ResourceId "
            ." GROUP BY RU.UserId"
            ." ORDER BY MAX(R.DateLastModified) DESC"
            ." LIMIT ".intval($Limit));
        $UserIds = $this->DB->FetchColumn("UserId");

        # for each user id found
        foreach ($UserIds as $UserId)
        {
            $Users[$UserId] = new CWUser($UserId);
        }

        # return the newest users
        return $Users;
    }

    /**
    * Find all users that meet the requirements for a specified privilege set.  This
    * is (currently) a very brute-force, inefficient implementation, so it should
    * not be used anywhere long execution times might be an issue.
    * @param PrivilegeSet $Privset Privilege Set giving the requirements to satisfy.
    * @param array $ResourceIds IDs of resources to use for comparisons, if the
    *       set includes resource-dependent conditions.  (OPTIONAL)
    * @return array Array with IDs of users that meet requirements for the index,
    *       and which resource IDs match for that user for the values.
    */
    public function FindUsersThatMeetRequirements($Privset, $ResourceIds = array())
    {
        # if there are necessary privileges for this privilege set
        $ReqPrivs = $Privset->GetPossibleNecessaryPrivileges();
        if (count($ReqPrivs))
        {
            # start with only those users who have at least one of those privileges
            $UserIds = array_keys($this->GetUsersWithPrivileges($ReqPrivs));
        }
        else
        {
            # start with all users
            $UserIds = $this->GetUserIds();
        }

        # determine of individual resources will need to be checked
        $NeedToCheckResources =
            (count($ResourceIds) && count($Privset->FieldsWithUserComparisons()))
            ? TRUE : FALSE ;

        # build up a list of matching users
        $UsersThatMeetReqs = array();

        $MemLimit = $GLOBALS["AF"]->GetPhpMemoryLimit();
        $ResourceCache = [];

        # iterate over all the users
        foreach ($UserIds as $UserId)
        {
            # load user
            $User = new CWUser($UserId);

            if ($NeedToCheckResources)
            {
                # iterate over all the resources
                foreach ($ResourceIds as $ResourceId)
                {
                    # if we're running low on memory, nuke the resource cache
                    if ($GLOBALS["AF"]->GetFreeMemory() / $MemLimit
                        < self::$LowMemoryThresh)
                    {
                        $ResourceCache = [];
                    }

                    # load resource
                    if (!isset($ResourceCache[$ResourceId]))
                    {
                        $ResourceCache[$ResourceId] = new Resource($ResourceId);
                    }

                    # if user meets requirements for set
                    if ($Privset->MeetsRequirements(
                        $User, $ResourceCache[$ResourceId]))
                    {
                        # add resource to user's list
                        $UsersThatMeetReqs[$UserId][] = $ResourceId;
                    }
                }
            }
            else
            {
                # if user meets requirements for set
                if ($Privset->MeetsRequirements($User))
                {
                    # add user to list
                    $UsersThatMeetReqs[$UserId] = $ResourceIds;
                }
            }
        }

        # return IDs for users that meet requirements to caller
        return $UsersThatMeetReqs;
    }

    /**
    * Derive a unique username from an email address.
    * @param string $Email Source email address.
    * @return string Unique username.
    */
    public static function GenerateUniqueUsernameFromEmail($Email)
    {
        $TrialName = explode('@', $Email);
        $TrialName = array_shift($TrialName);
        $TrialName = preg_replace("/[^A-Za-z0-9]/", "", $TrialName);
        $TrialName = strtolower($TrialName);

        # if this email address is very short, we'll pad it with some random
        # characters
        if (strlen($TrialName) < 2)
        {
            $TrialName .= GetRandomCharacters(2, "/[^a-hj-np-z0-9]/");
        }

        # see if the specified name exists
        $UFactory = new UserFactory();

        $Name = self::AppendSuffix($TrialName, '');

        while ($UFactory->UserNameExists($Name))
        {
            $Name = self::AppendSuffix(
                $TrialName, GetRandomCharacters(2) );
        }

        return $Name;
    }

    /**
    * Take a per-user list of resources and filter out the non-viewable ones.
    * @param array $ResourcesPerUser Array keyed by UserId giving a list of ResourceIds.
    * @return Array keyed by UserId giving the resources visible to
    * each user in the input.
    */
    public static function FilterNonViewableResourcesFromPerUserLists($ResourcesPerUser)
    {
        $Result = [];

        $RFactories = [];
        foreach ($ResourcesPerUser as $UserId => $ResourceList)
        {
            $User = new CWUser($UserId);
            $UserResources = [];

            $ResourcesPerSchema = ResourceFactory::BuildMultiSchemaResourceList(
                $ResourceList);
            foreach ($ResourcesPerSchema as $SchemaId => $ResourceIds)
            {
                if (!isset($RFactories[$SchemaId]))
                {
                    $RFactories[$SchemaId] = new ResourceFactory($SchemaId);
                }

                $VisibleResources = $RFactories[$SchemaId]->FilterNonViewableResources(
                    $ResourceIds, $User);

                if (count($VisibleResources))
                {
                    $UserResources[$SchemaId] = $VisibleResources;
                }
            }

            if (count($UserResources))
            {
                $Result[$UserId] = ResourceFactory::FlattenMultiSchemaResourceList(
                    $UserResources);
            }
        }

        return $Result;
    }


    # ---- OVERRIDDEN METHODS ------------------------------------------------

    /**
    * Create a new user. The second password and e-mail address parameters are
    * intended for second copies of each entered by the user.
    * @param string $UserName Login name for new user.
    * @param string $Password Password for new user.
    * @param string $PasswordAgain Second copy of password entered by user.
    * @param string $EMail E-mail address for new user.
    * @param string $EMailAgain Second copy of e-mail address entered by user.
    * @param bool $IgnoreErrorCodes Array containing any error codes that should
    *       be ignored.  (OPTIONAL)
    * @return CWUser object or array of error codes.
    */
    public function CreateNewUser(
            $UserName, $Password, $PasswordAgain, $EMail, $EMailAgain,
            $IgnoreErrorCodes = NULL)
    {
        # add the user to the APUsers table
        $User = parent::CreateNewUser(
            $UserName,
            $Password,
            $PasswordAgain,
            $EMail,
            $EMailAgain,
            $IgnoreErrorCodes);

        # user account creation did not succeed, so return the error codes
        if (!($User instanceof User))
        {
            return $User;
        }

        # create the user resource
        $Resource = Resource::Create(MetadataSchema::SCHEMAID_USER);

        # set the user ID for the resource
        $Resource->Set("UserId", $User->Id());

        $Schema = new MetadataSchema(MetadataSchema::SCHEMAID_USER);
        $TimestampFields = $Schema->GetFields(MetadataSchema::MDFTYPE_TIMESTAMP);

        # update timestamps as required
        foreach ($TimestampFields as $Field)
        {
            if ($Field->UpdateMethod()
                == MetadataField::UPDATEMETHOD_ONRECORDCREATE)
            {
                $Resource->Set($Field, "now");
            }
        }

        # make the user resource permanent
        $Resource->IsTempResource(FALSE);

        # get the CWUser object for the user
        $CWUser = new CWUser(intval($User->Id()));

        # couldn't get the CWUser object
        if ($CWUser->Status() != U_OKAY)
        {
            return array($CWUser->Status());
        }

        # set up initial UI setting
        $CWUser->Set("ActiveUI",
            $GLOBALS["G_SysConfig"]->DefaultActiveUI());

        # set up initial privileges
        foreach ($GLOBALS["G_SysConfig"]->DefaultUserPrivs() as $Privilege)
        {
            $CWUser->GivePriv($Privilege);
        }

        # return new user object to caller
        return $CWUser;
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    /**
    * Append a suffix to a string (typically a username), truncating
    *   to keep the total length less than 24 characters.
    * @param string $TrialName Base string.
    * @param string $Suffix Suffix to append.
    * @param int $MaxLength Maximum length of the result (OPTIONAL,
    *   default 24 matching IsValidUsername() from Axis--User)
    * @return string Constructed string.
    */
    private static function AppendSuffix($TrialName, $Suffix, $MaxLength=24)
    {
        if (strlen($TrialName.$Suffix)>$MaxLength)
        {
            $TrialName = substr(
                $TrialName, 0, $MaxLength - strlen($Suffix));
        }

        return $TrialName.$Suffix;
    }

    /**
    * The resource factory for user resources.
    */
    protected $ResourceFactory;

    private static $LowMemoryThresh = 0.25;
}
