<?PHP
#
#   FILE:  PieChart.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Class for generating and displaying a pie chart.
*/
class PieChart extends Chart_Base
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Class constructor.
    * @param array $Data Data for chart, with labels for the index.
    */
    public function __construct($Data)
    {
        $this->Data = $Data;
    }

    /**
    * Set the precision used to display percentages.
    * @param int $Prec Number of digits to display after the decimal.
    */
    public function PercentPrecision($Prec)
    {
        $this->Precision = $Prec;
    }

    /**
    * Set the style for slice labels.
    * @param string $LabelType Label type as a PieChart::LABEL_
    * constant. LABEL_PERCENT will display percentages, LABEL_NAME
    * will display slice names, and LABEL_RAW will display the raw
    * data.
    * @throws Exception If an invalid slice label type is supplied.
    */
    public function SliceLabelType($LabelType)
    {
        if (!in_array($LabelType,
                      [static::LABEL_PERCENT, static::LABEL_NAME, static::LABEL_RAW]))
        {
            throw new Exception("Unsupported slice label type: ".$LabelType);
        }

        $this->SliceLabelType = $LabelType;
    }

    # label type constants
    const LABEL_PERCENT = "Percent";
    const LABEL_RAW = "Raw";
    const LABEL_NAME = "Name";

    # ---- PRIVATE INTERFACE --------------------------------------------------

    /**
    * Prepare data for display. @see ChartBase::PrepareData().
    */
    protected function PrepareData()
    {
        $this->Chart["data"]["columns"] = [];
        foreach ($this->Data as $Label => $Value)
        {
            if (isset($this->LegendLabels[$Label]))
            {
                $MyLabel = $this->LegendLabels[$Label];
                $this->LabelLUT[$MyLabel] = $Label;
            }
            else
            {
                $MyLabel = $Label;
            }

            $this->Chart["data"]["columns"][]= [$MyLabel, $Value];
        }

        $this->AddToChart([
            "data" => [
                "type" => "pie",
            ],
            "pie" => [
                "label" => [
                    "format" => "label_format_fn",
                ],
            ],
        ]);
    }

    /**
    * Output javascript that declares helper functions used to display
    * the chart.
    */
    protected function DeclareHelperFunctions()
    {
        // @codingStandardsIgnoreStart
        ?>
        function tooltip_value_fn(value, ratio, id, index) {
            return value + " ("+(new Number(100*ratio)).toFixed(<?= $this->Precision ?>)+"%)";
        }

        function label_format_fn(value, ratio, id, index) {
            <?PHP if ($this->SliceLabelType == self::LABEL_PERCENT) { ?>
            return (new Number(100*ratio)).toFixed(<?= $this->Precision ?>)+"%";
            <?PHP } elseif ($this->SliceLabelType == self::LABEL_RAW){ ?>
            return value;
            <?PHP } elseif ($this->SliceLabelType == self::LABEL_NAME){ ?>
            return id;
            <?PHP } ?>
        }
        <?PHP
        // @codingStandardsIgnoreEnd
    }

    private $SliceLabelType = self::LABEL_PERCENT;
    private $Precision = 1;
}
