<?PHP
#
#   FILE:  ControlledName.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2001-2016 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Metadata type representing non-hierarchical controlled vocabulary values.
* Hierarchical controlled vocabularies should use Classification.
*/
class ControlledName extends Item
{

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Create a new empty ControlledName if it's not already present.
    * Caller should set other parameters after it's created.
    * If a controlledname with given name and field ID already
    *       exists, this will just return that controlledname.
    * @param string $Term New controlled vocabulary term.
    * @param int $FieldId ID of MetadataField for new term.
    * @return A new ControlledName just created or the one in
    *       the database, if the controlledname already exists.
    */
    public static function Create($Term, $FieldId)
    {
        $DB = new Database();

        # check if this controlledname is already present
        $DB->Query("SELECT * FROM ControlledNames".
                " WHERE ControlledName = '".addslashes($Term).
                "' AND FieldId = ".intval($FieldId));
        if ($Row = $DB->FetchRow())
        {
            $NameId = $Row["ControlledNameId"];
        }
        else
        {
            # add new controlled name
            $DB->Query("INSERT INTO ControlledNames ".
                    "(FieldId, ControlledName) VALUES (".
                    intval($FieldId).", '".addslashes($Term)."')");

            $NameId = $DB->LastInsertId();
        }
        # instantiate new controlledname and return
        $NewCN = new ControlledName(intval($NameId));
        return $NewCN;
    }

    /**
    * Check if there exists a controlledname with a ControlledName
    *       and FieldId same as given. This method is different
    *       from ItemExists(), which does check base on Id.
    * @param string $Term ControlledName of the controlledname.
    * @param string $FieldId ID of the MetadataField.
    * @return bool True if there exists such controlledname,
    *       false otherwise.
    */
    public static function ControlledNameExists($Term, $FieldId)
    {
        $DB = new Database();

        $DB->Query("SELECT * FROM ControlledNames".
                " WHERE ControlledName = '".addslashes($Term).
                "' AND FieldId = ".intval($FieldId));
        return $DB->NumRowsSelected();
    }


    /**
    * Get or set the controlled vocabulary term.
    * @param string $NewValue New value for term.  (OPTIONAL)
    * @return Controlled vocabulary term.
    */
    public function Name($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("ControlledName", $NewValue);
    }

    /**
    * Get, set, or clear any variant terms for this controlled name .
    * @param string or bool $NewValue New value for variant terms. (OPTIONAL)
    *       Pass no argument to just retrieve current variant name.
    *       Pass FALSE to unset any variant name attached.
    * @return Return the current variant name.
    *       If this ControlledName is not attached to any variant name,
    *       NULL will be returned.
    */
    public function VariantName($NewValue = DB_NOVALUE)
    {
        # unset any variant name attached if asked
        if ($NewValue === FALSE)
        {
            $this->DB->Query("DELETE FROM VariantNames WHERE "
                    ."ControlledNameId = ".$this->Id);
            $this->VNCache = NULL;
        }

        else if ($NewValue != DB_NOVALUE)
        {
            $this->VNCache = $NewValue;

            # try to load variant name and update cache
            $this->DB->Query("SELECT VariantName FROM VariantNames WHERE "
                    ."ControlledNameId = ".$this->Id);

            # variant name exists so do an update
            if ($this->DB->NumRowsSelected() > 0)
            {
                $this->DB->Query("UPDATE VariantNames SET VariantName = '"
                        .addslashes($NewValue)."' WHERE ControlledNameId = "
                        .$this->Id);
            }
            # no variant name so do an insert
            else
            {
                $this->DB->Query("INSERT INTO VariantNames ".
                        "(VariantName, ControlledNameId) VALUES ".
                        "('".addslashes($NewValue)."', ".$this->Id.")");
            }
        }

        # if variant name is not cached
        else if (!isset($this->VNCache))
        {
            $this->VNCache = $this->DB->Query("SELECT VariantName FROM VariantNames".
                   " WHERE ControlledNameId = ".$this->Id, "VariantName");
        }

        return $this->VNCache;
    }

    /**
    * Get or set the MetadataField associated with this term.
    * @param int $NewValue ID of new MetadataField.  (OPTIONAL)
    * @return ID of associated MetadataField.
    */
    public function FieldId($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("FieldId", $NewValue);
    }

    /**
    * Get or set the Qualifier associated with this term via ID.
    * @param int $NewValue ID of new Qualifier.  (OPTIONAL)
    * @return ID of currently associated Qualifier.
    */
    public function QualifierId($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("QualifierId", $NewValue);
    }

    /**
    * Get or set the Qualifier associated with this term via object.
    * @param Qualifier $NewValue New Qualifier.  (OPTIONAL)
    * @return Currently associated Qualifier.
    */
    public function Qualifier($NewValue = DB_NOVALUE)
    {
        # if new qualifier supplied
        if ($NewValue !== DB_NOVALUE)
        {
            # set new qualifier ID
            $this->QualifierId($NewValue->Id());

            # use new qualifier for return value
            $Qualifier = $NewValue;
        }
        else
        {
            # if qualifier is available
            if ($this->QualifierId() !== NULL
                    && Qualifier::ItemExists($this->QualifierId()))
            {
                # create qualifier object using stored ID
                $Qualifier = new Qualifier($this->QualifierId());

                # if ID was zero and no name available for qualifieR
                # (needed because some controlled name records in DB
                #       have 0 instead of NULL when no controlled name assigned)
                # (NOTE:  this is problematic if there is a qualifier with an
                #       ID value of 0!!!)
                if ($this->QualifierId() == 0 && !strlen($Qualifier->Name()))
                {
                    # return NULL to indicate no qualifier
                    $Qualifier = NULL;
                }
            }
            else
            {
                # return NULL to indicate no qualifier
                $Qualifier = NULL;
            }
        }

        # return qualifier to caller
        return $Qualifier;
    }

    /**
    * Check if the given controlled name already exists for a given field ID.
    * @param string $ControlledName Name to search for.
    * @param int $FieldId ID of field to search.
    * @return array Matching controlled name IDs, if any.
    */
    static public function SearchForControlledName($ControlledName, $FieldId)
    {
        $Database = new Database();

        # query for the controlled name
        $Database->Query("
            SELECT ControlledNameId FROM
            ControlledNames WHERE FieldId='".addslashes($FieldId)."'
            AND ControlledName='".addslashes($ControlledName)."'");

        # return the controlled name IDs found, if any
        return $Database->FetchColumn("ControlledNameId");
    }

    /**
    * See if ControlledName is currently associated with any Resources.
    * @return TRUE if associated with at least one Resource, otherwise FALSE.
    */
    public function InUse()
    {
        return $this->DB->Query("SELECT COUNT(*) AS Count FROM ".
                "ResourceNameInts WHERE ControlledNameId = ".$this->Id, "Count");
    }

    /**
    * Get resourceIds associated with this ControlledName.
    * @return array of ResourceIds.
    */
    public function GetAssociatedResources()
    {
        $this->DB->Query(
            "SELECT ResourceId FROM ResourceNameInts "
            ."WHERE ControlledNameId = ".$this->Id);

        return $this->DB->FetchColumn("ResourceId");
    }

    /**
    * Change all currently associated Resources to be instead associated with
    * another ControlledName.
    * @param int $NewNameId ID of ControlledName to remap resources to.
    */
    public function RemapTo($NewNameId)
    {
        # Get a list of resources associated with the new name
        $this->DB->Query("SELECT ResourceId FROM ResourceNameInts "
                         ."WHERE ControlledNameId = ".intval($NewNameId));
        $NewNameResources = array();
        while ($Row = $this->DB->FetchRow())
        {
            $NewNameResources[$Row["ResourceId"]]=1;
        }

        # Get a list of resources associated with the old name
        $this->DB->Query("SELECT ResourceId FROM ResourceNameInts "
                         ."WHERE ControlledNameId = ".intval($this->Id));
        $OldNameResources = array();
        while ($Row = $this->DB->FetchRow())
        {
            $OldNameResources[]= $Row["ResourceId"];
        }

        # Foreach of the old name resources, check to see if it's already
        # associated with the new name.  If not, associate it.
        foreach ($OldNameResources as $ResourceId)
        {
            if (!isset($NewNameResources[$ResourceId]))
            {
                $this->DB->Query("INSERT INTO ResourceNameInts "
                                 ."(ResourceId, ControlledNameId) VALUES "
                                 ."(".intval($ResourceId).",".intval($NewNameId).")");
            }
        }

        # Clear out all the associations to the old name
        $this->DB->Query("DELETE FROM ResourceNameInts WHERE ControlledNameId = "
                .intval($this->Id));
    }

    /**
    * Update the LastAssigned timestamp for this classification.
    */
    public function UpdateLastAssigned()
    {
        $this->DB->Query("UPDATE ControlledNames SET LastAssigned=NOW() "
                         ."WHERE ControlledNameId=".intval($this->Id));
    }

    /**
    * Remove ControlledName (and any accompanying associations from database.
    * This must be the last use of this object.
    * @param bool $DeleteIfHasResources Remove ControlledName even if Resources are
    *       currently associated with it.  (OPTIONAL, defaults to FALSE)
    */
    public function Delete($DeleteIfHasResources = FALSE)
    {
        $DB = &$this->DB;

        if ($DeleteIfHasResources || !$this->InUse())
        {
            # delete this controlled name
            $this->Destroy();

            # delete associated variant name
            $DB->Query("DELETE FROM VariantNames WHERE ControlledNameId=".
                $this->Id);

            if ($DeleteIfHasResources)
            {
                $DB->Query("DELETE FROM ResourceNameInts WHERE ".
                   "ControlledNameId=".$this->Id);
            }
        }
    }

    private $VNCache;
}
