<?PHP
#
#   FILE:  AddControlledName.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2011-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

# ----- LOCAL FUNCTIONS ------------------------------------------------------

/**
* Get the controlled name metadata field to use.
* @param MetadataSchema $Schema The schema from which to get the field.
* @return Returns a metadata field or NULL if there are no controlled name fields.
*/
function GetField(MetadataSchema $Schema)
{
    $FieldId = intval(GetFormValue("F_FieldId", GetFormValue("FieldId")));

    if ($Schema->FieldExists($FieldId))
    {
        return $Schema->GetField($FieldId);
    }

    $Fields = $Schema->GetFields(MetadataSchema::MDFTYPE_CONTROLLEDNAME);

    # there are controlled name fields
    if (count($Fields))
    {
        $Field = array_shift($Fields);
        return $Field;
    }

    return NULL;
}

/**
* Attempt to add a controlled name.
* @param MetadataSchema $Schema The schema to which the field belongs.
* @param MetadataField $Field The field to which to add the controlled name.
* @param string $ControlledName The new controlled name.
* @param string $Variants Variants for the controlled name.
* @param string $Qualifier Qualifier for the controlled name.
* @return Returns NULL on success and an error message otherwise.
*/
function AddControlledName(
    MetadataSchema $Schema,
    MetadataField $Field,
    $ControlledName,
    $Variants,
    $Qualifier)
{
    # validate the controlled name
    if (!strlen(trim($ControlledName)))
    {
        return "The controlled name cannot be blank.";
    }

    # set "--" and NULL to "NULL"
    if ($Qualifier == "--" || is_null($Qualifier))
    {
        $Qualifier = "NULL";
    }

    # check if controlledname already exists, if not, create one
    if (ControlledName::ControlledNameExists($ControlledName, $Field->Id()))
    {
        return "The controlled name already exists.";
    }
    else
    {
        $CN = ControlledName::Create($ControlledName, $Field->Id());
        $CN->QualifierId($Qualifier);
        $CN->VariantName($Variants);
    }

}

# ----- MAIN -----------------------------------------------------------------

PageTitle("Add Controlled Name");

# check that the user is authorized to view the page
if (!CheckAuthorization(PRIV_NAMEADMIN)) {  return;  }

# get the schema and field
$H_Schema = new MetadataSchema(
  GetArrayValue($_GET, "SC", MetadataSchema::SCHEMAID_DEFAULT)
               );
$H_Field = GetField($H_Schema);

# form values
$H_ControlledName = GetFormValue("F_ControlledName");
$H_VariantName = GetFormValue("F_VariantName");
$H_Qualifier = GetFormValue("F_Qualifier");

# values used only when the form is submitted
$H_ErrorMessage = NULL;
$H_SuccessfullyAdded = FALSE;

# the form was submitted
if (GetFormValue("F_Submit") == "Add")
{
    $H_ErrorMessage = AddControlledName(
        $H_Schema,
        $H_Field,
        $H_ControlledName,
        $H_VariantName,
        $H_Qualifier);

    $H_SuccessfullyAdded = !$H_ErrorMessage;
}
