<?PHP
#
#   FILE:  Tags.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2011 Internet Scout Project
#   http://scout.wisc.edu/
#

class Tags extends Plugin
{
    /**
     * @const TAG_FIELD_NAME the name of the tag metadata field
     */
    const TAG_FIELD_NAME = "Tags";

    /**
     * @const TAG_FIELD_XML file name of the XML representation of the tag field
     */
    const TAG_FIELD_XML = "Tags_Tag.xml";

    /**
     * @const DEFAULT_TAGS_CSV CSV file with the default tags
     */
    const DEFAULT_TAGS_CSV = "DefaultTags.csv";

    /**
     * Register information about this plugin.
     */
    public function Register()
    {
        $this->Name = "Tags";
        $this->Version = "1.1.1";
        $this->Description = "Adds tag metadata support to resources.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array("CWISCore" => "2.4.1");
        $this->EnabledByDefault = FALSE;
    }

    /**
    * Upgrade from a previous version.
    * @param string $PreviousVersion Previous version of the plugin.
    * @return Returns NULL on success and an error message otherwise.
    */
    public function Upgrade($PreviousVersion)
    {
        # upgrade from version < 1.1.1 to 1.1.1
        if (version_compare($PreviousVersion, "1.1.1", "<"))
        {
            # update or add the tag metadata field name
            $Schema = new MetadataSchema();
            if ($Schema->FieldExists("Tags (Plugin)"))
            {
                $Field = $Schema->GetFieldByName("Tags (Plugin)");
                $Field->Name(self::TAG_FIELD_NAME);
            }
            else
            {
                $Schema->AddFieldFromXml(self::TAG_FIELD_XML);
            }
        }
        return NULL;
    }

    /**
     * Install necessary SQL tables.
     * @return NULL on success or error message on error
     */
    public function Install()
    {
        # try to get the field
        $Field = $this->GetTagField();

        # try to create it if it doesn't exist
        if (!($Field instanceof MetadataField))
        {
            # create the tag field from its XML representation
            $Schema = new MetadataSchema();
            $Xml = $this->GetTagXml();
            $Field = $Schema->AddFieldFromXml($Xml);

            if (!($Field instanceof MetadataField))
            {
                return "Could not create the tag metadata field.";
            }
        }

        # get the file that holds the default tags
        $DefaultTagsFile = @fopen($this->GetDefaultTagsFile(), "r");

        if ($DefaultTagsFile === FALSE)
        {
            return "Could not prepopulate the tag metadata field.";
        }

        # get the tags
        $Tags = @fgetcsv($DefaultTagsFile);

        if ($Tags === FALSE)
        {
            return "Could not parse the default tags";
        }

        # add each tag
        foreach ($Tags as $Tag)
        {
            $ControlledName = ControlledName::Create($Tag, $Field->Id());
        }

        # close the default tag file
        @fclose($DefaultTagsFile);

        return NULL;
    }

    /**
     * Uninstall the plugin.
     * @return NULL|string NULL if successful or an error message otherwise
     */
    public function Uninstall()
    {
        # remove the tag field
        $Field = $this->GetTagField();
        $Schema = new MetadataSchema($Field->SchemaId());
        $Schema->DropField($Field->Id());

        # report success to caller
        return NULL;
    }

    /**
    * Hook the events into the application framework.
    * @return an array of events to be hooked into the application framework
    */
    public function HookEvents()
    {
        return array(
            "EVENT_APPEND_HTML_TO_FIELD_DISPLAY" => "AppendToFieldDisplay",
            "EVENT_FIELD_DISPLAY_FILTER" => "EditTagValue");
    }

    /**
    * Append HTML to field display to allow adding tags on-the-fly.
    * @param MetadataField $Field MetadataField object
    * @param Resource $Resource Resource object
    * @param string $Context Context in which the field is displayed
    * @param string $Html Current HTML to display with the field
     */
    public function AppendToFieldDisplay($Field, $Resource, $Context, $Html)
    {
        # bail out without making changes if passed NULL values
        if (is_null($Field) || is_null($Resource))
        {
            return array(
                "Field" => $Field,
                "Resource" => $Resource,
                "Context" => $Context,
                "Html" => $Html);
        }

        $TagField = $this->GetTagField();
        $IsTag = $Field->Name() == $TagField->Name();

        if ($IsTag && $Context == "DISPLAY")
        {
            if ($Resource->UserCanEditField($GLOBALS["G_User"], $Field))
            {
                $EditResourcePage = $GLOBALS["AF"]->BaseUrl().
                    "index.php?P=EditResource&ID=".$Resource->Id();

                $Html = "<td><a href=".$EditResourcePage." class='cw-button ".
                    "cw-button-elegant cw-button-constrained'>Edit Tags</a></td>";
            }
        }

        return array(
            "Field" => $Field,
            "Resource" => $Resource,
            "Context" => $Context,
            "Html" => $Html);
    }

    /**
    * Assign a non-null value so that Tags will correctly display in record pages.
    * @param MetadataField $Field MetadataField object
    * @param Resource $Resource Resource object
    * @param mixed $Value Tag value. Empty array if no tags are associated with
            resource. Otherwise, array of ControlledName objects.
    */
    public function EditTagValue($Field, $Resource, $Value)
    {
        # bail out without making changes if passed NULL values
        if (is_null($Field) || is_null($Resource))
        {
            return array(
                "Field" => $Field,
                "Resource" => $Resource,
                "Value" => $Value);
        }

        $TagField = $this->GetTagField();
        $IsTag = $Field->Name() == $TagField->Name();
        $User = $GLOBALS["G_User"];

        if ($IsTag)
        {
            # if the user has privs to edit tags, we assign a non-null value to
            # display so that the Tags row will display with an "Edit Tags" button
            if ($Resource->UserCanModify($User) && empty($Value))
            {
                $Value = "<!-- Empty Tags --> ";
            }
        }

        return array(
            "Field" => $Field,
            "Resource" => $Resource,
            "Value" => $Value);
    }

    /**
    * Get the metadata field object for the tag field.
    * @return MetadataField object for the tag field
    */
    protected function GetTagField()
    {
        $Schema = new MetadataSchema();
        $Field = $Schema->GetFieldByName(self::TAG_FIELD_NAME);

        return $Field;
    }

    /**
    * Get the XML representation of the tag metadata field as a string.
    * @return XML representation of the tag metadata field as a string
    */
    protected function GetTagXml()
    {
        $Path = dirname(__FILE__) . "/install/" . self::TAG_FIELD_XML;
        $Xml = file_get_contents($Path);

        return $Xml;
    }

    /**
    * Get the path to the default tags file.
    * @return string path to the default tags file
    */
    protected function GetDefaultTagsFile()
    {
        $Path = dirname(__FILE__) . "/install/" . self::DEFAULT_TAGS_CSV;

        return $Path;
    }
}
