<?PHP
#
#   FILE:  MetadataFieldOrderingComplete.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2012 Internet Scout Project
#   http://scout.wisc.edu/
#

# ----- LOCAL FUNCTIONS ------------------------------------------------------

/**
 * Add a new group with the given name and for the given metadata field
 * ordering.
 * @param string $Name new group name
 * @param string $Ordering metadata field ordering (Display or Edit)
 * @return void
 */
function AddGroup($Name, $Ordering)
{
    try
    {
        if ($Ordering == "Edit")
        {
            $Order = MetadataFieldOrder::GetEditOrderObject();
        }

        else
        {
            $Order = MetadataFieldOrder::GetDisplayOrderObject();
        }

        $Order->CreateGroup($Name);
    }

    catch (Exception $Exception){}
}

/**
 * Delete the metadata field group with the given ID from the metadata field
 * order with the given ID.
 * @param int $OrderId metadata field order ID
 * @param int $GroupId metadata field group ID
 * @return void
 */
function DeleteGroup($OrderId, $GroupId)
{
    try
    {
        $Order = new MetadataFieldOrder($OrderId);
        $Group = new MetadataFieldGroup($GroupId);

        $Order->DeleteGroup($Group);
    }

    catch (Exception $Exception){}
}

/**
 * Modify the name of the group with the given ID.
 * @param int $Id group ID
 * @param string $Name new group name
 * @return void
 */
function EditGroupName($Id, $Name)
{
    try
    {
        $Group = new MetadataFieldGroup($Id);
        $Group->Name($Name);
    }

    catch (Exception $Exception){}
}

/**
 * Move up the item with the given ID and type in the order with the given ID.
 * @param int $OrderId order ID
 * @param int $ItemId item ID
 * @param string $ItemType item type
 * @return void
 */
function MoveItemUp($OrderId, $ItemId, $ItemType)
{
    # need to be working with a metadata field or metadata group
    if (!IsValidItemType($ItemType))
    {
        return;
    }

    try
    {
        $Order = new MetadataFieldOrder($OrderId);
        $Item = new $ItemType($ItemId);

        $Order->MoveItemUp($Item, "ItemFilter");
    } catch (Exception $Exception){}
}

/**
 * Move up the item with the given ID and type in the order with the given ID.
 * @param int $OrderId order ID
 * @param int $ItemId item ID
 * @param string $ItemType item type
 * @return void
 */
function MoveItemDown($OrderId, $ItemId, $ItemType)
{
    # need to be working with a metadata field or metadata group
    if (!IsValidItemType($ItemType))
    {
        return;
    }

    try
    {
        $Order = new MetadataFieldOrder($OrderId);
        $Item = new $ItemType($ItemId);

        $Order->MoveItemDown($Item, "ItemFilter");
    } catch (Exception $Exception){}
}

/**
 * Move the item with the given ID and type in the order with the given ID after
 * the item with the given target ID and type.
 * @param int $OrderId order ID
 * @param int $ItemId item ID
 * @param string $ItemType item type
 * @param int $TargetId target item ID
 * @param string $TargetType target item type
 * @return void
 */
function MoveItemAfter($OrderId, $ItemId, $ItemType, $TargetId, $TargetType)
{
    # need to be working with metadata fields and metadata field groups
    if (!IsValidItemType($ItemType) || !IsValidItemType($TargetType))
    {
        return;
    }

    try
    {
        $Order = new MetadataFieldOrder($OrderId);
        $Item = new $ItemType($ItemId);
        $Target = new $TargetType($TargetId);

        $Order->MoveItemAfter($Target, $Item);
    } catch (Exception $Exception){}
}

/**
 * Move the item with the given ID and type to the top of the order with the
 * given ID.
 * @param int $OrderId order ID
 * @param int $ItemId item ID
 * @param string $ItemType item type
 * @return void
 */
function MoveItemToTop($OrderId, $ItemId, $ItemType)
{
    # need to be working with a metadata field or metadata field group
    if (!IsValidItemType($ItemType))
    {
        return;
    }

    try
    {
        $Order = new MetadataFieldOrder($OrderId);
        $Item = new $ItemType($ItemId);

        $Order->MoveItemToTop($Item);
    } catch (Exception $Exception){}
}

/**
 * Move the field with the given ID to the top of the group with the given ID
 * within the order with the given ID.
 * @param int $OrderId order ID
 * @param int $GroupId group ID
 * @param int $FieldId field ID
 * @return void
 */
function MoveFieldToTopOfGroup($OrderId, $GroupId, $FieldId)
{
    try
    {
        $Order = new MetadataFieldOrder($OrderId);
        $Group = new MetadataFieldGroup($GroupId);
        $Field = new MetadataField($FieldId);

        $Order->MoveFieldToTopOfGroup($Group, $Field);
    } catch (Exception $Exception){}
}

/**
 * Filter used to remove some disabled fields from the order when moving a field
 * up or down.
 * @param array $Item item ID and type (keys are "ID" and "Type")
 * @param bool TRUE to keep the item or FALSE to filter it out
 */
function ItemFilter($Item)
{
    if ($Item["Type"] == "MetadataField")
    {
        $Field = new MetadataField($Item["ID"]);

        if (!$Field->Enabled())
        {
            return FALSE;
        }
    }

    return TRUE;
}

/**
 * Extract the given values from the POST variables.
 * @param array $Keys array of keys of the values to extract
 * @param array $Defaults array of default values used if a value doesn't exist
 * @return array extracted values
 */
function GetOrderValues(array $Keys, array $Defaults=array())
{
    return GetArrayValues($_POST, $Keys, $Defaults);
}

/**
 * Determine if the given item type is a valid one.
 * @param string $Type item type
 * @return bool TRUE if the item type is valid or FALSE otherwise
 */
function IsValidItemType($Type)
{
    return $Type == "MetadataField" || $Type == "MetadataFieldGroup";
}

# ----- MAIN -----------------------------------------------------------------

CheckAuthorization(PRIV_SYSADMIN, PRIV_COLLECTIONADMIN);

$AF->SuppressHTMLOutput();

# only redirect if a response is expected
if (GetArrayValue($_GET, "SendResponse", TRUE))
{
    $AF->SetJumpToPage("MetadataFieldOrdering");
}

$Submit = GetArrayValue($_POST, "F_Submit");

if ($Submit == "AddGroup")
{
    # get values from the user
    extract(GetOrderValues(array(
        "F_GroupName",    # $F_GroupName
        "F_Ordering")));  # $F_Ordering

    if ($F_GroupName && ($F_Ordering == "Display" || $F_Ordering == "Edit"))
    {
        AddGroup($F_GroupName, $F_Ordering);
    }
}

else if ($Submit == "DeleteGroup")
{
    # get values from the user
    extract(GetOrderValues(array(
        "F_OrderId",     # $F_OrderId
        "F_GroupId")));  # $F_GroupId

    if ($F_OrderId && $F_GroupId)
    {
        DeleteGroup($F_OrderId, $F_GroupId);
    }
}

else if ($Submit == "EditGroupName")
{
    # get values from the user
    extract(GetOrderValues(array(
        "F_GroupId",       # $F_GroupId
        "F_GroupName")));  # $F_GroupName

    if ($F_GroupId && $F_GroupName)
    {
        EditGroupName($F_GroupId, $F_GroupName);
    }
}

else if ($Submit == "MoveUp")
{
    # get values from the user
    extract(GetOrderValues(array(
        "F_OrderId",          # $F_OrderId
        "F_ItemId",           # $F_ItemId
        "F_ItemType")));      # $F_ItemType

    MoveItemUp($F_OrderId, $F_ItemId, $F_ItemType);
}

else if ($Submit == "MoveDown")
{
    # get values from the user
    extract(GetOrderValues(array(
        "F_OrderId",          # $F_OrderId
        "F_ItemId",           # $F_ItemId
        "F_ItemType")));      # $F_ItemType

    MoveItemDown($F_OrderId, $F_ItemId, $F_ItemType);
}

else if ($Submit == "MoveAfter")
{
    # get values from the user
    extract(GetOrderValues(array(
        "F_OrderId",        # $F_OrderId
        "F_ItemId",         # $F_ItemId
        "F_ItemType",       # $F_ItemType
        "F_TargetId",       # $F_TargetId
        "F_TargetType")));  # $F_TargetType

    MoveItemAfter(
        $F_OrderId,
        $F_ItemId,
        $F_ItemType,
        $F_TargetId,
        $F_TargetType);
}

else if ($Submit == "MoveItemToTop")
{
    # get values from the user
    extract(GetOrderValues(array(
        "F_OrderId",      # $F_OrderId
        "F_ItemId",       # $F_ItemId
        "F_ItemType")));  # $F_ItemType

    MoveItemToTop(
        $F_OrderId,
        $F_ItemId,
        $F_ItemType);
}

else if ($Submit == "MoveFieldToTopOfGroup")
{
    # get values from the user
    extract(GetOrderValues(array(
        "F_OrderId",     # $F_OrderId
        "F_GroupId",     # $F_GroupId
        "F_FieldId")));  # $F_FieldId

    MoveFieldToTopOfGroup(
        $F_OrderId,
        $F_GroupId,
        $F_FieldId);
}
