<?PHP

#
#   FILE:  ControlledName.php
#
#   Part of the Collection Workflow Integration System
#   Copyright 2001-2009 Edward Almasy and Internet Scout
#   http://scout.wisc.edu
#

/**
* Metadata type representing non-hierarchical controlled vocabulary values.
* Hierarchical controlled vocabularies should use Classification.
*/
class ControlledName {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /** @name Status Codes (set by constructor and retrieved via Status()) */ /*@{*/
    /** Successful execution. */
    const STATUS_OK = 0;
    /** No ControlledName exists with specified ID. */
    const STATUS_INVALID_ID = 1;
    /** ControlledName already exists with this term. */
    const STATUS_EXISTS = 2;
    /*@}*/

    /**
    * Class constructor.  This can be used both to access an existing
    * controlled vocabulary term or to add a new term.  For existing
    * terms pass in just the ControlledName ID.  To create a new term
    * pass in NULL for the ID, and specify the term, the metadata field
    * ID, and the qualifier ID (if any) and variant name (if any).
    * @param NameId ControlledName ID (NULL if creating new).
    * @param Name New controlled vocabulary term.  (OPTIONAL)
    * @param FieldId ID of MetadataField for new term.  (OPTIONAL)
    * @param QualifierId ID of Qualifier for new term (if any).  (OPTIONAL)
    * @param VariantName Variant names for new term (if any).  (OPTIONAL)
    */
    function ControlledName($NameId, $Name = NULL, $FieldId = NULL,
                    $QualifierId = "NULL", $VariantName = NULL)
    {
        # assume everything will turn out okay
        $this->ErrorStatus = self::STATUS_OK;

        # create DB handle for our use
        $this->DB = new SPTDatabase();
        $DB =& $this->DB;

        # remove whitespace padding
        $Name = trim($Name);
        $VariantName = trim($VariantName);

        # look for passed in name and type
        if (!empty($Name) && !empty($FieldId))
        {
            $DB->Query("SELECT * FROM ControlledNames".
                   " WHERE ControlledName = \"".addslashes($Name)."\"".
                   " AND FieldId = ".intval($FieldId));

            while ($this->DBFields = $DB->FetchRow())
            {
                # this controlled name already exists
                if ($this->DBFields["ControlledName"] == $Name)
                {
                    $this->ErrorStatus = self::STATUS_EXISTS;
                    $NameId = $this->DBFields["ControlledNameId"];

                    # cache the variant name separately
                    $VN = $DB->Query("SELECT VariantName FROM VariantNames".
                        " WHERE ControlledNameId = ".
                        $this->DBFields["ControlledNameId"], "VariantName");

                    $this->DBFields["VariantName"] = $VN;
                    break;
                }
            }
            # controlled name not found, create it
            if ($this->ErrorStatus == self::STATUS_OK)
            {
                # add new controlled name
                $DB->Query("INSERT INTO ControlledNames ".
                        "(FieldId, ControlledName, QualifierId)".
                        " VALUES (".intval($FieldId).", '".addslashes($Name)
                        ."', ".intval($QualifierId).")");

                # get name ID  for new controlled name
                $NameId = $DB->LastInsertId("ControlledNames");

                # check for Variant
                if (!empty($VariantName))
                {
                    $DB->Query("INSERT INTO VariantNames ".
                        "(ControlledNameId, VariantName) ".
                        "VALUES (".intval($NameId).", '"
                        .addslashes($VariantName)."') ");
                }
            }
        }
        # Name Id passed in, look it up
        if (!empty($NameId) && $NameId != -1)
        {
            $DB->Query("SELECT * FROM ControlledNames".
                   " WHERE ControlledNameId = ".intval($NameId));
            $this->DBFields = $DB->FetchRow();

            # cache the variant name separately
            $VN = $DB->Query("SELECT VariantName FROM VariantNames".
                   " WHERE ControlledNameId = ".intval($NameId), "VariantName");

            $this->DBFields["VariantName"] = $VN;
        }

        # save supplied or generated controlled name ID
        $this->Id = intval($NameId);

        # set error status if controlled name info not loaded
        if ($this->DBFields["ControlledNameId"] != $this->Id)
        {
            $this->ErrorStatus = self::STATUS_INVALID_ID;
        }
    }

    /**
    * Check success of constructor.
    * @return Status code for constructor.
    */
    function Status()   {  return $this->ErrorStatus;  }

    /**
    * Get ID.
    * @return ControlledName ID.
    */
    function Id()       {  return $this->Id;  }

    /**
    * Get or set the controlled vocabulary term.
    * @param NewValue New value for term.  (OPTIONAL)
    * @return Controlled vocabulary term.
    */
    function Name($NewValue = DB_NOVALUE)
            {  return $this->UpdateValue("ControlledName", $NewValue);  }

    /**
    * Get or set any variant terms for this controlled name .
    * @param NewValue New value for variant terms.  (OPTIONAL)
    * @return Variant term.
    */
    function VariantName($NewValue = DB_NOVALUE)
            {  return $this->UpdateVariantValue("VariantName", $NewValue);  }

    /**
    * Get or set the MetadataField associated with this term.
    * @param NewValue ID of new MetadataField.  (OPTIONAL)
    * @return ID of associated MetadataField.
    */
    function FieldId($NewValue = DB_NOVALUE)
            {  return $this->UpdateValue("FieldId", $NewValue);  }

    /**
    * Get or set the Qualifier associated with this term via ID.
    * @param NewValue ID of new Qualifier.  (OPTIONAL)
    * @return ID of currently associated Qualifier.
    */
    function QualifierId($NewValue = DB_NOVALUE)
            {  return $this->UpdateValue("QualifierId", $NewValue);  }

    /**
    * Get or set ???
    * @param NewValue New value for ???.  (OPTIONAL)
    * @return ???.
    */
    function Variant($NewValue = DB_NOVALUE)
            {  return $this->VariantName($NewValue);  }

    /**
    * Get or set the Qualifier associated with this term via object.
    * @param NewValue New Qualifier.  (OPTIONAL)
    * @return Currently associated Qualifier.
    */
    function Qualifier($NewValue = DB_NOVALUE)
    {
        # if new qualifier supplied
        if ($NewValue !== DB_NOVALUE)
        {
            # set new qualifier ID
            $this->QualifierId($NewValue->Id());

            # use new qualifier for return value
            $Qualifier = $NewValue;
        }
        else
        {
            # if qualifier is available
            if ($this->QualifierId() !== NULL)
            {
                # create qualifier object using stored ID
                $Qualifier = new Qualifier($this->QualifierId());

                # if ID was zero and no name available for qualifieR
                # (needed because some controlled name records in DB
                #       have 0 instead of NULL when no controlled name assigned)
                # (NOTE:  this is problematic if there is a qualifier with an
                #       ID value of 0!!!)
                if (($this->QualifierId() == 0) && !strlen($Qualifier->Name()))
                {
                    # return NULL to indicate no qualifier
                    $Qualifier = NULL;
                }
            }
            else
            {
                # return NULL to indicate no qualifier
                $Qualifier = NULL;
            }
        }

        # return qualifier to caller
        return $Qualifier;
    }

    /**
     * Check if a controlled name already exists for a given field.
     * @return array of matching ControlledNameIds if found, empty array otherwise.
     * @return TRUE if the name exists, FALSE otherwise.
     */
    static function SearchForControlledName($ControlledName, $FieldId)
    {
        global $DB;
        $DB->Query("SELECT ControlledNameId FROM ".
                   "ControlledNames WHERE FieldId=".intval($FieldId).
                   " AND ControlledName='".addslashes($ControlledName)."'");

        $rc = array();
        while ($Row = $DB->FetchRow())
        {
            $rc []= $Row["ControlledNameId"];
        }
        return $rc;

    }
    /**
    * See if ControlledName is currently associated with any Resources.
    * @return TRUE if associated with at least one Resource, otherwise FALSE.
    */
    function InUse()
    {
        return $this->DB->Query("SELECT COUNT(*) AS Count FROM ".
                "ResourceNameInts WHERE ControlledNameId = ".$this->Id, "Count");
    }

    /**
    * Change all currently associated Resources to be instead associated with
    * another ControlledName.
    * @param NewNameId ID of ControlledName to remap resources to.
    */
    function RemapTo($NewNameId)
    {
        # Get a list of resources associated with the new name
        $this->DB->Query("SELECT ResourceId FROM ResourceNameInts "
                         ."WHERE ControlledNameId = ".intval($NewNameId));
        $NewNameResources = array();
        while ($Row = $this->DB->FetchRow())
        {
            $NewNameResources[$Row["ResourceId"]]=1;
        }

        # Get a list of resources associated with the old name
        $this->DB->Query("SELECT ResourceId FROM ResourceNameInts "
                         ."WHERE ControlledNameId = ".intval($this->Id));
        $OldNameResources = array();
        while ($Row = $this->DB->FetchRow())
        {
            $OldNameResources []= $Row["ResourceId"];
        }

        # Foreach of the old name resources, check to see if it's already
        # associated with the new name.  If not, associate it.
        foreach ($OldNameResources as $ResourceId)
        {
            if (!isset($NewNameResources[$ResourceId]))
            {
                $this->DB->Query("INSERT INTO ResourceNameInts "
                                 ."(ResourceId, ControlledNameId) VALUES "
                                 ."(".intval($ResourceId).",".intval($NewNameId).")");
            }
        }

        # Clear out all the associations to the old name
        $this->DB->Query("DELETE FROM ResourceNameInts WHERE ControlledNameId = ".intval($this->Id));
    }

    /**
    * Remove ControlledName (and any accompanying associations from database.
    * This must be the last use of this object.
    * @param DeleteIfHasResources Remove ControlledName even if Resources are
    *       currently associated with it.  (OPTIONAL, defaults to FALSE)
    */
    function Delete($DeleteIfHasResources = FALSE)
    {
        $DB =& $this->DB;

        if ($DeleteIfHasResources || !$this->InUse())
        {
            # delete this controlled name
            $DB->Query("DELETE FROM ControlledNames WHERE ControlledNameId=".
                $this->Id);

            # delete associated variant name
            $DB->Query("DELETE FROM VariantNames WHERE ControlledNameId=".
                $this->Id);

            if ($DeleteIfHasResources)
            {
                $DB->Query("DELETE FROM ResourceNameInts WHERE ".
                   "ControlledNameId=".$this->Id);
            }
        }
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $DB;
    private $DBFields;
    private $Id;
    private $ErrorStatus;

    # convenience function to supply parameters to Database->UpdateValue()
    private function UpdateValue($FieldName, $NewValue)
    {
        return $this->DB->UpdateValue("ControlledNames", $FieldName,
                $NewValue, "ControlledNameId = ".$this->Id,
                $this->DBFields, TRUE);
    }

    # convenience function for VariantNames table
    private function UpdateVariantValue($FieldName, $NewValue)
    {
        if (!empty($NewValue))
        {
            # see if variant name exists for the controlled Name
            $this->DB->Query("SELECT * from VariantNames WHERE ".
                        "ControlledNameId = ".$this->Id);

            # variant name exists so do an update
            if ($this->DB->NumRowsSelected() > 0)
            {
                return $this->DB->UpdateValue("VariantNames",
                            $FieldName, $NewValue,
                            "ControlledNameId = ".$this->Id,
                            $this->DBFields, TRUE);
            }
            # no variant name so do an insert
            else if ($NewValue != DB_NOVALUE)
            {
                $this->DB->Query("INSERT into VariantNames ".
                        "(VariantName, ControlledNameId) VALUES ".
                        "('".addslashes($NewValue)."', ".$this->Id.")");
            }
        }
        # delete variant name
        else
        {
            $this->DB->Query("Delete from VariantNames where ".
                "ControlledNameId = ".$this->Id);
        }
    }
}

?>
