<?PHP

#
#   FILE:  SPT--Classification.php
#
#   METHODS PROVIDED:
#       Classification($ClassId, $FullName = NULL, $TypeId = NULL)
#           - constructor
#       TypeName()
#           - get classification type as name
#       RecalcDepthAndFullName()
#           - rebuild classification full name and recalculate depth in hierarchy
#       RecalcResourceCount()
#           - recalculate number of resources assigned to class and any parent classes
#       ChildCount()
#           - calculate number of classifications that have this class as parent
#       Delete($DeleteParents = FALSE, $DeleteIfHasResources = FALSE, $DeleteIfHasChildren = FALSE)
#           - remove classification (and accompanying associations) from database
#       Id()
#       FullName()
#       Depth()
#       ParentId()
#       ResourceCount()
#           - get attributes
#       SegmentName($NewValue = DB_NOVALUE)
#       TypeId($NewValue = DB_NOVALUE)
#       LinkString($NewValue = DB_NOVALUE)
#       QualiferId($NewValue = DB_NOVALUE)
#           - get/set attributes
#
#   AUTHOR:  Edward Almasy
#
#   Part of the Scout Portal Toolkit
#   Copyright 2002-2003 Internet Scout Project
#   http://scout.wisc.edu
#

require_once(dirname(__FILE__)."/SPT--SPTDatabase.php");

# error status codes
define("CLASSSTAT_OK",                0);
define("CLASSSTAT_INVALIDID",         1);

class Classification {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function Classification($ClassId, $FullName = NULL, $FieldId = NULL)
    {
        static $IdCache;

        # assume everything will turn out okay
        $this->ErrorStatus = CLASSSTAT_OK;
        
        # create DB handle for our use
        $this->DB =& new SPTDatabase();
        $DB =& $this->DB;

        # if class ID not given (indicating class must be created)
        if ($ClassId == NULL)
        {
            # parse classification name into separate segments
            $Segments = preg_split("/--/", $FullName);

            # start out with top as parent
            $ParentId = -1;

            # for each segment
            $CurrentDepth = -1;
            $CurrentFullName = "";
            foreach ($Segments as $Segment)
            {
                # track segment depth and full classification name for use in adding new entries
                $Segment = trim($Segment);
                $CurrentDepth++;
                $CurrentFullName .= (($CurrentFullName == "") ? "" : " -- ").$Segment;

                # if we have added classifications
                $Segment = addslashes($Segment);
		if ($this->SegmentsCreated)
		{
		    # we know that current segment will not be found
		    $ClassId = NULL;
		}
		else
		{
                    # look up classification with current parent and segment name
                    if (!isset($IdCache[$FieldId][$ParentId][$Segment]))
                    {
		        if ($ParentId == -1)
		        {
                            $IdCache[$FieldId][$ParentId][$Segment] = $DB->Query(
                                    "SELECT ClassificationId FROM Classifications"
                                    ." WHERE ParentId = -1"
                                    ." AND SegmentName = '".addslashes($Segment)."'"
                                    ." AND FieldId = ".intval($FieldId),
                                    "ClassificationId");
		        }
		        else
		        {
                            $IdCache[$FieldId][$ParentId][$Segment] = $DB->Query(
                                    "SELECT ClassificationId FROM Classifications "
                                    ."WHERE ParentId = ".intval($ParentId)
                                    ." AND SegmentName = '".addslashes($Segment)."'",
                                    "ClassificationId");
		        }
                    }
                    $ClassId = $IdCache[$FieldId][$ParentId][$Segment];
		}

                # if classification not found
                if ($ClassId == NULL)
                {
                    # add new classification
                    $DB->Query("INSERT INTO Classifications "
                             ."(FieldId, ParentId, SegmentName, ClassificationName, Depth, ResourceCount) "
                             ."VALUES (".intval($FieldId).", "
                                .intval($ParentId).", "
                                ."'".addslashes($Segment)."', "
                                ."'".addslashes($CurrentFullName)."', "
                                .intval($CurrentDepth).", 0)");
                    $ClassId = $DB->LastInsertId("Classifications");
                    $IdCache[$FieldId][$ParentId][$Segment] = $ClassId;

                    # track total number of new classification segments created
                    $this->SegmentsCreated++;
                }

                # set parent to created or found class
		$PreviousParentId = $ParentId;
                $ParentId = $ClassId;
            }

	    # our class ID is the one that was last found
            $this->Id = $ClassId;

	    # if we have added new classifications
	    if ($this->SegmentsCreated)
	    {
	        # set attributes from values we already know
		$this->DBFields["ClassificationId"] = $this->Id;
		$this->DBFields["FieldId"] = $FieldId;
		$this->DBFields["ParentId"] = $PreviousParentId;
		$this->DBFields["SegmentName"] = $Segment;
		$this->DBFields["ClassificationName"] = $CurrentFullName;
		$this->DBFields["Depth"] = $CurrentDepth;
		$this->DBFields["ResourceCount"] = 0;
	    }
	    else
	    {
                # load in attributes from database
                $DB->Query("SELECT * FROM Classifications"
                           ." WHERE ClassificationId = ".intval($this->Id));
                $this->DBFields = $DB->FetchRow();
	    }
        }
	else
	{
	    # our class ID is the one that was supplied by caller
            $this->Id = $ClassId;

            # load in attributes from database
            $DB->Query("SELECT * FROM Classifications"
                       ." WHERE ClassificationId = ".intval($this->Id));
            $this->DBFields = $DB->FetchRow();
	}
        
        # set error status if class info not loaded
        if ($this->DBFields["ClassificationId"] != $this->Id)
        {
            $this->ErrorStatus = CLASSSTAT_INVALIDID;
        }
    }
    
    # check success of last call
    function Status() {  return $this->ErrorStatus;  }

    # get attributes
    function Id()            {  return $this->Id;  }
    function FullName()      {  return stripslashes($this->DBFields["ClassificationName"]);  }
    function Name()          {  return $this->FullName();  }
    function VariantName()   {  return NULL;  }
    function Depth()         {  return $this->DBFields["Depth"];  }
    function ParentId()      {  return $this->DBFields["ParentId"];  }
    function ResourceCount() {  return $this->DBFields["ResourceCount"];  }
    function SegmentsCreated() { return $this->SegmentsCreated; }

    # get/set attributes
    function SegmentName($NewValue = DB_NOVALUE) {  return stripslashes($this->UpdateValue("SegmentName", $NewValue));  }
    function LinkString($NewValue = DB_NOVALUE) {  return stripslashes($this->UpdateValue("LinkString", $NewValue));  }
    function QualifierId($NewValue = DB_NOVALUE) {  return $this->UpdateValue("QualifierId", $NewValue);  }
    function FieldId($NewValue = DB_NOVALUE)     {  return $this->UpdateValue("FieldId", $NewValue);  }

    # get/set associated qualifier
    function Qualifier($NewValue = DB_NOVALUE)
    {
        # if new qualifier supplied
        if ($NewValue !== DB_NOVALUE) 
        {
            # set new qualifier ID
            $this->QualifierId($NewValue->Id());

            # use new qualifier for return value
            $Qualifier = $NewValue;
        }
        else
        {
            # if qualifier is available
            if ($this->QualifierId() !== NULL)
            {   
                # create qualifier object using stored ID
                $Qualifier = new Qualifier($this->QualifierId());
            }
            else
            {
                # return NULL to indicate no qualifier
                $Qualifier = NULL;
            }
        }

        # return qualifier to caller
        return $Qualifier;
    }

    # rebuild classification full name and recalculate depth in hierarchy
    function RecalcDepthAndFullName()
    {
        $DB =& $this->DB;

        # start with full classification name set to our segment name
        $FullClassName = $this->DBFields["SegmentName"];

        # assume to begin with that we're at the top of the hierarchy
        $Depth = 0;

        # while parent available
        $ParentId = $this->DBFields["ParentId"];
        while ($ParentId != -1)
        {
            # retrieve classification information
            $DB->Query("SELECT SegmentName, ParentId "
                    ."FROM Classifications "
                    ."WHERE ClassificationId=".$ParentId);
            $Record = $DB->FetchNextRowArray();

            # prepend segment name to full classification name
            $FullClassName = stripslashes($Record["SegmentName"])
                    ." -- ".$FullClassName;

            # increment depth value
            $Depth++;

            # move to parent of current classification
            $ParentId = $Record["ParentId"];
        }

        # for each child
        $DB->Query("SELECT ClassificationId "
                ."FROM Classifications "
                ."WHERE ParentId=".intval($this->Id));
        while ($Record = $DB->FetchNextRowArray())
        {
            # perform depth and name recalc
            $Child =& new Classification($Record["ClassificationId"]);
            $Child->RecalcDepthAndFullName();
        }

        # save new depth and full classification name
        $DB->Query("UPDATE Classifications SET "
                ."Depth=".intval($Depth).", "
                ."ClassificationName='".addslashes($FullClassName)."' "
                ."WHERE ClassificationId=".intval($this->Id));
        $this->DBFields["ClassificationName"] = $FullClassName;
        $this->DBFields["Depth"] = $Depth;
    }

    # recalculate number of resources assigned to class and any parent classes
    function RecalcResourceCount()
    {
        $DB =& $this->DB;

        # retrieve new count of resources directly associated with class
        $DB->Query("SELECT COUNT(*) AS ResourceCount"
                ." FROM ResourceClassInts, Resources"
                ." WHERE ClassificationId=".intval($this->Id)
                ." AND ResourceClassInts.ResourceId = Resources.ResourceId"
                ." AND ReleaseFlag = 1");
        $Record = $DB->FetchNextRowArray();
        $this->DBFields["ResourceCount"] = $Record["ResourceCount"];

        # add on resources associated with all children
        $this->DBFields["ResourceCount"] += $DB->Query(
                "SELECT SUM(ResourceCount) AS ResourceCountTotal "
                    ."FROM Classifications "
                    ."WHERE ParentId = ".intval($this->Id),
                "ResourceCountTotal");

        # save new resource count
        $DB->Query("UPDATE Classifications SET "
                ."ResourceCount=".$this->DBFields["ResourceCount"]." "
                ."WHERE ClassificationId=".intval($this->Id));

        # update resource count for our parent (if any)
        if ($this->DBFields["ParentId"] != -1)
        {
            $Class =& new Classification($this->DBFields["ParentId"]);
            $Class->RecalcResourceCount();
        }
    }

    # calculate number of classifications that have this class as parent
    function ChildCount()
    {
        # return count of classifications that have this one as parent
        return $this->DB->Query("SELECT COUNT(*) AS ClassCount "
                    ."FROM Classifications "
                    ."WHERE ParentId=".intval($this->Id),
                "ClassCount");
    }

    # return array of classificationids that have this class as parent
    # this also returns grandchildren, great grandchildren, etc.
    function ChildList()
    {
        $ChildList = array();
        
        $this->DB->Query("SELECT ClassificationId  "
                    ."FROM Classifications "
                    ."WHERE ParentId=".intval($this->Id));

         while ($Entry = $this->DB->FetchNextRowArray())
         {
            $ChildList[] = $Entry["ClassificationId"];
            $Child = & new Classification($Entry["ClassificationId"]);
            if($Child->ChildCount() > 0)
            {
                $GrandChildList = $Child->ChildList();
                $ChildList = array_merge($GrandChildList, $ChildList);
            }
         }
         return $ChildList;
    }
    
    # remove classification (and accompanying associations) from database
    function Delete($DeleteParents = FALSE, 
            $DeleteIfHasResources = FALSE, $DeleteIfHasChildren = FALSE)
    {
        $DB =& $this->DB;

        # if no resources or okay to delete with resources
        #         and no children or okay to delete with children
        if (($DeleteIfHasResources || ($this->ResourceCount() == 0))
                && ($DeleteIfHasChildren || ($this->ChildCount() == 0)))
        {
            $ParentId = $this->DBFields["ParentId"];

            if ($DeleteIfHasResources)
            {
                $DB->Query("DELETE FROM ResourceClassInts "
                        ."WHERE ClassificationId=".intval($this->Id));
                $this->RecalcResourceCount();        
            }
            # delete this classification
            $DB->Query("DELETE FROM Classifications "
                    ."WHERE ClassificationId=".intval($this->Id));

            # delete parent classification (if requested)
            if (($DeleteParents) && ($this->DBFields["ParentId"] != -1))
            {
                $Parent =& new Classification($this->DBFields["ParentId"]);
                $Parent->Delete(
                        TRUE, $DeleteIfHasResources, $DeleteIfHasChildren);
            }
        }
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $DB;
    var $DBFields;
    var $Id;
    var $ErrorStatus;
    var $SegmentsCreated;

    # convenience function to supply parameters to Database->UpdateValue()
    function UpdateValue($FieldName, $NewValue)
    {
        return $this->DB->UpdateValue("Classifications", $FieldName, $NewValue,
                               "ClassificationId = ".$this->Id,
                               $this->DBFields, TRUE);
    }
}


?>
