<?PHP

# add the current directory to the include path for the Zend framework
set_include_path(get_include_path().PATH_SEPARATOR.dirname(__FILE__));

class YouTube extends Plugin
{

    /**
     * @const MAX_FILE_SIZE maximum size, in bytes, of files to upload
     */
    const MAX_FILE_SIZE = 2147483648;

    /**
     * @const MAX_TIME_TO_LIVE maximum time to live
     */
    const MAX_TIME_TO_LIVE = 4;

    /**
     * @const AUTHSUB_NEXT page to return to after authentication is complete
     */
    const AUTHSUB_NEXT = "AuthenticationCallback.php";

    /**
     * @const AUTHSUB_SCOPE identifier of the Google service to access
     */
    const AUTHSUB_SCOPE = "https://gdata.youtube.com";

    /**
     * @const DEVELOPER_KEY GData developer key
     */
    const DEVELOPER_KEY = "AI39si5u3pgLDT6hsz3YBuiOfdSh1ysx9FamkHQW6SGSRfKuWhZSUaMF0JzyZGRu-qOmlRHiRAxcyPmi2baR6k8l48UsDzQg4w";

    /**
     * @const MAJOR_PROTOCOL_VERSION major protocol version for the YouTube API
     */
    const MAJOR_PROTOCOL_VERSION = 2;

    /**
     * @const RESUME_URI URI used for resuming uploads
     */
    const RESUME_URI = "plugins/YouTube/ResumeUpload.php?UploadId=";

    /**
     * Register information about this plugin.
     */
    public function Register()
    {
        $this->Name = "YouTube";
        $this->Version = "1.0.1";
        $this->Description = "Adds the ability to upload videos to YouTube.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array("CWISCore" => "2.2.3");
        $this->EnabledByDefault = FALSE;

        $this->CfgSetup["AutomaticDelete"] = array(
          "Type" => "Flag",
          "Label" => "Automatically Delete Videos from YouTube",
          "Help" => "Automatically delete videos from YouTube when the"
              ." associated files are deleted from CWIS.",
          "OnLabel" => "Yes",
          "OffLabel" => "No");
    }

    /**
     * Initialize objects, settings, and include paths.
     */
    public function Initialize()
    {
        # add the current directory to the include path for the Zend framework
        set_include_path(get_include_path().PATH_SEPARATOR.dirname(__FILE__));

        # load classes from the Zend library that are necessary for this method
        require_once "Zend/Loader.php";
        Zend_Loader::loadClass("Zend_Gdata_YouTube");
        Zend_Loader::loadClass("Zend_Gdata_AuthSub");
        Zend_Loader::loadClass("Zend_Gdata_App_MediaFileSource");

        # load plugin classes that are necessary for this method
        require_once "YouTube_Zend_Gdata_YouTube.php";
        require_once "YouTube_Zend_Gdata_HttpClient.php";

        # revoke authentication if requested
        if ($this->ConfigSetting("RevokeAuthentication"))
        {
            $this->RevokeAuthentication();
            $this->ConfigSetting("RevokeAuthentication", FALSE);
        }

        if ($this->IsAuthenticated())
        {
            $SafeName = defaulthtmlentities($this->GetUserName());

            # check for a bad user name
            if (!strlen($SafeName))
            {
                return '
                    Could not get user details from YouTube. If you are using a
                    Google account, make sure it has been
                    <a href="http://www.youtube.com/create_channel">set up for
                    YouTube</a>. It may take awhile for the changes to take
                    effect.';
            }

            # only show authentication revocation option if authenticated
            $this->CfgSetup["RevokeAuthentication"] = array(
              "Type" => "Flag",
              "Label" => "Remove Association with YouTube Account",
              "Help" => "Remove this plugin's association with the YouTube account.",
              "OnLabel" => "Yes",
              "OffLabel" => "No");

            $Association = '
              The plugin is currently associated with
              <a href="http://www.youtube.com/user/'.$SafeName.'" target="_blank">
              <i>'.$SafeName.'</i></a>.';

            $this->Instructions = $Association;
            $this->Description .= " " . $Association;
        }

        else
        {
            # fail initialization if the plugin isn't authenticated
            $Prefix = is_int(strpos(__FILE__, "local/plugins")) ? "local/" : "";
            $Error = '
                The plugin needs to be
                <a href="'.$Prefix.'plugins/YouTube/InitiateAuthentication.php">
                  associated with a YouTube account</a>
                prior to use.';

            $this->Instructions = $Error;
            return $Error;
        }

        # provide a fallback in case the version of PHP running doesn't ship
        # with json_encode
        if (!function_exists("json_encode"))
        {
            require_once dirname(__FILE__)."/include/json_encode.php";
        }
    }

    /**
     * Install necessary SQL tables
     * @return NULL on success, error message on error
     */
    public function Install()
    {
        $Database = new Database();
        $Schema = new MetadataSchema();

        # table for uploaded videos
        if (FALSE === $Database->Query("
            CREATE TABLE IF NOT EXISTS YouTube_Videos (
                VideoId        INT,
                ResourceId     INT,
                FieldId        INT,
                FileId         INT,
                YouTubeId      TEXT,
                UserName       TEXT,
                State          ENUM(
                                 'initialized',
                                 'uploading',
                                 'failed',
                                 'processing',
                                 'ready',
                                 'deleted'),
                Title          TEXT,
                Description    TEXT,
                PRIMARY KEY    (VideoId)
            );"))
        { return "Could not create the videos table"; }

        # table for current uploads
        if (FALSE === $Database->Query("
            CREATE TABLE IF NOT EXISTS YouTube_UploadQueue (
                UploadId         INT,
                ResourceId       INT,
                FileId           INT,
                UploadUrl        TEXT,
                LastPid          INT,
                PercentComplete  INT,
                StartTime        TIMESTAMP,
                LastUpdate       TIMESTAMP,
                PRIMARY KEY      (UploadId)
            );"))
        { return "Could not create the upload queue table"; }

        # get the XML representations of the fields used by this plugin
        $Xml = $this->GetFieldXml();

        # create each field from its XML representation
        foreach ($Xml as $Path => $Data)
        {
            $Field = $Schema->AddFieldFromXml($Data);

            if (!($Field instanceof MetadataField))
            {
                return "Could not create the YouTube video metadata fields.";
            }
        }

        $this->ConfigSetting("AutomaticDelete", FALSE);
        $this->ConfigSetting("AuthSubToken", NULL);
        $this->ConfigSetting("UserName", NULL);

        return NULL;
    }

    /**
     * Uninstall the plugin.
     * @return NULL|string NULL if successful or an error message otherwise
     */
    public function Uninstall()
    {
        $Database = new Database();

        # videos table
        if (FALSE === $Database->Query("DROP TABLE YouTube_Videos;"))
        { return "Could not remove the videos table"; }

        # upload queue table
        if (FALSE === $Database->Query("DROP TABLE YouTube_UploadQueue;"))
        { return "Could not remove the upload queue table"; }

        # remove the video file field
        $this->GetVideoFileField()->Drop();
    }

    /**
     * Upgrade from a previous version.
     * @param string $PreviousVersion previous version
     */
    public function Upgrade($PreviousVersion)
    {
        # load classes from the Zend library that are necessary for this method
        require_once "Zend/Loader.php";
        Zend_Loader::loadClass("Zend_Gdata_YouTube");
        Zend_Loader::loadClass("Zend_Gdata_AuthSub");
        Zend_Loader::loadClass("Zend_Gdata_App_MediaFileSource");

        # load plugin classes that are necessary for this method
        require_once "YouTube_Zend_Gdata_YouTube.php";
        require_once "YouTube_Zend_Gdata_HttpClient.php";

        # ugprade from versions < 1.0.1 to 1.0.1
        if (version_compare($PreviousVersion, "1.0.1", "<"))
        {
            # fetch the user name if authenticated
            if ($this->IsAuthenticated())
            {
                $this->GetUserName(TRUE);
            }
        }
    }

    /**
     * Declare the events this plugin provides to the application framework.
     * @return an array of the events this plugin provides
     */
    public function DeclareEvents()
    {
        return array(
            "YOUTUBE_INITIATE_AUTHENTICATION"
              => ApplicationFramework::EVENTTYPE_FIRST,
            "YOUTUBE_EXCHANGE_TOKEN"
              => ApplicationFramework::EVENTTYPE_FIRST,
            "YOUTUBE_REVOKE_AUTHENTICATION"
              => ApplicationFramework::EVENTTYPE_DEFAULT,
            "YOUTUBE_GET_CURRENT_UPLOADS"
               => ApplicationFramework::EVENTTYPE_FIRST,
            "YOUTUBE_DIRECT_RESUME"
               => ApplicationFramework::EVENTTYPE_DEFAULT,
            "YOUTUBE_REUPLOAD_YOUTUBE_VIDEO"
               => ApplicationFramework::EVENTTYPE_DEFAULT,
            "YOUTUBE_DELETE_YOUTUBE_VIDEO"
               => ApplicationFramework::EVENTTYPE_DEFAULT);
    }

    /**
     * Hook the events into the application framework.
     * @return an array of events to be hooked into the application framework
     */
    public function HookEvents()
    {
        return array(
            "EVENT_PERIODIC" => "CheckDelegation",
            "EVENT_RESOURCE_FILE_ADD" => "FileAdded",
            "EVENT_RESOURCE_FILE_DELETE" => "FileDeleted",
            "EVENT_APPEND_HTML_TO_FIELD_DISPLAY" => "FieldDisplayed",
            "EVENT_IN_HTML_HEADER" => "InHtmlHeader",
            "YOUTUBE_INITIATE_AUTHENTICATION" => "InitiateAuthentication",
            "YOUTUBE_EXCHANGE_TOKEN" => "ExchangeToken",
            "YOUTUBE_REVOKE_AUTHENTICATION" => "RevokeAuthentication",
            "YOUTUBE_GET_CURRENT_UPLOADS" => "GetCurrentUploads",
            "YOUTUBE_DIRECT_RESUME" => "DirectResume",
            "YOUTUBE_REUPLOAD_YOUTUBE_VIDEO" => "Reupload",
            "YOUTUBE_DELETE_YOUTUBE_VIDEO" => "DeleteYouTubeVideo");
    }

    /**
     * Print stylesheet and Javascript elements in the page header
     */
    public function InHtmlHeader()
    {
        $BaseUrl = $this->GetBaseUrl();
        $Prefix = (is_int(strpos(__FILE__, "local/plugins"))) ? "/local/" : "/";
        $CssUrl = $BaseUrl . $Prefix . "plugins/YouTube/include/YouTube.css";
?>
<link rel="stylesheet" type="text/css" href="<?PHP print $CssUrl; ?>" />
<?PHP
    }

    /**
     * Begin various checks.
     */
    public function CheckDelegation()
    {
        global $AF;

        $UploadFactory = new YouTube_UploadFactory();
        $Uploads = $UploadFactory->GetItems();

        # restart any stalled uploads
        foreach ($Uploads as $Upload)
        {
            $AF->QueueUniqueTask(
                array($this, "Resume"),
                array($Upload->UploadId),
                ApplicationFramework::PRIORITY_BACKGROUND,
                "Resume uploading a video to YouTube.");
        }

        # return in 45 mins
        return 45;
    }

    /**
     * Callback for the EVENT_RESOURCE_FILE_ADD event.
     * @param $Field a MetadataField object
     * @param $Resource a Resource object
     * @param $File a File object
     */
    public function FileAdded($Field, $Resource, $File)
    {
        # get the associated values, if available
        $Title = GetArrayValue($_POST, "P_YouTube_Title");
        $Description = GetArrayValue($_POST, "P_YouTube_Description");

        $this->HandleUpload($File, $Title, $Description);
    }

    /**
     * Callback for the EVENT_RESOURCE_FILE_DELETE event.
     * @param $Field a MetadataField object
     * @param $Resource a Resource object
     * @param $File a File object
     */
    public function FileDeleted($Field, $Resource, $File)
    {
        global $AF;

        $YouTubeService = $this->GetYouTubeService();

        try
        {
            # delete any uploads associated with the file
            $UploadFactory = new YouTube_UploadFactory();
            $Upload = $UploadFactory->GetUploadForFile($File);
            $Upload->Delete();
        } catch (Exception $Exception) {}

        try
        {
            $VideoFactory = new YouTube_VideoFactory();
            $Video = $VideoFactory->GetVideoForFile($File);

            # this handles deleting the YouTube video if configured to do so
            # and video deletion
            $this->Delete($Video);
        } catch (YouTube_VideoDoesNotExistException $Exception) {}

        catch (Exception $Exception)
        {
            # an error occurred, so retry deletion
            $AF->QueueUniqueTask(
                array($this, "RetryDelete"),
                array($Video->VideoId, 0),
                ApplicationFramework::PRIORITY_BACKGROUND,
                "YouTube video deletion failed. Trying again.");
        }
    }

    /**
     * Hook called whenever a field is displayed.
     * @param $Field a MetadataField object
     * @param $Resource a Resource object
     * @param $Context display context identifier
     * @param $Html current HTML to be displayed
     */
    public function FieldDisplayed($Field, $Resource, $Context, $Html)
    {
        if ($Field->Type() != MetadataSchema::MDFTYPE_FILE)
        {
            return array(
                "Field" => $Field,
                "Resource" => $Resource,
                "Context" => $Context,
                "Html" => $Html);
        }

        if ($Field->Name() != "YouTube Videos")
        {
            return array(
                "Field" => $Field,
                "Resource" => $Resource,
                "Context" => $Context,
                "Html" => $Html);
        }

        $VideoFactory = new YouTube_VideoFactory();
        $Videos = $VideoFactory->GetVideosForResource($Resource);
        $YouTubeService = $this->GetYouTubeService();
        $ExistsCache = array();

        # update video info before displaying them
        foreach ($Videos as $Key => $Video)
        {
            $State = $Video->State;
            $Exists = $YouTubeService->videoExists($Video);

            # remove videos that have been deleted from YouTube and aren't
            # just initialized or uploading

            if ($State != "failed" && $State != "initialized" && $State != "uploading" && !$Exists)
            {
                $File = $Video->GetFile();

                if ($File->Status() == File::FILESTAT_DOESNOTEXIST)
                {
                    $Video->Delete();
                    unset($Videos[$Key]);
                    continue;
                }

                else
                {
                    $Video->FlagAsDeleted();
                }
            }

            if ($Video->State == "processing")
            {
                # nest the statement to avoid the API call if possible
                if ($YouTubeService->isProcessed($Video))
                {
                    $Video->FlagAsReady();
                }
            }

            # cache whether it exists or not for use below
            $ExistsCache[$Video->VideoId] = $Exists;
        }

        if ($Context == "DISPLAY")
        {
            # return if there are no videos to consider
            if (count($Videos) < 1)
            {
                return array($Field, $Resource, $Context, $Html);
            }

            $Variables = array("Videos" => $Videos);

            $Html .= $this->IncludeAndBuffer("Display", $Variables);
        }

        else if ($Context == "EDIT")
        {
            $State = array();

            foreach ($Videos as $Video)
            {
                $Exists = $ExistsCache[$Video->VideoId];

                if ($Exists && $Video->YouTubeId)
                {
                    $VideoEntry = $YouTubeService->getVideoEntry($Video->YouTubeId);
                    $Title = $VideoEntry->getVideoTitle();
                    $Description = $VideoEntry->getVideoDescription();
                    $ShortTitle = NeatlyTruncateString($Title, 20);

                    # update the YouTube_Video object title and description
                    $Video->SetTitle($Title);
                    $Video->SetDescription($Description);
                }

                else
                {
                    $Title = $Video->Title;
                    $Description = $Video->Description;
                    $ShortTitle = NeatlyTruncateString($Title, 20);
                }

                $State[$Video->FileId] = array(
                    "VideoId" => $Video->VideoId,
                    "ResourceId" => $Video->ResourceId,
                    "FieldId" => $Video->FieldId,
                    "FileId" => $Video->FileId,
                    "YouTubeId" => $Video->YouTubeId,
                    "UserName" => $Video->UserName,
                    "State" => $Video->State,
                    "Title" => $Title,
                    "ShortTitle" => $ShortTitle,
                    "Description" => $Description,
                    "Exists" => $Exists);
            }

            $Prefix = (is_int(strpos(__FILE__, "local/plugins"))) ? "local/" : "";

            $ReturnTo = $this->GetBaseUrl() . "/index.php?";
            $ReturnTo .= http_build_query($_GET);
            $ReturnTo = defaulthtmlentities(urlencode($ReturnTo));

            $Variables = array(
                "UserName" => $this->GetUserName(),
                "UrlPrefix" => $Prefix,
                "ReturnTo" => $ReturnTo,
                "State" => $State);

            $Html .= $this->IncludeAndBuffer("EditForm", $Variables);
        }

        return array(
            "Field" => $Field,
            "Resource" => $Resource,
            "Context" => $Context,
            "Html" => $Html);
    }

    /**
     * Determine whether the plugin is authenticated with Google.
     * @param $ForceCheck force a check instead of using the cached value
     * @return TRUE if authenticated or FALSE otherwise
     */
    public function IsAuthenticated($ForceCheck=FALSE)
    {
        static $IsAuthenticated;

        if (!isset($IsAuthenticated) || $ForceCheck)
        {
            $Token = $this->ConfigSetting("AuthSubToken");

            # assume the plugin is not authenticated by default
            $IsAuthenticated = FALSE;

            # only send a token info request when a token is set
            if (!empty($Token))
            {
                try
                {
                    $Info = Zend_Gdata_AuthSub::getAuthSubTokenInfo($Token);

                    # the plugin is authenticated only if no HTML was given
                    $IsAuthenticated = FALSE === strpos($Info, "<HTML>");
                }

                # there was an error getting the token info, so the plugin isn't
                # authenticated for all intents and purposes
                catch (Exception $Exception)
                {
                    $IsAuthenticated = FALSE;
                }
            }
        }

        return $IsAuthenticated;
    }

    /**
     * Initiate an AuthSub authentication request with Google and return the
     * URL the user needs to go to in order to provide his or her login
     * credentials.
     * @return the URL to redirect to so the user can authenticate with Google
     */
    public function InitiateAuthentication()
    {
        # reset existing AuthSub configuration
        $this->ConfigSetting("AuthSubToken", NULL);

        # create an absolute URI based on the current URI to use with AuthSub
        $Prefix = (is_int(strpos(__FILE__, "local/plugins"))) ? "/local/" : "/";
        $Next = $this->GetBaseUrl() . $Prefix . self::AUTHSUB_NEXT;

        # fetch the AuthSub token URI
        $TokenUri = Zend_Gdata_AuthSub::getAuthSubTokenUri(
            $Next, self::AUTHSUB_SCOPE, FALSE, TRUE);

        return $TokenUri;
    }

    /**
     * Exchange an AuthSub temp token with a session one.
     * @param $Token temporary AuthSub token
     * @return TRUE if the exchange was successful or FALSE otherwise
     */
    public function ExchangeToken($Token)
    {
        try
        {
            # exchange the temp token for a session one and save the session one
            $SessionToken = Zend_Gdata_AuthSub::getAuthSubSessionToken($Token);
            $this->ConfigSetting("AuthSubToken", $SessionToken);

            # cause the user name to be fetched
            $this->GetUserName(TRUE);

            return TRUE;
        } catch (Exception $Exception) {}

        return FALSE;
    }

    /**
     * Revoke AuthSub authentication.
     */
    public function RevokeAuthentication()
    {
        $Token = $this->ConfigSetting("AuthSubToken");

        # only send a request for token revocation if a token is set
        if (!empty($Token))
        {
            try
            {
                Zend_Gdata_AuthSub::AuthSubRevokeToken($Token);
            } catch (Exception $Exception) {}
        }

        # clear AuthSub configuration
        $this->ConfigSetting("AuthSubToken", NULL);
        $this->ConfigSetting("UserName", NULL);
    }

    /**
     * Get enqueued and in-progress uploads.
     * @return an array of YouTube_Upload objects
     */
    public function GetCurrentUploads()
    {
        $UploadFactory = new YouTube_UploadFactory();
        $Uploads = $UploadFactory->GetItems();

        return $Uploads;
    }

    /**
     * Resume the upload with the given ID. This function does not carry out
     * the actual upload and does not block as a result. See
     * self::DirectResume().
     * @param $UploadId upload ID
     */
    public function Resume($UploadId)
    {
        global $AF;

        try
        {
            $Upload = new YouTube_Upload($UploadId);
        }

        # the upload doesn't exist and likely finished after the task was queued
        # so just return
        catch (YouTube_UploadDoesNotExistException $Exception)
        {
            return;
        }

        # only trigger an actual upload process if the file isn't already in
        # the process of being uploaded
        if (!$Upload->InProcessing())
        {
            # get the URI of file that handles the actual upload directly
            $UploadId = $Upload->UploadId;
            $BaseUrl = $this->GetBaseUrl();
            $Prefix = (is_int(strpos(__FILE__, "local/plugins"))) ? "/local/" : "/";
            $Uri = $BaseUrl . $Prefix . self::RESUME_URI . $UploadId;

            # trigger the direct resume
            $HttpClient = $this->GetHttpClient();
            $HttpClient->setUri($Uri);
            $HttpClient->setHeaders("Connection", "close");
            $Response = $HttpClient->request();
        }

        # assume the file won't be fully uploaded or that the task will need to
        # be run again to complete the upload, so requeue the task
        $AF->QueueUniqueTask(array($this, "Resume"),
            array($UploadId), ApplicationFramework::PRIORITY_BACKGROUND,
            "Resume uploading a video to YouTube.");
    }

    /**
     * Directly resume an upload. This will block execution and may cause PHP
     * to time out. See self::Resume().
     * @param $Upload a YouTube_Upload object
     */
    public function DirectResume(YouTube_Upload $Upload)
    {
        # return if the upload is in the process of being uploaded by another
        # process
        if ($Upload->InProcessing())
        {
            return;
        }

        $Upload->UpdateLastPid();

        $YouTubeService = $this->GetYouTubeService();

        # since this tries to upload the rest of the file, it may hit the PHP
        # timeout. in that case, this method will need to be called again
        try
        {
            $VideoEntry = $YouTubeService->resumeUpload($Upload);
        }

        # something in the request went wrong return
        catch (YouTube_UploadFailureException $Exception)
        {
            return;
        }

        $VideoFactory = new YouTube_VideoFactory();
        $File = new File(intval($Upload->FileId));

        # update the video object
        $Video = $VideoFactory->GetVideoForFile($File);
        $Video->SetYouTubeId($VideoEntry->getVideoId());
        $Video->SetUserName($this->GetUserName());
        $Video->FlagAsProcessing();

        $EscapedTitle = trim(strip_tags($Video->Title));

        # the API calls will fail if it's empty
        if (strlen($EscapedTitle))
        {
            $VideoEntry->setVideoTitle($EscapedTitle);
        }

        $EscapedDescription = trim(strip_tags($Video->Description));

        # the API calls will fail if it's empty
        if (strlen($EscapedDescription))
        {
            $VideoEntry->setVideoDescription($EscapedDescription);
        }

        # send the updated info to YouTube
        $UpdateUri = $VideoEntry->getEditLink()->getHref();
        $YouTubeService->updateEntry($VideoEntry, $UpdateUri);

        # at this point, all the info has been saved and updated so the upload
        # object is no longer necessary and can be deleted safely
        $Upload->Delete();
    }

    /**
     * Re-upload the video for the given file.
     * @param $FileId file ID
     */
    public function Reupload($FileId)
    {
        $File = new File(intval($FileId));

        try
        {
            $VideoFactory = new YouTube_VideoFactory();
            $Video = $VideoFactory->GetVideoForFile($File);

            # pass off to the upload handler
            $this->HandleUpload(
                $File,
                $Video->Title,
                $Video->Description);

            # delete the video since a new one will be created
            $Video->Delete();
        }

        catch (YouTube_VideoDoesNotExistException $Exception)
        {
            # the video no longer exists
            $this->HandleUpload($File);
        }
    }

    /**
     * Delete the YouTube video associated with the YouTube_Video object
     * with the given video ID or video object.
     * @param $VideoId YouTube_Video ID or YouTube_Video object
     */
    public function DeleteYouTubeVideo($VideoId)
    {
        $YouTubeService = $this->GetYouTubeService();

        if ($VideoId instanceof YouTube_Video)
        {
            $Video = $VideoId;
        }

        else
        {
            $Video = new YouTube_Video($VideoId);
        }

        # can't delete videos if they don't exist
        if (!$YouTubeService->videoExists($Video))
        {
            return;
        }

        # can't remove videos uploaded by another user
        if ($Video->UserName != $this->GetUserName())
        {
            return;
        }

        $YouTubeService->deleteVideo($Video);
    }

    /**
     * Retry the upload for the video with the given ID.
     * @param $VideoId video ID
     * @param $TimeToLive TTL value
     */
    public function RetryUpload($VideoId, $TimeToLive=0)
    {
        global $AF;

        try
        {
            $Video = new YouTube_Video($VideoId);
        }

        catch (YouTube_VideoDoesNotExistException $Exception)
        {
            # the video doesn't exist, so give up
            return;
        }

        # exhausted the TTL, so flag the video and give up
        if ($TimeToLive >= self::MAX_TIME_TO_LIVE)
        {
            $Video->FlagAsFailed();
            return;
        }

        try
        {
            # try the upload again
            $this->Upload($Video);
        }

        catch (Exception $Exception)
        {
            # increment the TTL
            $TimeToLive++;

            # an error occurred yet again, so retry using the incremented TTL
            $AF->QueueUniqueTask(
                array($this, "RetryUpload"),
                array($Video->VideoId, $TimeToLive),
                ApplicationFramework::PRIORITY_BACKGROUND,
                "YouTube video upload failed. Trying again.");
        }
    }

    /**
     * Retry deleting the video with the given ID.
     * @param $VideoId video ID
     * @param $TimeToLive TTL value
     */
    public function RetryDelete($VideoId, $TimeToLive=0)
    {
        global $AF;

        try
        {
            $Video = new YouTube_Video($VideoId);
        }

        catch (YouTube_VideoDoesNotExistException $Exception)
        {
            # the video doesn't exist, so give up
            return;
        }

        # exhausted the TTL, so flag the video and give up
        if ($TimeToLive >= self::MAX_TIME_TO_LIVE)
        {
            $Video->FlagAsDeleted();
            return;
        }

        try
        {
            # try deleting the video again
            $this->Delete($Video);
        }

        catch (Exception $Exception)
        {
            # increment the TTL
            $TimeToLive++;

            # an error occurred yet again, so retry using the incremented TTL
            $AF->QueueUniqueTask(
                array($this, "RetryDelete"),
                array($Video->VideoId, $TimeToLive),
                ApplicationFramework::PRIORITY_BACKGROUND,
                "YouTube video deletion failed. Trying again.");
        }
    }

    /**
     * Redirect to another URI.
     * @param $Uri URI to redirect to
     */
    public function Redirect($Uri)
    {
        # set HTTP headers to redirect
        header("HTTP/1.1 303 See Other");
        header("Location: ".$Uri);

?><!DOCTYPE html>
<html lang="en">
  <head>
    <meta http-equiv="refresh" content="0; URL=<?PHP print $Uri;  ?>" />
    <meta charset="UTF-8" />
  </head>
  <body></body>
</html>
<?PHP
    }

    /**
     * Determine whether the file is allowed to be uploaded.
     * @param $File a File object
     * @return TRUE if the file is allowed to be uploaded
     */
    protected function FileAllowedToBeUploaded(File $File)
    {
        # don't upload invalid files
        if ($File->Status() != File::FILESTAT_OK)
        {
            return FALSE;
        }

        # don't upload files that are greater than the maximum file size
        if ($File->GetLength() > self::MAX_FILE_SIZE)
        {
            return FALSE;
        }

        $Resource = new Resource($File->ResourceId());

        # don't upload files added to invalid resources
        if ($Resource->Status() != 1)
        {
            return FALSE;
        }

        $Schema = new MetadataSchema();
        $Field = $Schema->GetField($File->FieldId());

        # don't upload files add to invalid fields
        if ($Field->Status() != MetadataSchema::MDFSTAT_OK)
        {
            return FALSE;
        }

        $VideoField = $this->GetVideoFileField();

        # don't upload files that don't belong to the YouTube video field
        if ($Field->Id() != $VideoField->Id())
        {
            return FALSE;
        }

        return TRUE;
    }

    /**
     * Get the field used for YouTube uploads.
     * @return a MetadataField object
     */
    protected function GetVideoFileField()
    {
        static $Field;

        # get the field if it isn't already set
        if (!isset($Field))
        {
            $Schema = new MetadataSchema();
            $Field = $Schema->GetFieldByName("YouTube Videos");
        }

        return $Field;
    }

    /**
     * Do the prep work necessary to upload the file and then upload it.
     * @param $File a File object
     * @param $Title video title
     * @param $Description video description
     */
    protected function HandleUpload($File, $Title=NULL, $Description=NULL)
    {
        global $AF;

        # ignore files that should be ignored
        if (!$this->FileAllowedToBeUploaded($File))
        {
            return;
        }

        # use the file name as the title if no title is given since that is
        # what YouTube does by default
        if (!strlen($Title))
        {
            $Title = $File->Name();
        }

        # create a video object
        $Video = YouTube_Video::Create($File);
        $Video->SetTitle($Title);
        $Video->SetDescription($Description);

        try
        {
            # and upload it
            $this->Upload($Video);
        }

        catch (Exception $Exception)
        {
            # an error occurred, so retry
            $AF->QueueUniqueTask(
                array($this, "RetryUpload"),
                array($Video->VideoId, 0),
                ApplicationFramework::PRIORITY_BACKGROUND,
                "YouTube video upload failed. Trying again.");
        }
    }

    /**
     * Start a resumable upload for the given video and queue a task to begin
     * the uploading process.
     * @param $Video a YouTube_Video object
     * @throws Exception if resumable upload initiation failed
     */
    protected function Upload(YouTube_Video $Video)
    {
        global $AF;

        $YouTubeService = $this->GetYouTubeService();
        $File = $Video->GetFile();

        # create the media source for the YouTube video upload
        $MediaSource = new Zend_Gdata_App_MediaFileSource(
            $File->GetNameOfStoredFile());
        $MediaSource->setContentType($File->GetMimeType());
        $MediaSource->setSlug($File->Name());

        # create the video entry for the YouTube video upload
        $VideoEntry = new Zend_Gdata_YouTube_VideoEntry();
        $VideoEntry->setMediaSource($MediaSource);

        # this will throw an exception if the request fails
        $UploadUrl = $YouTubeService->startResumableUpload($VideoEntry);

        $Upload = YouTube_Upload::Create($File, $UploadUrl);
        $Video->FlagAsUploading();

        $AF->QueueUniqueTask(
            array($this, "Resume"),
            array($Upload->UploadId),
            ApplicationFramework::PRIORITY_BACKGROUND,
            "Resume uploading a video to YouTube.");
    }

    /**
     * Delete the given video.
     * @param $Video a YouTube_Video object
     * @throws Exception if deletion failed
     */
    protected function Delete(YouTube_Video $Video)
    {
        global $AF;

        if ($this->ConfigSetting("AutomaticDelete"))
        {
            $this->DeleteYouTubeVideo($Video);
        }

        $Video->Delete();
    }

    /**
     * Get the user name of the account associated with the plugin.
     * @param $ForceCheck force a check instead of using the cached value
     * @return user name
     */
    protected function GetUserName($ForceCheck=FALSE)
    {
        $UserName = $this->ConfigSetting("UserName");

        if (!$UserName || $ForceCheck)
        {
            try {
                $YouTubeService = $this->GetYouTubeService();
                $Profile = $YouTubeService->getUserProfile("default");
                $UserName = $Profile->getUsername()->getText();

                $this->ConfigSetting("UserName", $UserName);
            }

            catch (Exception $Exception)
            {
                return NULL;
            }
        }

        return $UserName;
    }

    /**
     * Construct a Zend_Http_Client object with the AuthSub token attached. This
     * can only be called reliably when authenticated.
     * @return a Zend_Gdata_HttpClient object
     */
    protected function GetHttpClient()
    {
        $Token = $this->ConfigSetting("AuthSubToken");
        $HttpClient = Zend_Gdata_AuthSub::getHttpClient(
            $Token,
            new YouTube_Zend_Gdata_HttpClient());

        return $HttpClient;
    }

    /**
     * Construct a YouTube_Zend_Gdata_YouTube object with the AuthSub token,
     * User-Agent header, and GData developer key. This can only be called
     * reliably when in an operational state.
     * @return a YouTube_Zend_Gdata_YouTube object
     */
    protected function GetYouTubeService()
    {
        # construct the YouTube service object
        $YouTubeService = new YouTube_Zend_Gdata_YouTube(
            $this->GetHttpClient(),
            "CWIS-YouTube-Plugin/".$this->Version,
            NULL,
            self::DEVELOPER_KEY);

        # set the major protocol version
        $YouTubeService->setMajorProtocolVersion(self::MAJOR_PROTOCOL_VERSION);

        return $YouTubeService;
    }

    /**
     * Get the base URL where PHP began execution.
     * @return the base URL where PHP began execution
     */
    protected function GetBaseUrl()
    {
        $Protocol = (isset($_SERVER["HTTPS"])) ? "https://" : "http://";
        $Server = $_SERVER["SERVER_NAME"];
        $Path = dirname(htmlentities(
            substr(
                $_SERVER["PHP_SELF"],
                0,
                strcspn($_SERVER["PHP_SELF"], "\n\r")),
            ENT_QUOTES));
        $Url = $Protocol . $Server . $Path;

        return $Url;
    }

    /**
     * Get the XML representation of the fields created by this plugin.
     * @return an array of field XML representations
     */
    protected function GetFieldXml()
    {
        $DirectoryPath = dirname(__FILE__);
        $Directory = dir($DirectoryPath);
        $Xml = array();

        # loop through the files in the current directory
        while (FALSE !== ($Path = $Directory->read()))
        {
            if (preg_match('/YouTube_YouTubeVideo[a-zA-Z]*\.xml/', $Path))
            {
                $FullPath = $DirectoryPath . "/" . $Path;
                $Xml[$Path] = file_get_contents($FullPath);
            }
        }

        return $Xml;
    }

    /**
     * Include the given file in the same directory, buffering its output,
     * and then return that output.
     * @param $FileName name of the file to include (in the same directory)
     * @param $Variables array of variables to put in the scope of the include
     * @return the buffered output
     */
    protected function IncludeAndBuffer($FileName, array $Variables=array())
    {
        return $this->CallAndBuffer(
            array($this, "IncludeAndBufferAux"),
            dirname(__FILE__)."/".$FileName.".php",
            $Variables);
    }

    /**
     * Call the given function and buffer its output, if any. Any additional
     * parameters passed to this method will be passed as parameters to the
     * given function.
     * @param $Function function or method
     * @return the buffered output
     */
    protected function CallAndBuffer($Function)
    {
        ob_start();

        $Parameters = func_get_args();
        array_shift($Parameters);

        call_user_func_array($Function, $Parameters);
        $Buffer = ob_get_contents();

        ob_end_clean();

        return $Buffer;
    }

    /**
     * Auxiliary method for self::IncludeAndBuffer that is used for creating an
     * unpolluted scope.
     * @param $FilePath path to file to include
     * @param $Variables array of variables to put in the scope of the include
     * @return the buffered output
     */
    private function IncludeAndBufferAux()
    {
        extract(func_get_arg(1));
        include(func_get_arg(0));
    }

}
