<?PHP

class YouTube_Zend_Gdata_YouTube extends Zend_Gdata_YouTube
{

    /**
     * @const RESUMABLE_UPLOAD_URI URI for resumable uploads
     */
    const RESUMABLE_UPLOAD_URI = "http://uploads.gdata.youtube.com/resumable/feeds/api/users/default/uploads";

    /**
     * @const VIDEO_URI_PREFIX URI prefix for the video API calls
     */
    const VIDEO_URI_PREFIX = "http://gdata.youtube.com/feeds/api/users/default/uploads/";

    /**
     * Determine if the given video exists on YouTube.
     * @param $video YouTube_Video object
     * @return TRUE if the video exists or FALSE otherwise
     */
    public function videoExists(YouTube_Video $video)
    {
      try {
        $this->getVideoEntry($video->YouTubeId);
        return TRUE;
      } catch (Zend_Gdata_App_InvalidArgumentException $e) {
        // bad YouTube ID, so it doesn't exist
        return FALSE;
      } catch (Exception $e) {
        // assume it still exists if the error message isn't found
        return !is_int(strpos($e->getMessage(), 'Video not found'));
      }
    }

    /**
     * Start a resumable upload for the given video entry.
     * @param $data a media entry object
     * @param $uri optional URI to send the request to
     * @param $extraHeaders optional additional headers to add to the request
     * @return the upload URL for the new resumable upload
     * @throws Zend_Gdata_App_Exception if the request fails
     */
    public function startResumableUpload($data, $uri=null, $extraHeaders = array())
    {
      if (is_null($uri)) {
        $uri = self::RESUMABLE_UPLOAD_URI;
      }

      $requestData = $this->prepareRequest(
        Zend_Http_Client::POST, $uri, $extraHeaders, $data, NULL);

      // modify some of the request data since some if it will cause problems
      // when just initiating an upload
      $requestData['data'] = '';
      $requestData['contentType'] = NULL;
      unset($requestData['headers']['MIME-version']);

      $response = $this->performHttpRequest(
        $requestData['method'], $requestData['url'],
        $requestData['headers'], $requestData['data'],
        $requestData['contentType']);

      if ($response->getStatus() != 200) {
        require_once 'Zend/Gdata/App/Exception.php';
        throw new Zend_Gdata_App_Exception(
          'Resumable upload initiation failed');
      }

      // get the upload URI from the Location header
      $location = $response->getHeader('Location');

      return $location;
    }

    /**
     * Resume an upload.
     * @param $Upload a YouTube_Upload object
     * @return a Zend_Gdata_YouTube_VideoEntry object of the completed upload
     * @throws YouTube_UploadFailureException if the upload fails
     */
    public function resumeUpload(YouTube_Upload $Upload)
    {
        $State = $this->GetUploadState($Upload);
        $HasCompleted = $State->HasCompleted;

        if ($HasCompleted) {
            $Body = $State->ResponseBody;
            $VideoEntry = new Zend_Gdata_YouTube_VideoEntry($Body);
            $YouTubeId = $VideoEntry->getVideoId();

            return $YouTubeId;
        }

        $FileId = $Upload->FileId;
        $File = new File($FileId);
        $Mimetype = $File->GetMimeType();

        try {
            # the byte offsets are zero-based indices
            $FirstByte = $State->LastByteUploaded + 1;

            # get an already-seeked file handle for the File object
            $Handle = $this->PrepareFile($File, $FirstByte);
        }

        # file operations failed
        catch (Exception $Exception) {
            throw new YouTube_UploadFailureException($Upload);
        }

        # uses a custom adapter so that the percent complete value of the
        # upload object can be updated while the upload is in progress
        $Adapter = new YouTube_Zend_Http_Client_Adapter_Socket();
        $Adapter->setUpload($Upload);

        // temporarily replace the HTTP client adapter
        $HttpClient = $this->getHttpClient();
        $OrigAdapter = $HttpClient->getAdapter();
        $HttpClient->setAdapter($Adapter);

        try {
            $Response = $this->performHttpRequest(
                Zend_Http_Client::PUT,
                $Upload->UploadUrl,
                NULL,
                $Handle,
                $Mimetype);
        } catch (Exception $Exception) {
            throw new YouTube_UploadFailureException($Upload);
        }

        // restore the original HTTP client adapter
        $HttpClient->setAdapter($OrigAdapter);

        # update the state of the upload
        $State = $this->GetUploadState($Upload);
        $HasCompleted = $State->HasCompleted;

        # the video should have fully uploaded
        if (!$HasCompleted) {
            throw new YouTube_UploadFailureException($Upload);
        }

        // remove the <yt:incomplete /> element if it exists to avoid issues
        // when trying to use the video entry object. accept any XML namespace
        // since it tends to change
        $Body = preg_replace(
            '/<[a-zA-Z]+:incomplete(\/>|\s+.*?\/>)/',
            '',
            $State->ResponseBody);

        $VideoEntry = new Zend_Gdata_YouTube_VideoEntry($Body);

        // there's a bug in the Zend library somewhere that doesn't set the
        // major protocol version and consequently causes havoc
        $VideoEntry->setMajorProtocolVersion($this->getMajorProtocolVersion());

        return $VideoEntry;
    }

    /**
     * Determine only if the video has completed processing on YouTube's side.
     * @param $video YouTube_Video object
     * @return TRUE if the video has completed processing or FALSE otherwise
     */
    public function isProcessed(YouTube_Video $video)
    {
      // if the video doesn't exist, it can't be processed either
      if (!$this->videoExists($video)) {
        return FALSE;
      }

      // YouTube API call
      $uri = self::VIDEO_URI_PREFIX . $video->YouTubeId;
      $body = $this->getHttpClient($uri)->request()->getBody();

      // if the video has not been processed, it will have an <app:control>
      // element in its XML representation
      $processed = !preg_match('/<[a-z]+:control(\s+[^>]*>|>)/i', $body);

      return $processed;
    }

    /**
     * Delete the given video from YouTube.
     * @param $video YouTube_Video object
     */
    public function deleteVideo(YouTube_Video $video)
    {
      $id = $video->YouTubeId;
      $data = $this->getVideoEntry($id);
      $uri = self::VIDEO_URI_PREFIX . $id;
      $headers = array();

      $requestData = $this->prepareRequest(
        Zend_Http_Client::DELETE,
        $uri,
        NULL,
        $data);

      return $this->performHttpRequest(
        $requestData['method'],
        $requestData['url'],
        $requestData['headers'],
        '',
        $requestData['contentType']);
    }

    /**
     * Get the current state of an upload.
     * @param $Upload a YouTube_Upload object
     * @return a YouTube_UploadState object
     */
    protected function GetUploadState(YouTube_Upload $Upload)
    {
        $UploadUrl = $Upload->UploadUrl;

        # set up an HTTP client
        # http://code.google.com/apis/youtube/2.0/developers_guide_protocol.html#Checking_Resumable_Upload_Status
        $HttpClient = new Zend_Http_Client();
        $HttpClient->setUri($UploadUrl);
        $HttpClient->setMethod(Zend_Http_Client::PUT);
        $HttpClient->setHeaders("Content-Range", "bytes */*");

        # send the request
        $Response = $HttpClient->request();

        $StatusCode = $Response->getStatus();
        $Body = $Response->getBody();
        $HasCompleted = ($StatusCode >= 200 && $StatusCode < 300);
        $LastByteUploaded = -1;

        # if the upload hasn't completed
        if ($StatusCode == 308)
        {
            $Range = $Response->getHeader("range");

            # the range is set only if at least one byte has been uploaded
            if (!is_null($Range))
            {
                list(, $LastByteUploaded) = sscanf($Range, "bytes=%d-%d");
            }
        }

        $Values = array(
            "ResponseStatusCode" => $StatusCode,
            "ResponseBody" => $Body,
            "HasCompleted" => $HasCompleted,
            "LastByteUploaded" => $LastByteUploaded);

        $State = new YouTube_UploadState($Values);

        return $State;
    }

    /**
     * Prepare a File object by getting a file handle for the file it represents
     * and by seeking to the given offset.
     * @param $File File object
     * @param $Offset byte offset (zero-based index)
     * @return a file handle
     */
    protected function PrepareFile(File $File, $Offset)
    {
        $Handle = $this->GetFileHandle($File);

        $this->FileSeek($Handle, $Offset);

        return $Handle;
    }

    /**
     * Get a file handle for the given File object.
     * @param $File File object
     * @return a file handle
     * @throws YouTube_UnreadableFileException if the file is unreadable
     */
    protected function GetFileHandle(File $File)
    {
        $Path = $File->GetNameOfStoredFile();

        if (!is_readable($Path))
        {
            throw new YouTube_UnreadableFileException($File);
        }

        # open the file for reading
        $Handle = fopen($Path, "r");

        return $Handle;
    }

    /**
     * Seek the file offset pointer of the given handle to the given offset.
     * @param $Handle file handle
     * @param $Offset byte offset (zero-based index)
     * @throws YouTube_CouldNotSeekException if the file seek fails
     */
    protected function FileSeek($Handle, $Offset)
    {
        $State = fseek($Handle, $Offset);

        if ($State !== 0)
        {
            throw new YouTube_CouldNotSeekException($File);
        }
    }

}
