<?PHP

#
#   FILE:  SPT--BrowseResources.php
#
#   FUNCTIONS PROVIDED:
#       DisplayClassificationList()
#           - outputs HTML for list of classifications for this page
#       DisplayResourceList()
#           - outputs HTML for list of resources for this page
#       PrintAlphabeticClassificationLinks()
#           - returns HTML for links to classification by starting letter
#       GetClassificationCount()
#           - returns number of classifications on this page
#       GetResourceCount()
#           - returns number of resources on this page
#       PrintRootClassification($ParentId, $LinkStyle)
#           - returns linked HTML for base classification for current page
#       EditingEnabled()
#           - whether user has editing privileges and editing was requested
#       PrintAddClassificationLink()
#           - print link address to add a classification at this point in tree
#       PreviousResourcesAvailable()
#       NextResourcesAvailable()
#           - return TRUE if this is not the first/last page of resources
#       PrintPreviousResourcesLink()
#       PrintNextResourcesLink()
#           - print URL for previous or next page of resources
#       PrintNumberOfPreviousResources()
#       PrintNumberOfNextResources()
#           - print number of resources on previous/next resources page
#
#   FUNCTIONS EXPECTED:
#       MoveToNextClassificationColumn()
#           - called to move to next column in classification HTML table
#       PrintClassificationEntry($ClassName, $LinkAddress, $ResourceCount, 
#                   $EditLinkAddress)
#           - called to display classification entry
#       PrintResourceEntry($Title, $Description, $FullRecordLink, $Url, 
#                   $DateOfRecordCreation, $DateRecordAdded, $Contributor, 
#                   $EditOkay, $EditUrl, $DateLastVerified, $DateLastModified)
#           - called to display resource entry
#
#   Part of the Scout Portal Toolkit
#   Copyright 2004 Internet Scout Project
#   http://scout.wisc.edu
#

require_once("include/SPT--Common.php");
require_once("include/SPT--CommonSearch.php");
require_once("include/SPT--Classification.php");
require_once("include/SPT--Resource.php");
require_once("include/SPT--MetadataSchema.php");

if (EditingEnabled())
{
    PageTitle("Add / Edit Classifications");
}
else
{
    PageTitle("Browse Resources");
}


# ----- EXPORTED FUNCTIONS ---------------------------------------------------

function PrintBrowsingLinks()
{
    $Schema = & new MetadataSchema;

    $Fields = $Schema->GetFields(MDFTYPE_TREE);
    $BrowsingFieldId = GetBrowsingFieldId();

    foreach ($Fields as $Field)
    {
        if ($Field->Id() != $BrowsingFieldId)
        {
            $LinkString = 
                "<a href=\"SPT--BrowseResources.php?FieldId=".$Field->Id()."\">Browse by ".$Field->Name()." Classifications</a><br>";
            print $LinkString;
        }
    }
}

function PrintAlphabeticClassificationLinks()
{
    global $ParentId;
    global $DB;
    global $SysConfig;

    # if classification ID passed in
    if ($ParentId > 0)
    {
        # retrieve link string for classification
        $Class = new Classification($ParentId);
        $LinkString = $Class->LinkString();

        # if link string has not yet been set
        if ($LinkString == "")
        {
            # create and save new link string
            $LinkString = BuildClassificationLinkString($ParentId);
            $Class->LinkString($LinkString);
        }
    }
    # else
    else
    {
        # retrieve top-level link string
        $LinkString = $SysConfig->TopLinkString();

        # if top-level link string has not yet been set
        if ($LinkString == "")
        {
            # create and save new top-level link string
            $LinkString = BuildClassificationLinkString(0);
            $SysConfig->TopLinkString($LinkString);

            # extract and save new default top-level begin and end letters
            global $StartingLetter;
            global $EndingLetter;
            $StartingLetter = "";
            $EndingLetter = "";
            if ((strlen($LinkString) > 0) && ($LinkString != "X"))
            {
                list($StartingLetter, $EndingLetter) = 
                        explode("&", $LinkString);
                list($Dummy, $Dummy, $StartingLetter) = 
                        explode("=", $StartingLetter);
                list($Dummy, $EndingLetter) = 
                        explode("=", $EndingLetter);
            }
            $SysConfig->TopLinkBeginLetter($StartingLetter);
            $SysConfig->TopLinkEndLetter($EndingLetter);
        }
    }

    # if link string is placeholder
    if ($LinkString == "X")
    {
        # clear link string
        $LinkString = "";
    }
    # else
    else
    {
        # if link string is not empty
        if ($LinkString != "")
        {
            # insert target browse page name into link string
            $LinkString = preg_replace(
                    "/BROWSEPAGE/", "SPT--BrowseResources.php", $LinkString);

            # insert editing flag value into link string
            $LinkString = preg_replace(
                    "/EDITFLAG/", (EditingEnabled() ? "1" : "0"), $LinkString);
        }
    }

    # return link string to caller
    return $LinkString;
}

function GetTreeName()
{
    global $Schema;
    
    $Field = $Schema->GetField(GetBrowsingFieldId());
    return $Field->Name();
}

function PrintTreeName()
{
    print GetTreeName();
}

function GetClassificationCount()
{
    global $ParentId;
    global $StartingLetter;
    global $EndingLetter;

    $ClassDB =& new SPTDatabase();

    $ClassDB->Query(GetClassificationDBQuery(
            $ParentId, $StartingLetter, $EndingLetter));

    # retrieve count of entries to be displayed
    return $ClassDB->NumRowsSelected();
}

function DisplayClassificationList()
{
    global $NumberOfColumns;
    global $MinEntriesPerColumn;
    global $ParentId;
    global $StartingLetter;
    global $EndingLetter;

    $ClassDB =& new SPTDatabase();

    # retrieve entries to be displayed
    $ClassDB->Query(GetClassificationDBQuery(
            $ParentId, $StartingLetter, $EndingLetter));

    # retrieve count of entries to be displayed
    $RecordCount = $ClassDB->NumRowsSelected();

    # for each entry
    $ClassCount = 0;
    while ($Class = $ClassDB->FetchNextRowArray())
    {
        # if filter function defined
        $ClassId = $Class["ClassificationId"];
        if (function_exists("FilterClassificationBrowseList"))
        {
            # call filter function to find out if okay to display entry
            $DoNotDisplay = FilterClassificationBrowseList($ClassId);
        }
        else
        {
            # assume okay to display entry
            $DoNotDisplay = FALSE;
        }

        # if okay to display entry
        if ($DoNotDisplay == FALSE)
        {
            # if entries per column limit reached
            $ClassCount++;
            if (($ClassCount > (int)($RecordCount / $NumberOfColumns))
                    && ($ClassCount > $MinEntriesPerColumn))
            {
                # move to next column
                MoveToNextClassificationColumn();
                $ClassCount = 0;
            }
        
            # construct link address
            $LinkUrl = sprintf("SPT--BrowseResources.php?ParentId=%d",
                    $Class["ClassificationId"]);
            if (EditingEnabled()) {  $LinkUrl .= "&Editing=1";  }

            # construct link address for editing
            $EditLinkUrl = sprintf("MetadataTool/SPT--EditClassification.php?ClassificationId=%d",
                    $Class["ClassificationId"]);

            # print entry
            PrintClassificationEntry(stripslashes($Class["SegmentName"]), $LinkUrl, 
                    $Class["ResourceCount"], $EditLinkUrl);
        }
    }
}

function PrintRootClassification($LinkStyle = "")
{
    global $ParentId;

    # print root classification HTML string
    print(GetRootClassification($ParentId, $LinkStyle=""));
}

function EditingEnabled()
{
    global $User;
    global $Editing;

    if (($Editing == 1) && $User->HasPriv(PRIV_CLASSADMIN)) 
            {  return true;  } else {  return false;  }
}

function PrintAddClassificationLink()
{
    global $ParentId;
    global $SysConfig;

    print("MetadataTool/SPT--AddClassification.php?ParentId=".$ParentId
            ."&FieldId=".GetBrowsingFieldId());
}

function GetResourceCount()
{
    static $ResourceCount;

    # if we have not already calculated count of resources
    if (isset($ResourceCount) == FALSE)
    {
        global $ParentId;

        # if resource browsing filter function is defined
        $ClassId = $ParentId;
        $DB =& new SPTDatabase();
        if (function_exists("FilterResourceBrowseList"))
        {
            # call filter func for each resource to determine count
            $ResourceCount = 0;
            $DB->Query("SELECT Resources.ResourceId AS RsrcId "
                    ."FROM Resources, ResourceClassInts "
                    ."WHERE ResourceClassInts.ClassificationId = ".$ClassId." "
                            ."AND Resources.ResourceId = ResourceClassInts.ResourceId "
                            ."AND Resources.ReleaseFlag = 1");
            while ($ResourceId = $DB->FetchField("RsrcId"))
            {
                if (FilterResourceBrowseList($ResourceId) == FALSE)
                {
                    $ResourceCount++;
                }
            }
        }
        else
        {
            # retrieve count via DB query
            $ResourceCount = $DB->Query("SELECT COUNT(*) AS ResourceCount "
                    ."FROM Resources, ResourceClassInts "
                    ."WHERE ResourceClassInts.ClassificationId = ".$ClassId." "
                            ."AND Resources.ResourceId = ResourceClassInts.ResourceId "
                            ."AND Resources.ReleaseFlag = 1",
                    "ResourceCount");
        }
    }

    # return count to caller
    return $ResourceCount;
}

function PreviousResourcesAvailable()
{
    global $StartingResourceIndex;
    return ($StartingResourceIndex > 0) ? TRUE : FALSE;
}

function NextResourcesAvailable()
{
    global $StartingResourceIndex;
    global $MaxResourcesPerPage;

    if (($StartingResourceIndex + $MaxResourcesPerPage) < GetResourceCount())
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}

function PrintPreviousResourcesLink()
{
    global $StartingResourceIndex;
    global $MaxResourcesPerPage;
    global $ParentId;

    $Url = "SPT--BrowseResources.php?ParentId=".$ParentId
            ."&StartingResourceIndex="
                    .($StartingResourceIndex - $MaxResourcesPerPage);
    if (EditingEnabled()) {  $Url .= "&Editing=1";  }
    print($Url);
}

function PrintNextResourcesLink()
{
    global $StartingResourceIndex;
    global $MaxResourcesPerPage;
    global $ParentId;

    $Url = "SPT--BrowseResources.php?ParentId=".$ParentId
            ."&StartingResourceIndex="
                    .($StartingResourceIndex + $MaxResourcesPerPage);
    if (EditingEnabled()) {  $Url .= "&Editing=1";  }
    print($Url);
}

function PrintNumberOfPreviousResources()
{
    global $MaxResourcesPerPage;
    print($MaxResourcesPerPage);
}

function PrintNumberOfNextResources()
{
    global $MaxResourcesPerPage;
    global $StartingResourceIndex;
    print(min($MaxResourcesPerPage, 
            (GetResourceCount() - ($StartingResourceIndex + $MaxResourcesPerPage))));
}

function DisplayResourceList()
{
    global $NumberOfColumns;
    global $MinEntriesPerColumn;
    global $ParentId;
    global $StartingLetter;
    global $EndingLetter;
    global $User;
    global $StartingResourceIndex;
    global $MaxResourcesPerPage;

    $ResourceDB =& new SPTDatabase();

    # retrieve entries to be displayed
    $ResourceDB->Query("SELECT * FROM Resources, ResourceClassInts "
            ."WHERE ResourceClassInts.ClassificationId = ".$ParentId." "
                    ."AND Resources.ResourceId = ResourceClassInts.ResourceId "
                    ."AND Resources.ReleaseFlag = 1 "
            ."ORDER BY Resources.Title");

    # for each entry
    $ResourceIndex = 0;
    while ($Record = $ResourceDB->FetchNextRowArray())
    {
        # if filter function defined
        $ResourceId = $Record["ResourceId"];
        if (function_exists("FilterResourceBrowseList"))
        {
            # call filter function to find out if okay to display entry
            $DoNotDisplay = FilterResourceBrowseList($ResourceId);
        }
        else
        {
            # assume okay to display entry
            $DoNotDisplay = FALSE;
        }

        # if okay to display entry
        if ($DoNotDisplay == FALSE)
        {
            # if within resource range for this page
            if (($ResourceIndex >= $StartingResourceIndex)
                    && ($ResourceIndex < ($StartingResourceIndex + $MaxResourcesPerPage)))
            {
                # print entry
                $ResourceId = $Record["ResourceId"];
                $Resource =& new Resource($ResourceId);
                PrintResourceEntry($Resource,
                        "SPT--FullRecord.php?ResourceId=${ResourceId}",
                        CheckEditResource( $Resource ),
"MetadataTool/SPT--DBEntry.php?ResourceId=${ResourceId}",
                        $Resource->ScaledCumulativeRating());
            }

            # increment count of resources displayed
            $ResourceIndex++;
        }
    }
}


# ----- LOCAL FUNCTIONS ------------------------------------------------------

function GetBrowsingFieldId()
{
    global $SysConfig;
    global $User;
    global $BrowsingFieldId;
    global $Editing;

    if (isset($BrowsingFieldId) && $Editing)
        $FieldId = $BrowsingFieldId;
    else if ($User->IsLoggedIn())
    {
        $FieldId = $User->Get("BrowsingFieldId");
        if (empty($FieldId))
            $FieldId = $SysConfig->BrowsingFieldId();
    }
    else
        $FieldId = $SysConfig->BrowsingFieldId();
    
    return $FieldId; 
}

function BuildClassificationLinkString($ParentId)
{
    global $SysConfig;

    $DB =& new SPTDatabase();

    $MaxClassesPerPage = $SysConfig->NumClassesPerBrowsePage();
    if ($MaxClassesPerPage < 1) {  $MaxClassesPerPage = 1;  }

    # load classification names from database
    $DB->Query("SELECT ClassificationName FROM Classifications "
               ."WHERE FieldId="
               .GetBrowsingFieldId()." "
                       ."AND ".(($ParentId > 0) ? "ParentId=".$ParentId." " : "Depth=0 "
                       ."AND ResourceCount != 0 ")
               ."ORDER BY ClassificationName ");
    $TotalClasses = 0;
    while ($Row = $DB->FetchNextRowArray())
    {
        $ClassName = trim(stripslashes($Row["ClassificationName"]));
        if ($ClassName != "")
        {
            $ClassName = preg_replace("/[^A-Za-z]/", "", $ClassName);
            $ClassNames[$TotalClasses] = ucfirst(strtolower($ClassName));
            $TotalClasses++;  
        }
    }

    # if all classes will fit on a single page
    if ($TotalClasses <= $MaxClassesPerPage)
    {
        # set link string to null value
        $LinkString = "X";
    }
    else
    {
        # while not end of classifications
        $Offset = 0;
        $ClassesPerPage = $MaxClassesPerPage;
        $StartingLetter = "A";
        while ($Offset < $TotalClasses)
        {
            # if starting letter is one letter
            if (strlen($StartingLetter) == 1)
            {
                # if starting letter has more classes than will fit on one page
                $ClassCount = $DB->Query("SELECT COUNT(*) AS ClassCount FROM Classifications "
                        ."WHERE FieldId="
                        .GetBrowsingFieldId()." "
                        ."AND ".(($ParentId > 0) ? "ParentId=".$ParentId." " : "Depth=0 "
                        ."AND ClassificationName>='".$StartingLetter."' "
                        ."AND ClassificationName<='".$StartingLetter."zzzzzzzzzz' "),
                        "ClassCount");
                if ($ClassCount > $MaxClassesPerPage)
                {
                    # calculate classes per page based on total classes for letter
                    $ClassesPerPage = intval($ClassCount / ceil($ClassCount / $MaxClassesPerPage));
    
                    # reset starting letter to be first two-letter combo for letter
                    $StartingLetter .= "a";
                }
            }
    
            # calc new offset
            $Offset += $ClassesPerPage;
    
            # if starting letter is two letters 
            if (strlen($StartingLetter) == 2) 
            {
                # if first letter of offset class is same as starting letter
                if (ord($ClassNames[$Offset]) == ord($StartingLetter))
                {
                    # if second letter of offset class is different from starting letter
                    $OffsetClassLetter = substr($ClassNames[$Offset], 1, 1);
                    if (ord($OffsetClassLetter) 
                            > ord(substr($ClassNames[$Offset - $ClassesPerPage], 1, 1)))
                    {
                        # step offset backward until second letter of class changes
                        do
                        {
                            $Offset--;
                            $NewOffsetClassLetter = substr($ClassNames[$Offset], 1, 1);
                        } while ($NewOffsetClassLetter == $OffsetClassLetter);
                    }

                    # set ending letter to first two letters of class at offset
                    $EndingLetter = substr($ClassNames[$Offset], 0, 2);
                }
                else
                {
                    # step offset backward until first letter is same as starting letter
                    do
                    {
                        $Offset--;
                    } while (ord($ClassNames[$Offset]) != ord($StartingLetter));
    
                    # set ending letter to first letter of class at offset plus "z"
                    $EndingLetter = substr($ClassNames[$Offset], 0, 1)."z";
    
                    # reset classes per page value
                    $ClassesPerPage = $MaxClassesPerPage;
                }
            }
            # else
            else
            {
                # step offset backward until first letter of class changes
                $OffsetClassLetterVal = ord(substr($ClassNames[$Offset], 0, 1));
                do
                {
                    $Offset--;
                } while (ord($ClassNames[$Offset]) == $OffsetClassLetterVal);
    
                # if starting letter is two letters
                if (strlen($StartingLetter) == 2) 
                {
                    # set ending letter to last two-letter combo for letter
                    $EndingLetter = substr($StartingLetter, 0, 1)."z";
    
                    # reset classes per page value
                    $ClassesPerPage = $MaxClassesPerPage;
                }
                # else
                else
                {
                    # set ending letter to first letter of class at offset
                    $EndingLetter = substr($ClassNames[$Offset], 0, 1);
                }
            }
    
            # add startng and ending letter to link string
            if ($StartingLetter == $EndingLetter)
            {
                $LinkString .= " <a href='BROWSEPAGE"
                        ."?StartingLetter=${StartingLetter}"
                        ."&EndingLetter=${EndingLetter}"
                        ."&Editing=EDITFLAG'>${StartingLetter}</a>";
            }
            else
            {
                $LinkString .= " <a href='BROWSEPAGE"
                        ."?StartingLetter=${StartingLetter}"
                        ."&EndingLetter=${EndingLetter}"
                        ."&Editing=EDITFLAG'>${StartingLetter}-${EndingLetter}</a>";
            }
    
            # step offset forward one
            $Offset++;
    
            # if ending letter is one letter or is last two-letter combo
            if ((strlen($EndingLetter) == 1) || (substr($EndingLetter, 1, 1) == "z"))
            {
                # set starting letter to next letter after ending letter
                $StartingLetter = chr(ord($EndingLetter)+1);
            }
            else
            {
                # set starting letter to next two-letter combo after ending letter
                $StartingLetter = substr($EndingLetter, 0, 1)
                        .chr(ord(substr($EndingLetter, 1, 1))+1);
            }
        }
    }
    return $LinkString;
}

function GetClassificationDBQuery($ParentId, $StartingLetter, $EndingLetter)
{
    global $SysConfig;

    if ($ParentId > 0)
    {
        $QueryString = "SELECT * FROM Classifications "
                ."WHERE ParentId=".$ParentId." AND FieldId="
                .GetBrowsingFieldId()." "
                .(EditingEnabled() ? "" : "AND ResourceCount != 0 ")
                ."ORDER BY ClassificationName";
    }
    else
    {
        $QueryString = "SELECT * FROM Classifications "
                ."WHERE Depth=0  AND FieldId="
                .GetBrowsingFieldId()." "
                        ."AND ClassificationName >= '".$StartingLetter."' "
                        ."AND ClassificationName <= '".$EndingLetter."ZZZZZZZZZZ' "
                        .(EditingEnabled() ? "" : "AND ResourceCount != 0 ")
                ."ORDER BY ClassificationName";
    }
    return $QueryString;
}

function GetResourceDBQuery($ClassId)
{
    $QueryString = "SELECT * FROM Resources, ResourceClassInts "
            ."WHERE ResourceClassInts.ClassificationId = ".$ClassId." "
                    ."AND Resources.ResourceId = ResourceClassInts.ResourceId "
                    ."AND Resources.ReleaseFlag = 1 "
            ."ORDER BY Resources.Title ";

    return $QueryString;
}

function GetRootClassification($ParentId, $LinkStyle="")
{
    # start with empty string
    $RootClassString = "";

    # if top of classification tree specified
    if ($ParentId > 0)
    {
        # do while classification segments left to add
        $ClassDB =& new SPTDatabase();
        do
        {
            # if not first segment in string
            if ($RootClassString != "")
            {
                # add segment separator to string
                $RootClassString = " -- ".$RootClassString;
            }

            # get current segment name
            $Class =& new Classification($ParentId);

            # add current segment to string
            $RootClassString = "<a href='SPT--BrowseResources.php?ParentId="
                    .$ParentId
                    .(EditingEnabled() ? "&Editing=1" : "")
                    ."' class='".$LinkStyle."'>"
                    .$Class->SegmentName()."</a>"
                    .$RootClassString;

            # move to next segment
            $ParentId = $Class->ParentId();

        } while ($ParentId > 0);
    }

    # return root classification HTML string to caller
    return $RootClassString;
}


# ----- MAIN -----------------------------------------------------------------

# set up default display parameters
$NumberOfColumns = $SysConfig->NumColumnsPerBrowsePage();
$MinEntriesPerColumn = 3;
$MaxResourcesPerPage = 10;

if (isset($HTTP_GET_VARS["Editing"]))
    $Editing = $HTTP_GET_VARS["Editing"];
else
    $Editing = 0;

$Schema = & new MetadataSchema();

if (isset($HTTP_POST_VARS["Submit"]))
{
    $BrowsingFieldId = $HTTP_POST_VARS["Submit"];
}
else
    $BrowsingFieldId = $Session->Get("BrowsingFieldId");

# implicitly set the current Browsing hierarchy if not editing
if (!$Editing && isset($HTTP_GET_VARS["FieldId"]))
{
    $BrowsingFieldId = $HTTP_GET_VARS["FieldId"];
    $User->Set("BrowsingFieldId", $BrowsingFieldId);
}
$Session->PassVariable("BrowsingFieldId", $BrowsingFieldId);

if (isset($HTTP_GET_VARS["StartingResourceIndex"]))
    $StartingResourceIndex = $HTTP_GET_VARS["StartingResourceIndex"];
else
    $StartingResourceIndex = 0;

# set to stored system default browse range
if (isset($HTTP_GET_VARS["StartingLetter"]))
    $StartingLetter = $HTTP_GET_VARS["StartingLetter"];
else
    $StartingLetter = NULL;

if (isset($HTTP_GET_VARS["EndingLetter"]))
    $EndingLetter = $HTTP_GET_VARS["EndingLetter"];
else
    $EndingLetter = NULL;
    
if (!strlen($StartingLetter))
{
    $StartingLetter = $SysConfig->TopLinkBeginLetter();  
    $EndingLetter = $SysConfig->TopLinkEndLetter();  
}

if (!strlen($StartingLetter)) {  $StartingLetter = "A";  }
if (!strlen($EndingLetter))   {  $EndingLetter = "Z";  }
if (empty($ParentId))         {  $ParentId = -1;  }

include("include/SPT--StandardHtmlPageLoad.php");


?>
