<?PHP

#
#   FILE:  SPT--SearchEngine.php
#
#   FUNCTIONS PROVIDED:
#       SPTSearchEngine->SPTSearchEngine()
#           - constructor
#       (see Scout--SearchEngine.php for other public methods)
#
#   AUTHOR:  Edward Almasy
#
#   Part of the Scout Portal Toolkit
#   Copyright 2002-2004 Internet Scout Project
#   http://scout.wisc.edu
#

require_once(dirname(__FILE__)."/SPT--SPTDate.php");
require_once(dirname(__FILE__)."/../Scout--SearchEngine.php");

require_once(dirname(__FILE__)."/SPT--SPTDatabase.php");
require_once(dirname(__FILE__)."/SPT--Resource.php");
require_once(dirname(__FILE__)."/SPT--MetadataSchema.php");

class SPTSearchEngine extends SearchEngine {
    
    function SPTSearchEngine()
    {
        # create a database handle
        $DB =& new SPTDatabase();
        
        # pass database handle and config values to real search engine object
        $this->SearchEngine($DB, "Resources", "ResourceId");
        
        # for each field defined in schema
        $this->Schema =& new MetadataSchema();
        $Fields = $this->Schema->GetFields();
        foreach ($Fields as $Field)
        {
            # determine field type for searching
            switch ($Field->Type())
            {
                case MDFTYPE_TEXT:
				case MDFTYPE_PARAGRAPH:
				case MDFTYPE_USER:
                case MDFTYPE_TREE:
                case MDFTYPE_CONTROLLEDNAME:
                case MDFTYPE_OPTION:
                case MDFTYPE_IMAGE:
                    $FieldType = SEARCHFIELD_TEXT;
                    break;

				case MDFTYPE_NUMBER:
                case MDFTYPE_FLAG:
                    $FieldType = SEARCHFIELD_NUMERIC;
                    break;

				case MDFTYPE_DATE:
                    $FieldType = SEARCHFIELD_DATERANGE;
                    break;

				case MDFTYPE_TIMESTAMP:
                    $FieldType = SEARCHFIELD_DATE;
                    break;
                    
                default:
                    exit("ERROR: unknown field type in SPT--SearchEngine.php");
                    break;
            }
            
            # add field to search engine
            $this->AddField($Field->Name(), $Field->DBFieldName(), $FieldType,
                            $Field->SearchWeight(), $Field->IncludeInKeywordSearch());
        }
    }

    # overloaded version of method to retrieve text from DB
    function GetFieldContent($ItemId, $FieldName)
    {
        # get resource object
        $Item =& new Resource($ItemId);

        # retrieve text (including variants) from resource object and return to caller
        return $Item->Get($FieldName, FALSE, TRUE);
    }

    # overloaded version of method to retrieve resource/phrase match list
    function SearchFieldForPhrases($FieldName, $Phrase)
    {
        # normalize and escape search phrase for use in SQL query
        $SearchPhrase = strtolower(addslashes($Phrase));

        # query DB for matching list based on field type
        $Field = $this->Schema->GetFieldByName($FieldName);
        switch ($Field->Type())
        {
            case MDFTYPE_TEXT:
            case MDFTYPE_PARAGRAPH:
                $this->DB->Query("SELECT ResourceId FROM Resources "
                        ."WHERE POSITION('".$SearchPhrase."' IN LOWER(".$Field->DBFieldName().")) ");
                break;

            case MDFTYPE_IMAGE:
                $this->DB->Query("SELECT ResourceId FROM Resources "
                        ."WHERE POSITION('".$SearchPhrase."' IN LOWER(".$Field->DBFieldName()."AltText)) ");
                break;

            case MDFTYPE_CONTROLLEDNAME:
            case MDFTYPE_OPTION:
                $this->DB->Query("SELECT DISTINCT ResourceNameInts.ResourceId "
                        ."FROM ResourceNameInts, ControlledNames, VariantNames "
                        ."WHERE (POSITION('".$SearchPhrase."' IN LOWER(ControlledName)) "
                            ."OR (POSITION('".$SearchPhrase."' IN LOWER(VariantName)) "
                                 ."AND (VariantNames.ControlledNameId = ControlledNames.ControlledNameId))) "
                        ."AND ControlledNames.ControlledNameId = ResourceNameInts.ControlledNameId "
                        ."AND ControlledNames.FieldId = ".$Field->Id());
                break;

            case MDFTYPE_TREE:
                $this->DB->Query("SELECT DISTINCT ResourceClassInts.ResourceId "
                        ."FROM ResourceClassInts, Classifications "
                        ."WHERE POSITION('".$SearchPhrase."' IN LOWER(ClassificationName)) "
                        ."AND Classifications.ClassificationId = ResourceClassInts.ClassificationId "
                        ."AND Classifications.FieldId = ".$Field->Id());
                break;
                
            case MDFTYPE_USER:
                $UserId = $this->DB->Query("SELECT UserId FROM APUsers "
                                           ."WHERE POSITION('".$SearchPhrase."' IN LOWER(UserName)) "
                                           ."OR POSITION('".$SearchPhrase."' IN LOWER(RealName))", "UserId");
                if ($UserId != NULL)
                {
                    $this->DB->Query("SELECT ResourceId FROM Resources "
                                     ."WHERE ".$Field->DBFieldName()." = ".$UserId);
                }
                break;

            case MDFTYPE_NUMBER:
                if ($SearchPhrase > 0)
                {
                    $this->DB->Query("SELECT ResourceId FROM Resources "
                                     ."WHERE ".$Field->DBFieldName()." = ".(int)$SearchPhrase);
                }
                break;
                
            case MDFTYPE_FLAG:
            case MDFTYPE_DATE:
            case MDFTYPE_TIMESTAMP:
                # (these types not yet handled by search engine for phrases)
                break;
        }

        # build match list based on results returned from DB
        $MatchList = array();
        while ($Record = $this->DB->FetchRow())
        {
            $MatchList[] = $Record["ResourceId"];
        }

        # return list of matching resources to caller
        return $MatchList;
    }
    
    # search field for records that meet comparison
    function SearchFieldsForComparisonMatches($FieldNames, $Operators, $Values)
    {
        # use SQL keyword appropriate to current search logic for combining operations
        $CombineWord = ($this->DefaultSearchLogic == SEARCHLOGIC_AND) ? " AND " : " OR ";
        
        # for each comparison
        foreach ($FieldNames as $Index => $FieldName)
        {
            $Operator = $Operators[$Index];
            $Value = $Values[$Index];
            
            # determine query based on field type
            $Field = $this->Schema->GetFieldByName($FieldName);
            if ($Field != NULL)
            {
                switch ($Field->Type())
                {
                    case MDFTYPE_TEXT:
                    case MDFTYPE_PARAGRAPH:
                    case MDFTYPE_NUMBER:
                    case MDFTYPE_FLAG:
                        if (isset($Queries["Resources"]))
                        {
                            $Queries["Resources"] .= $CombineWord;
                        }
                        else
                        {
                            $Queries["Resources"] = "SELECT ResourceId FROM Resources WHERE ";
                        }
                        $Queries["Resources"] .= $Field->DBFieldName()." ".$Operator." '".addslashes($Value)."' ";
                        break;

                    case MDFTYPE_CONTROLLEDNAME:
                    case MDFTYPE_OPTION:
                        $QueryIndex = "ResourceNameInts".$Field->Id();
                        if (!isset($Queries[$QueryIndex]))
                        {
                            $Queries[$QueryIndex] = 
                                    "SELECT ResourceId FROM ResourceNameInts, ControlledNames, VariantNames "
                                    ." WHERE ControlledNames.FieldId = ".$Field->Id()
                                    ." AND ( ";
                            $CloseQuery[$QueryIndex] = TRUE;
                        }
                        else
                        {
                            $Queries[$QueryIndex] .= $CombineWord;
                        }
                        $Queries[$QueryIndex] .= "((ResourceNameInts.ControlledNameId = ControlledNames.ControlledNameId"
                                                       ." AND ControlledName ".$Operator." '".addslashes($Value)."')"
                                                   ." OR "
                                                   ."(ResourceNameInts.ControlledNameId = VariantNames.ControlledNameId"
                                                       ." AND VariantName ".$Operator." '".addslashes($Value)."'))";
                        break;

                    case MDFTYPE_TREE:
                        $QueryIndex = "ResourceClassInts".$Field->Id();
                        if (!isset($Queries[$QueryIndex]))
                        {
                            $Queries[$QueryIndex] = "SELECT ResourceId FROM ResourceClassInts, Classifications "
                                                 ." WHERE ResourceClassInts.ClassificationId = Classifications.ClassificationId"
                                                 ." AND Classifications.FieldId = ".$Field->Id()." AND ( ";
                            $CloseQuery[$QueryIndex] = TRUE;
                        }
                        else
                        {
                            $Queries[$QueryIndex] .= $CombineWord;
                        }
                        $Queries[$QueryIndex] .= " ClassificationName ".$Operator." '".addslashes($Value)."'";
                        break;

                    case MDFTYPE_TIMESTAMP:
                        if (isset($Queries["Resources"]))
                        {
                            $Queries["Resources"] .= $CombineWord;
                        }
                        else
                        {
                            $Queries["Resources"] = "SELECT ResourceId FROM Resources WHERE ";
                        }
                        $Date = new Date($Value);
                        $Queries["Resources"] .= " ( ".$Date->SqlCondition($Field->DBFieldName(), NULL, $Operator)." ) ";
                        break;

                    case MDFTYPE_DATE:
                        if (isset($Queries["Resources"]))
                        {
                            $Queries["Resources"] .= $CombineWord;
                        }
                        else
                        {
                            $Queries["Resources"] = "SELECT ResourceId FROM Resources WHERE ";
                        }
                        $Date = new Date($Value);
                        $Queries["Resources"] .= " ( ".$Date->SqlCondition(
                                $Field->DBFieldName()."Begin", $Field->DBFieldName()."End", $Operator)." ) ";
                        break;

                    case MDFTYPE_USER:
                    case MDFTYPE_IMAGE:
                        # (these types not yet handled by search engine for comparisons)
                        break;
                }
            }
        }
        
        # if queries found
        if (isset($Queries))
        {
            # for each assembled query
            foreach ($Queries as $QueryIndex => $Query)
            {
                # add closing paren if query was flagged to be closed
                if (isset($CloseQuery[$QueryIndex]))
                {
                    $Query .= " ) ";
                }
                
                # perform query and retrieve IDs
                if ($this->DebugLevel > 5) {  print("SE:  performing comparison query (<i>".$Query."</i>)<br>\n");  }
                $this->DB->Query($Query);
                $ResourceIds = $this->DB->FetchColumn("ResourceId");
                if ($this->DebugLevel > 5) {  print("SE:  comparison query produced <i>".count($ResourceIds)."</i> results<br>\n");  }
                
                # if we already have some results
                if (isset($Results))
                {
                    # if search logic is set to AND
                    if ($this->DefaultSearchLogic == SEARCHLOGIC_AND)
                    {
                        # remove anything from results that was not returned from query
                        $Results = array_intersect($Results, $ResourceIds);
                    }
                    else
                    {
                        # add values returned from query to results
                        $Results = array_unique(array_merge($Results, $ResourceIds));
                    }
                }
                else
                {
                    # set results to values returned from query
                    $Results = $ResourceIds;
                }
            }
        }
        else
        {
            # initialize results to empty list
            $Results = array();
        }
        
        # return results to caller
        return $Results;
    }

    var $Schema;

    # functions for backward compatability w/ old SPT code
    function UpdateForResource($ItemId) {  $this->UpdateForItem($ItemId);  }
}


?>
