<?php
#
#   FILE:  SPT--UserAgentSearch.php
#
#   FUNCTIONS PROVIDED:
#       PrintUserAgentSearchEntries()
#           - prints entries found as a result of an advanced search
#
#   FUNCTIONS EXPECTED:
#
#   Part of the Scout Portal Toolkit
#   Copyright 2004 Internet Scout Project
#   http://scout.wisc.edu
#

require_once("include/SPT--Common.php");
require_once("include/SPT--CommonSearch.php");
require_once("include/SPT--Resource.php");
require_once("include/SPT--SearchEngine.php");
require_once("include/SPT--MetadataSchema.php");
require_once("include/SPT--ControlledName.php");

PageTitle("UserAgent Search Results");

# ----- EXPORTED FUNCTIONS ---------------------------------------------------

# ----- LOCAL FUNCTIONS ------------------------------------------------------

# callback function for generating replacement text for keywords in e-mail message templates
function KeywordReplacementCallback($Matches)
{
    global $SysConfig;
    global $RCResourceIds, $RCUser, $RCSearchName;

    static $CurrentResourceId;
    static $Resources;
    static $FieldNameMappings;
    static $InSearchResults = FALSE;
    static $ResultNumber;
    
    # load record template (if not already loaded)
    static $EmailResourceTemplate;
    if (!isset($EmailResourceTemplate))
    {
        $EmailResourceTemplate = file("SPT--SearchMailTemplate--Resource.txt");
    }
    
    # switch on match string
    switch ($Matches[0])
    {
        case "X-PORTALNAME-X":
            $Replacement = $SysConfig->PortalName();
            break;
            
        case "X-ADMINEMAIL-X":
            $Replacement = $SysConfig->AdminEmail();
            break;
            
        case "X-LEGALNOTICE-X":
            $Replacement = $SysConfig->LegalNotice();
            break;
            
        case "X-USERLOGIN-X":
            $Replacement = $RCUser->Get("UserName");
            break;
            
        case "X-USERREALNAME-X":
            $Replacement = $RCUser->Get("RealName");
            break;
            
        case "X-USEREMAIL-X":
            $Replacement = $RCUser->Get("Email");
            break;
            
        case "X-DATE-X":
            $Replacement = date("M j Y");
            break;
            
        case "X-TIME-X":
            $Replacement = date("g:ia T");
            break;

        case "X-RESULTCOUNT-X":
            $Replacement = count($RCResourceIds);
            break;

        case "X-SEARCHNAME-X":
            $Replacement = $RCSearchName;
            break;

        case "X-SEARCHRESULTS-X":
            $Replacement = "";
            if ($InSearchResults == FALSE)
            {
                $InSearchResults = TRUE;
                $ResultNumber = 1;
                foreach ($RCResourceIds as $ResourceId)
                {
                    $CurrentResourceId = $ResourceId;
                    if (!isset($Resources[$ResourceId]))
                    {
                         $Resources[$ResourceId] = new Resource($ResourceId);
                    }
                    foreach ($EmailResourceTemplate as $ResourceTemplateLine)
                    {
                        if (!preg_match("/^#/", $ResourceTemplateLine))
                        {
                            $Replacement .= preg_replace_callback("/X-[A-Z0-9:]+-X/", "KeywordReplacementCallback", $ResourceTemplateLine);
                        }
                    }
                    $ResultNumber++;
                }
                $InSearchResults = FALSE;
            }
            break;

        case "X-RESULTNUMBER-X":
            $Replacement = $ResultNumber;
            break;

        default:
            # load field name mappings (if not already loaded)
            if (!isset($FieldNameMappings))
            {
                $Schema = new MetadataSchema();
                $Fields = $Schema->GetFields();
                foreach ($Fields as $Field)
                {
                    $NormalizedName = strtoupper(preg_replace("/[^A-Za-z0-9]/", "", $Field->Name()));
                    $FieldNameMappings[$NormalizedName] = $Field->Name();
                }
            }

            # if keyword refers to known field and we have a current resource
            $KeywordIsField = preg_match("/X-FIELD:([A-Z0-9]+)-X/", $Matches[0], $SubMatches);
            if ($KeywordIsField && isset($FieldNameMappings[$SubMatches[1]]) 
                    && isset($Resources[$CurrentResourceId]))
            {
                # replacement is value from current resource
                $Replacement = $Resources[$CurrentResourceId]->Get($FieldNameMappings[$SubMatches[1]]);
                if (is_array($Replacement))
                {
                    foreach ($Replacement as $ReplacementEntry)
                    {
                        if (!isset($RebuiltReplacement))
                        {
                            $RebuiltReplacement = $ReplacementEntry;
                        }
                        else
                        {
                            $RebuiltReplacement .= ", ".$ReplacementEntry;
                        }
                    }
                    $Replacement = isset($RebuiltReplacement) ? $RebuiltReplacement : "";
                }
                $Replacement = wordwrap(strip_tags($Replacement), 78);
            }
            else
            {
                # if none of our keywords were found, return the match unchanged
                $Replacement = $Matches[0];
            }
            break;
    }
    
    # return replacement string to caller
    return $Replacement;
}

# generate search results e-mail message body and return to caller
function GenerateSearchResultEmailBody($SearchName, $ResourceIds, $User)
{
    # pass values to callback function via global variables
    global $RCResourceIds, $RCUser, $RCSearchName;
    $RCResourceIds = $ResourceIds;
    $RCUser = $User;
    $RCSearchName = $SearchName;
    
    # start with empty message body
    $Body = "";
    
    # load search result e-mail templates
    $EmailBodyTemplate = file("SPT--SearchMailTemplate--Body.txt");
    
    # for each line of body template
    foreach ($EmailBodyTemplate as $BodyTemplateLine)
    {
        # if line does not look like a comment
        if (!preg_match("/^#/", $BodyTemplateLine))
        {
            # replace any keywords in template line with appropriate content and add line to body
            $Body .= preg_replace_callback("/X-[A-Z0-9]+-X/", "KeywordReplacementCallback", $BodyTemplateLine);
        }
    }
    
    # return message body to caller
    return $Body;
}

# email resource entry found as a result of a user agent search 
function EmailSearchResults($SearchResults, $EndUser, $SearchName)
{
    global $SysConfig;

    # generate body and subject of e-mail message
    $BodyLines = split("\n", GenerateSearchResultEmailBody(
            $SearchName, array_keys($SearchResults), $EndUser));
    $Subject = array_shift($BodyLines);
    $Body = implode("\n", $BodyLines);

    # mail report to User
    if (strlen($EndUser->Get("Email")) > 0)
    {
        mail($EndUser->Get("Email"), $Subject, $Body);
    }
}

# Peform the Search
function PerformSearch($SearchGroups)
{
    global $SysConfig;
    global $Engine;

    $SearchResults = NULL;

    # create search engine
    $Engine = new SPTSearchEngine;

    # set OR/AND terms status
    $Engine->SearchTermsRequiredByDefault(
            $SysConfig->SearchTermsRequired());

    # if advanced search
    if (isset($SearchGroups))
    {
	    # perform fielded search
        $SearchResults = $Engine->GroupedSearch($SearchGroups, 0, 99999999);
    }
    return $SearchResults;
}

# function to process Option lists for search 
function ProcessOptionLists($SearchGroups)
{
    global $Entry;
    $Schema = & new MetadataSchema();
    $Fields = $Schema->GetFields(MDFTYPE_OPTION); 

    # step through each option field
    foreach ($Fields as $Field)
    {
        $FName = $Field->Name();
        $DBName = $Field->DBFieldName();

        # option field values stored as a string with values separated by "|"
        $OptionArray = explode("|", $Entry["$DBName"]);

        if (is_array($OptionArray))
        {
            foreach ($OptionArray as $Value)
            {
                # check for enabled fields
                if ($Field->Enabled() && $Value != "No Limit" && !empty($Value))
                {
                    # force multi-option searches to use "OR" logic
                    $SearchGroups["$FName"]["Logic"] = SEARCHLOGIC_OR;
                    $SearchGroups["$FName"]["SearchStrings"]["$FName"][] = 
                                                           "=".$Value;
                }
            }
        }
    }
    # special case cumulative rating
    $Field = $Schema->GetFieldByName("Cumulative Rating");
    if ($Field->IncludeInAdvancedSearch())
    {
       $CumulativeRating = explode("|", $Entry["CumulativeRating"]);
       if (is_array($CumulativeRating))
        {
            foreach ($CumulativeRating as $Value)
            {
                if ($Value != "No Limit" && !empty($Value))
                {
                    switch($Value)
                    {
                        case 0:
                            $Stars = "Not Rated";
                            $Operator = "= ";
                            break;
                        case 20:
                            $Stars = "*";
                            $Operator = ">= ";
                            break;
                        case 40:
                            $Stars = "**";
                            $Operator = ">= ";
                            break;
                        case 60:
                            $Stars = "***";
                            $Operator = ">= ";
                            break;
                        case 80:
                            $Stars = "****";
                            $Operator = ">= ";
                            break;
                        case 100:
                            $Stars = "*****";
                            $Operator = "= ";
                            break;
                    }
                    $SearchGroups[]["SearchStrings"]["Cumulative Rating"] = 
                            $Operator.$Value;
                }
            }
        }
    }
    return $SearchGroups;
}

# function to process flags for the query
function ProcessFlags($SearchGroups)
{
    global $Entry;
    $Schema = & new MetadataSchema();
    $Fields = $Schema->GetFields(MDFTYPE_FLAG); 

    # step through each flag field
    foreach ($Fields as $Field)
    {
        $FName = $Field->Name();
        $DBName = $Field->DBFieldName();
        ${"$DBName"} = $Entry["$DBName"];

        # only process enabled fields - ReleaseFlag is a special case
        if ($Field->Enabled() && $FName != "Release Flag")
        {
            if (is_numeric(${"$DBName"}))
            {
                $SearchGroups[]["SearchStrings"]["$FName"] = ${"$DBName"};
            }
        }
    }
    return $SearchGroups;
}

# function to check frequency against the current time
function CheckFrequency($Frequency)
{
    $Today = getdate(); 
    $Hours = $Today['hours'];
    $Minutes = $Today['minutes'];
    $Wday  = $Today['wday']; 
    $Mday = $Today['mday']; 
    $Mon = $Today['mon'];

    # fire on Jan 1, Apr 1, Jul 1, Oct 1 between 02:50 and 03:10 am
    if ($Frequency == "Quarterly")
    {
        if (($Mon == 1 || $Mon == 4 || $Mon == 7 || $Mon == 10) &&
            $Mday == 1 && (($Minutes > 50 && $Hours == 2) || 
            ($Minutes < 10 && $Hours == 3)))
            return(true);
    }
    # fire on 1st day of the month between 02:50 and 03:10 am every other month
    if ($Frequency == "Bi-monthly")
    {
        if ( ($Mon%2 == 1) &&
            $Mday == 1 && (($Minutes > 50 && $Hours == 2) || 
            ($Minutes < 10 && $Hours == 3)))
            return(true);
    }
    # fire on 1st day of the month between 02:50 and 03:10 am
    if ($Frequency == "Once a month")
    {
        if ($Mday == 1 && (($Minutes > 50 && $Hours == 2) || 
            ($Minutes < 10 && $Hours == 3)))
            return(true);
    }
    # fire on Sunday nite between 02:50 and 03:10 am every other week
    else if ($Frequency == "Bi-weekly")
    {
        if ($Mday & 1 == 1 && $Wday == 0 && (($Minutes > 50 && $Hours == 2) || 
            ($Minutes < 10 && $Hours == 3)))
            return(true);
    }
    # fire on Sunday nite between 02:50 and 03:10 am
    else if ($Frequency == "Once a week")
    {
        if ($Wday == 0 && (($Minutes > 50 && $Hours == 2) || 
            ($Minutes < 10 && $Hours == 3)))
            return(true);
    }
    # fire between 02;50 and 03:10 am daily
    else if ($Frequency == "Once a day")
    {
        if (($Minutes > 50 && $Hours == 2) || 
            ($Minutes < 10 && $Hours == 3))
            return(true);
    }
    # fire hourly
    else if ($Frequency == "Hourly")
    {
        return(true);
    }
    return(false);
}

# ----- MAIN -----------------------------------------------------------------

$DB2 =& new SPTDatabase();

# make sure User Agents are enabled
if ($SysConfig->UserAgentsEnabled())
{
    # process all saved user searches 
    $query = "SELECT * FROM UserSearch ";

    $DB->Query($query);

    while ($Entry = $DB->FetchNextRowArray())
    {
        # should only run user agent if it meets the frequency criteria
        $OkayToRun = CheckFrequency($Entry["Frequency"]);
        $DateLastRun = $Entry["DateLastRun"];

        if ($OkayToRun)
        {
            $SearchGroups = array();
            $SearchName = stripslashes($Entry["SearchName"]);
            $EndUser =& new User($DB, intval($Entry["UserId"]));
            $SearchCat1 = $Entry["SearchCat1"];
            $TextField1 = stripslashes($Entry["SearchText1"]);
            $SearchCat2 = $Entry["SearchCat2"];
            $TextField2 = stripslashes($Entry["SearchText2"]);
            $SearchCat3 = $Entry["SearchCat3"];
            $TextField3 = stripslashes($Entry["SearchText3"]);
            $SearchCat4 = $Entry["SearchCat4"];
            $TextField4 = stripslashes($Entry["SearchText4"]);
            $ReleaseFlag = $Entry["ReleaseFlag"];
            $AddedBy = $Entry["AddedBy"];
            $LastModifiedBy = $Entry["LastModifiedBy"];

            $SearchGroups = ProcessOptionLists($SearchGroups);
            $SearchGroups = ProcessFlags($SearchGroups);

            # process the main search fields as a group
            if (!empty($TextField1) && strlen($SearchCat1))
                $SearchGroups["MAIN"]["SearchStrings"]["$SearchCat1"] =
                    $TextField1;
            if (!empty($TextField2) && strlen($SearchCat2))
                $SearchGroups["MAIN"]["SearchStrings"]["$SearchCat2"] = 
                    $TextField2;
            if (!empty($TextField3) && strlen($SearchCat3))
                $SearchGroups["MAIN"]["SearchStrings"]["$SearchCat3"] = 
                    $TextField3;
            if (!empty($TextField4) && strlen($SearchCat4))
                $SearchGroups["MAIN"]["SearchStrings"]["$SearchCat4"] = 
                    $TextField4;

            # add in special criteria to only find records added since
            # last time this user agent was run
            # reformat date from YYYYMMDD to YYYY-MM-DD
            $SearchGroups[]["SearchStrings"]["Date Of Record Release"] = 
                "> ".substr($DateLastRun, 0 , 4).
                "-".substr($DateLastRun, 4, 2).
                "-".substr($DateLastRun, 6, 2);

            # only process ReleaseFlag if not null (NULL means ALL records) 
            if (!is_null($ReleaseFlag))
            {
                $SearchGroups[]["SearchStrings"]["Release Flag"] = $ReleaseFlag;            }

            # add "added by" to search strings if not set to "All"
            if (strlen($AddedBy) && ($AddedBy != "All"))
            {  
                $SearchGroups["USER"]["SearchStrings"]["Added By Id"] = 
                                                                    $AddedBy;  
            }

            # add "last modified by" to search strings if not set to "All"
            if (strlen($LastModifiedBy) && ($LastModifiedBy != "All"))
            {  
                $SearchGroups["USER"]["SearchStrings"]["Last Modified By Id"] = 
                                                            $LastModifiedBy;  
            }

            $SearchResults = PerformSearch($SearchGroups);

            # email out search results only if resources found 
            if (count($SearchResults) > 0)
                EmailSearchResults($SearchResults, $EndUser, $SearchName);

            # update DateLastRun field in UserSearch
            $Query= "Update UserSearch set DateLastRun = NULL WHERE UserId = ".
                $Entry["UserId"]." and SearchName = \"$SearchName\"";
            $DB2->Query($Query); 
        }
    }
}

?>
