<?PHP

#
#   FILE:  SPT--SPTOAIServer.php
#
#   METHODS PROVIDED:
#       SPTOAIServer()
#           - constructor
#
#   AUTHOR:  Edward Almasy
#
#   Part of the Scout Portal Toolkit
#   Copyright 2002-2004 Internet Scout Project
#   http://scout.wisc.edu
#

require_once(dirname(__FILE__)."/../Scout--OAIServer.php");
require_once(dirname(__FILE__)."/SPT--MetadataSchema.php");
require_once(dirname(__FILE__)."/SPT--QualifierFactory.php");
require_once(dirname(__FILE__)."/SPT--Resource.php");
require_once(dirname(__FILE__)."/SPT--SearchEngine.php");
require_once(dirname(__FILE__)."/SPT--SPTDatabase.php");


class SPTOAIServer extends OAIServer {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    function SPTOAIServer()
    {
        global $SysConfig;
        global $HTTP_SERVER_VARS;

        # grab our own database handle
        $this->DB =& new SPTDatabase();
        $DB =& $this->DB;
        
        # set up repository description
        $DB->Query("SELECT * FROM SystemConfiguration");
        $Record = $DB->FetchRow();
        $RepDescr["Name"]            = $SysConfig->PortalName();
        $RepDescr["BaseURL"]         = "http://".$HTTP_SERVER_VARS["SERVER_NAME"].$HTTP_SERVER_VARS["PHP_SELF"];
        $RepDescr["DateGranularity"] = "DATE";
        $RepDescr["EarliestDate"]    = $Record["OaiEarliestDate"];
        $RepDescr["AdminEmail"][]    = $SysConfig->AdminEmail();
        $RepDescr["IDDomain"]        = $Record["OaiIdDomain"];
        $RepDescr["IDPrefix"]        = $Record["OaiIdPrefix"];

        # create item factory object for retrieving items from DB
        $this->SPTItemFactory =& new SPTOAIItemFactory();

        # call parent's constructor
        $this->OAIServer($DB, $RepDescr, $this->SPTItemFactory, TRUE, TRUE);

        # set up description of nsdl_dc format
        $NsdldcNamespaceList = array(
            "nsdl_dc" => "http://ns.nsdl.org/nsdl_dc_v1.01",
            "dc" => "http://purl.org/dc/elements/1.1/",
            "dct" => "http://purl.org/dc/terms/",
            "ieee" => "http://www.ieee.org/xsd/LOMv1p0",
            );
        $NsdldcElements = array(
            "dc:title",
            "dc:creator",
            "dc:subject",
            "dc:description",
            "dc:publisher",
            "dc:contributor",
            "dc:date",
            "dc:type",
            "dc:format",
            "dc:identifier",
            "dc:source",
            "dc:language",
            "dc:relation",
            "dc:coverage",
            "dc:rights",
            "dct:audience",
            "dct:alternative",
            "dct:tableOfContents",
            "dct:abstract",
            "dct:created",
            "dct:valid",
            "dct:available",
            "dct:issued",
            "dct:modified",
            "dct:extent",
            "dct:medium",
            "dct:isVersionOf",
            "dct:hasVersion",
            "dct:isReplacedBy",
            "dct:replaces",
            "dct:isRequiredBy",
            "dct:requires",
            "dct:isPartOf",
            "dct:hasPart",
            "dct:isReferencedBy",
            "dct:references",
            "dct:isFormatOf",
            "dct:hasFormat",
            "dct:conformsTo",
            "dct:spatial",
            "dct:temporal",
            "dct:mediator",
            "dct:dateAccepted",
            "dct:dateCopyrighted",
            "dct:dateSubmitted",
            "dct:educationLevel",
            "dct:accessRights",
            "dct:bibliographicCitation",
            "ieee:interactivityType",
            "ieee:interactivityLevel",
            "ieee:typicalLearningTime",
            );
        $NsdldcQualifiers = array(
            "dct:LCSH",
            "dct:MESH",
            "dct:DDC",
            "dct:LCC",
            "dct:UDC",
            "dct:DCMIType",
            "dct:IMT",
            "dct:ISO639-2",
            "dct:RFC1766",
            "dct:URI",
            "dct:Point",
            "dct:ISO3166",
            "dct:Box",
            "dct:TGN",
            "dct:Period",
            "dct:W3CDTF",
            "dct:RFC3066",
            );
        $this->AddFormat("nsdl_dc", "nsdl_dc:nsdl_dc", 
                         "http://ns.nsdl.org/nsdl_dc_v1.01"
                         ." http://ns.nsdl.org/schemas/nsdl_dc/nsdl_dc_v1.01.xsd",
                         "1.01.001",
                         $NsdldcNamespaceList, $NsdldcElements, $NsdldcQualifiers);

        # load field mappings from database and set in parent
        $Schema =& new MetadataSchema();
        $DB->Query("SELECT * FROM OAIFieldMappings");
        while ($Record = $DB->FetchRow())
        {
            if ($Record["OAIFieldName"] != "Unmapped")
            {
                parent::SetFieldMapping($Record["FormatName"], 
                                        $Record["SPTFieldId"], 
                                        $Record["OAIFieldName"]);
            }
        }

        # load qualifier mappings from database and set in parent
        $DB->Query("SELECT * FROM OAIQualifierMappings");
        while ($Record = $DB->FetchRow())
        {
            if ($Record["OAIQualifierName"] != "Unmapped")
            {
                $LocalQualifier = new Qualifier($Record["SPTQualifierId"]);
                $LocalQualifierName = $LocalQualifier->Name();
                parent::SetQualifierMapping($Record["FormatName"], 
                                        $LocalQualifierName, 
                                        $Record["OAIQualifierName"]);
            }
        }
    }

    # add SQL conditional for selecting resources
    function AddSQLConditionalForResources($Conditional)
    {
        # pass conditional on to item factory
        $this->SPTItemFactory->AddSQLConditionalForResources($Conditional);
    }
    
    # get/set mapping of local field to OAI field (overloads parent method)
    function GetFieldMapping($FormatName, $LocalFieldName)
    {
        # retrieve ID for local field
        $Schema =& new MetadataSchema();
        $LocalField = $Schema->GetFieldByName($LocalFieldName);
        $LocalFieldId = $LocalField->Id();

        # return stored value
        return parent::GetFieldMapping($FormatName, $LocalFieldId);
    }
    function SetFieldMapping($FormatName, $LocalFieldName, $OAIFieldName)
    {
        # retrieve ID for local field
        $Schema =& new MetadataSchema();
        $LocalField = $Schema->GetFieldByName($LocalFieldName);
        $LocalFieldId = $LocalField->Id();

        # check whether mapping is already in database
        $DB =& $this->DB;
        $MapCount = $DB->Query("SELECT COUNT(*) AS MapCount FROM OAIFieldMappings"
                               ." WHERE FormatName = '".$FormatName."'"
                               ." AND SPTFieldId = '".$LocalFieldId."'",
                               "MapCount");

        # if mapping is already in database
        if ($MapCount > 0)
        {
            # change mapping in database
            $DB->Query("UPDATE OAIFieldMappings"
                       ." SET OAIFieldName = '".addslashes($OAIFieldName)."'"
                       ." WHERE FormatName = '".$FormatName."'"
                       ." AND SPTFieldId = '".$LocalFieldId."'");
        }
        else
        {
            # add new mapping to database
            $DB->Query("INSERT INTO OAIFieldMappings"
                       ." (FormatName, SPTFieldId, OAIFieldName) VALUES"
                       ." ('".$FormatName."', '".$LocalFieldId."', '".addslashes($OAIFieldName)."')");
        }
        
        # call parent method
        parent::SetFieldMapping($FormatName, $LocalFieldId, $OAIFieldName);
    }

    # set mapping of local qualifier to OAI qualifier (overloads parent method)
    function SetQualifierMapping($FormatName, $LocalQualifierName, $OAIQualifierName)
    {
        # retrieve ID for local qualifier
        $QFactory =& new QualifierFactory();
        $LocalQualifier = $QFactory->GetQualifierByName($LocalQualifierName);
        $LocalQualifierId = $LocalQualifier->Id();

        # check whether mapping is already in database
        $DB =& $this->DB;
        $MapCount = $DB->Query("SELECT COUNT(*) AS MapCount FROM OAIQualifierMappings"
                               ." WHERE FormatName = '".$FormatName."'"
                               ." AND SPTQualifierId = '".$LocalQualifierId."'",
                               "MapCount");

        # if mapping is already in database
        if ($MapCount > 0)
        {
            # change mapping in database
            $DB->Query("UPDATE OAIQualifierMappings"
                       ." SET OAIQualifierName = '".addslashes($OAIQualifierName)."'"
                       ." WHERE FormatName = '".$FormatName."'"
                       ." AND SPTQualifierId = '".$LocalQualifierId."'");
        }
        else
        {
            # add new mapping to database
            $DB->Query("INSERT INTO OAIQualifierMappings"
                       ." (FormatName, SPTQualifierId, OAIQualifierName) VALUES"
                       ." ('".$FormatName."', '".$LocalQualifierId."', '".addslashes($OAIQualifierName)."')");
        }
        
        # call parent method
        parent::SetQualifierMapping($FormatName, $LocalQualifierName, $OAIQualifierName);
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $DB;
    var $SPTItemFactory;

}

class SPTOAIItemFactory extends OAIItemFactory {
    
    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function SPTOAIItemFactory()
    {
    }
    
    function GetItem($ItemId)
    {
        # attempt to create item
        $Item = new SPTOAIItem($ItemId);
        
        # if item creation failed
        if ($Item->Status() == -1)
        {
            # return NULL to indicate that no item was found with that ID
            return NULL;
        }
        else
        {
            # return item to caller
            return $Item;
        }
    }
    
    function GetItems($StartingDate = NULL, $EndingDate = NULL)
    {
        return $this->GetItemsInSet(NULL, $StartingDate, $EndingDate);
    }
    
    function GetItemsInSet($Set, $StartingDate = NULL, $EndingDate = NULL)
    {
        # initialize search parameters with release flag requirement
        $SearchStrings["Release Flag"] = "=1";
        
        # if both begin and end date supplied
        if (($StartingDate != NULL) && ($EndingDate != NULL))
        {
            # select resources created between starting and ending dates
            $SearchStrings["Date Of Record Creation"] = 
                    array(">=".$StartingDate, "<=".$EndingDate);
        }
        # else if begin date specified
        elseif ($StartingDate != NULL)
        {
            # select resources created after begin date
            $SearchStrings["Date Of Record Creation"] = ">=".$StartingDate;
        }
        # else if end date specified
        elseif ($EndingDate != NULL)
        {
            # select resources created after begin date
            $SearchStrings["Date Of Record Creation"] = "<=".$EndingDate;
        }
        
        # if set specified
        if ($Set != NULL)
        {
            # split set name into field name and field value
            list($FieldName, $FieldValue) = explode(":", $Set, 2);
            
            # if field exists and is flagged for use as an OAI set
            $Schema = new MetadataSchema;
            $Field = $Schema->GetFieldByName($FieldName);
            if (($Field != NULL) && ($Field->UseForOaiSets()))
            {
                # add field spec to search strings
                $SearchStrings[$FieldName] = "= ".$FieldValue;
            }
            else
            {
                # set will not match anything so return empty array to caller
                return array();
            }
        }
        
        # perform search for desired items
        $Engine = new SPTSearchEngine();
        $SearchResults = $Engine->FieldedSearch($SearchStrings, 0, 1000000);
        
        # extract resource IDs from search results
        $ItemIds = array_keys($SearchResults);
        
        # return array of resource IDs to caller
        return $ItemIds;
    }
    
    # return array containing all possible set names
    function GetListOfSets()
    {
        # start with empty list of sets
        $SetList = array();

        # for each metadata field that is a type that can be used for sets
        $Schema =& new MetadataSchema();
        $Fields = $Schema->GetFields(MDFTYPE_TREE|MDFTYPE_CONTROLLEDNAME|MDFTYPE_OPTION);
        foreach ($Fields as $Field)
        {
            # if field is flagged as being used for OAI sets
            if ($Field->UseForOaiSets())
            {
                # retrieve all possible values for field
                $FieldValues = $Field->GetPossibleValues();

                # prepend field name to each value and add to list of sets
                $FieldName = $Field->Name();
                foreach ($FieldValues as $Value)
                {
                    $SetList[] = $FieldName.":".$Value;
                }
            }
        }

        # return list of sets to caller
        return $SetList;
    }
    
    # retrieve IDs of items that match search parameters (only needed if OAI-SQ supported)
    function SearchForItems($SearchParams, $StartingDate = NULL, $EndingDate = NULL)
    {  
        # translate field IDs into field names for search parameters
        $Schema = new MetadataSchema;
        foreach ($SearchParams as $FieldId => $Value)
        {
            if ($FieldId == "X-KEYWORD-X")
            {
                $SearchStrings["Keyword"] = $Value;
            }
            else
            {
                $Field = $Schema->GetField($FieldId);
                $SearchStrings[$Field->Name()] = $Value;
            }
        }
        
        # add release flag requirement to search parameters
        $SearchStrings["Release Flag"] = "=1";
        
        # if both begin and end date supplied
        if (($StartingDate != NULL) && ($EndingDate != NULL))
        {
            # select resources created between starting and ending dates
            $SearchStrings["Date Of Record Creation"] = 
                    array(">=".$StartingDate, "<=".$EndingDate);
        }
        # else if begin date specified
        elseif ($StartingDate != NULL)
        {
            # select resources created after begin date
            $SearchStrings["Date Of Record Creation"] = ">=".$StartingDate;
        }
        # else if end date specified
        elseif ($EndingDate != NULL)
        {
            # select resources created after begin date
            $SearchStrings["Date Of Record Creation"] = "<=".$EndingDate;
        }
        
        # perform search for desired items
        $Engine = new SPTSearchEngine();
        $SearchResults = $Engine->FieldedSearch($SearchStrings, 0, 1000000);
        
        # extract resource IDs from search results
        $ItemIds = array_keys($SearchResults);
        
        # return array of resource IDs to caller
        return $ItemIds;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

}

class SPTOAIItem extends OAIItem {
    
    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function SPTOAIItem($ItemId)
    {
        # save ID for later use
        $this->Id = $ItemId;
        
        # attempt to create resource object
        $this->Resource =& new Resource($ItemId);
        
        # if resource object creation failed
        if ($this->Resource->Status() == -1)
        {
            # set status to -1 to indicate constructor failure
            $this->LastStatus = -1;
        }
        else
        {
            # set status to 1 to indicate constructor success
            $this->LastStatus = 1;
        }
    }
    
    function GetId() {  return $this->Id;  }
    
    function GetDatestamp()
    {
        $DateString = $this->Resource->Get("Date Of Record Creation");
        if ($DateString == "0000-00-00 00:00:00") {  $DateString = date("Y-m-d");  }
        $Date = new SPTDate($DateString);
        return $Date->FormattedISO8601();
    }
    
    function GetValue($ElementName)
    {
        # retrieve value
        $ReturnValue = $this->Resource->GetByFieldId($ElementName);

        # strip out any HTML tags if text value
        if (is_string($ReturnValue))
        {
            $ReturnValue = strip_tags($ReturnValue);
        }

        # return value to caller
        return $ReturnValue;
    }
    
    function GetQualifier($ElementName)
    {
        $ReturnValue = NULL;
        $Qualifier = $this->Resource->GetQualifierByFieldId($ElementName, TRUE);
        if (is_array($Qualifier))
        {
            foreach ($Qualifier as $ItemId => $QualObj)
            {
                if (is_object($QualObj))
                {
                    $ReturnValue[$ItemId] = $QualObj->Name();
                }
            }
        }
        else
        {
            if (isset($Qualifier) && is_object($Qualifier))
            {
                $ReturnValue = $Qualifier->Name();
            }
        }
        return $ReturnValue;
    }
    
    function GetSets()
    {
        # start out with empty list
        $Sets = array();
        
        # for each possible metadata field
        $Schema =& new MetadataSchema();
        $Fields = $Schema->GetFields(MDFTYPE_TREE|MDFTYPE_CONTROLLEDNAME|MDFTYPE_OPTION);
        foreach ($Fields as $Field)
        {
            # if field is flagged for use for OAI sets
            if ($Field->UseForOaiSets())
            {
                # retrieve values for resource for this field and add to set list
                $FieldName = $Field->Name();
                $Values = $this->Resource->Get($FieldName);
                if (is_array($Values))
                {
                    foreach ($Values as $Value)
                    {
                        $Sets[] = $FieldName.":".$Value;
                    }
                }
                else
                {
                    $Sets[] = $FieldName.":".$Values;
                }
            }
        }
        
        # return list of sets to caller
        return $Sets;
    }
    
    function Status()
    {
        return $this->LastStatus;
    }
    
    
    # ---- PRIVATE INTERFACE -------------------------------------------------
    
    var $Id;
    var $Resource;
    var $LastStatus;
}


?>
