<?PHP

#
#   Axis--User.php
#   An Object for Handling User Information
#
#   Copyright 1999-2001 Axis Data
#   This code is free software that can be used or redistributed under the
#   terms of Version 2 of the GNU General Public License, as published by the
#   Free Software Foundation (http://www.fsf.org).
#
#   Author:  Edward Almasy (almasy@axisdata.com)
#
#   Part of the AxisPHP library v1.2.4
#   For more information see http://www.axisdata.com/AxisPHP/
#

require_once("Axis--Session.php");


class User {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    function User(&$SessionOrDb, $UserInfo=NULL)
    {
        # assume user load will succeed
        $this->Result = U_OKAY;

        # if a session was passed in
        if (is_object($SessionOrDb) && method_exists($SessionOrDb, "Session"))
        {
            # swipe database handle from session
            $this->DB =& $SessionOrDb->DB;

            # save pointer to session
            $this->Session =& $SessionOrDb;

            # get user ID from session if available
            $APUserId = $SessionOrDb->Get("APUserId");
            if (strlen($APUserId) > 0) {  $this->UserId = $APUserId;  }
        }
        # else if database handle was passed in
        elseif (is_object($SessionOrDb) && method_exists($SessionOrDb, "Database"))
        {
            # save database handle
            $this->DB =& $SessionOrDb;

            # if user ID was passed in
            if (is_int($UserInfo))
            {
                # save user ID
                $this->UserId = $UserInfo;
            }
            # else if user name was passed in
            elseif (is_string($UserInfo))
            {
                # look up user ID in database
                $this->DB->query(sprintf("SELECT * FROM APUsers WHERE UserName='%s'",
                        $UserInfo));
                if ($this->DB->NumRowsSelected() > 0)
                {
                    $Record = $this->DB->FetchNextRowArray();
                    $this->UserId = $Record["UserId"];
                }
                else
                {
                    $this->Result = U_NOSUCHUSER;
                }
            }
        }
        else
        {
            # error out
            $this->Result = U_ERROR;
            exit(1);
        }
    }

    function Login($UserName, $Password)
    {
        global $APUserId;

        # if we are not part of a session
        if (empty($this->Session))
        {
            # error out
            exit(1);
        }

        # if user not found in DB
        $this->DB->query(sprintf("SELECT * FROM APUsers WHERE UserName='%s'",
                $UserName));
        if ($this->DB->NumRowsSelected() < 1)
        {
            # result is no user by that name
            $this->Result = U_NOSUCHUSER;
        }
        else
        {
            # grab password from DB
            $Record = $this->DB->FetchNextRowArray();
            $StoredPassword = $Record["UserPassword"];

            # if supplied password matches encrypted password
            $EncryptedPassword = crypt($Password, $StoredPassword);
            if ($EncryptedPassword == $StoredPassword)
            {
                # result is success
                $this->Result = U_OKAY;

                # store user ID for session
                $this->UserId = $Record["UserId"];
                $APUserId = $this->UserId;
                $this->Session->RegisterVariable("APUserId");

                # update last login date
                $this->DB->query(sprintf("UPDATE APUsers "
                        ."SET LastLoginDate=NOW() "
                        ."WHERE UserId='%d'",
                        $this->UserId));
            }
            else
            {
                # result is bad password
                $this->Result = U_BADPASSWORD;
            }
        }

        # return result to caller
        return $this->Result;
    }

    function Logout()
    {
        # if we are not part of a session
        if (empty($this->Session))
        {
            # error out
            exit(1);
        }

        # clear user ID for session
        $this->Session->UnregisterVariable("APUserId");

        # clear our copy of the user ID
        unset($this->UserId);
    }

    function ChangePassword($OldPassword, $NewPassword, $NewPasswordAgain)
    {
        # if we are part of a session make sure a user is logged in
        if (isset($this->Session) && ($this->IsLoggedIn() == FALSE)) 
        {  
            $this->Result = U_NOTLOGGEDIN;  
            return $this->Result;
        }

        # grab password from DB
        $this->DB->query(sprintf("SELECT * FROM APUsers WHERE UserId='%d'",
                $this->UserId));
        $Record = $this->DB->FetchNextRowArray();
        $StoredPassword = $Record["UserPassword"];

        # check old password
        $EncryptedPassword = crypt($OldPassword, $StoredPassword);
        if ($EncryptedPassword != $StoredPassword) 
        {  
            $this->Result = U_BADPASSWORD;
            return $this->Result;  
        }

        # check new password for legality
        if (strlen($NewPassword) < 6) 
        {  
            $this->Result = U_ILLEGALPASSWORD;
            return $this->Result;  
        }

        # make sure both instances of new password match
        if ($NewPassword != $NewPasswordAgain) 
        {  
            $this->Result = U_PASSWORDSDONTMATCH;
            return $this->Result;  
        }

        # generate encrypted new password
        $EncryptedNewPassword = crypt($NewPassword);

        # store encrypted new password
        $this->DB->query(sprintf("UPDATE APUsers "
                ."SET UserPassword='%s'"
                ."WHERE UserId='%d'",
                $EncryptedNewPassword,
                $this->UserId));

        # report to caller that everything succeeded
        $this->Result = U_OKAY;
        return $this->Result;
    }

    function CreateNewUser($UserName, $Password, $PasswordAgain)
    {
        # check user name for legality
        if ($this->IsValidUserName($UserName) == FALSE) 
        {  
            $this->Result = U_ILLEGALUSERNAME;
            return $this->Result;  
        }

        # check password for legality
        if ($this->IsValidPassword($Password) == FALSE) 
        {  
            $this->Result = U_ILLEGALPASSWORD;
            return $this->Result;  
        }

        # make sure both instances of new password match
        if ($Password != $PasswordAgain) 
        {  
            $this->Result = U_PASSWORDSDONTMATCH;
            return $this->Result;  
        }

        # check whether user name is already in use
        $this->DB->query(sprintf("SELECT * FROM APUsers WHERE UserName='%s'",
                $UserName));
        if ($this->DB->NumRowsSelected() > 0) 
        {  
            $this->Result = U_DUPLICATEUSERNAME;
            return $this->Result;  
        }

        # generate encrypted password
        $EncryptedPassword = crypt($Password);

        # store user name and encrypted password
        $this->DB->query(sprintf("INSERT INTO APUsers "
                ."(UserName, UserPassword, CreationDate) VALUES "
                ."('%s', '%s', NOW())",
                $UserName,
                $EncryptedPassword));

        # retrieve and save user record ID
        $this->DB->Query("SELECT LAST_INSERT_ID() AS InsertId FROM APUsers");
        $Record = $this->DB->FetchNextRowArray();
        $this->UserId = $Record["InsertId"];

        # report to caller that everything succeeded
        $this->Result = U_OKAY;
        return $this->Result;
    }

    function CreateNewUserWithEMailedPassword(
            $UserName, $EMail, $EMailAgain, 
            $TemplateFile = "Axis--User--EMailTemplate.txt")
    {
        return CreateNewUserAndMailPasswordFromFile(
                $UserName, $EMail, $EMailAgain, $TemplateFile);
    }

    function CreateNewUserAndMailPasswordFromFile(
            $UserName, $EMail, $EMailAgain, 
            $TemplateFile = "Axis--User--EMailTemplate.txt")
    {
        # load e-mail template from file (first line is subject)
        $Template = file($TemplateFile, 1);
        $EMailSubject = array_shift($Template);
        $EMailBody = join("", $Template);

        return CreateNewUserAndMailPassword(
                $UserName, $EMail, $EMailAgain, $EMailSubject, $EMailBody);
    }

    function CreateNewUserAndMailPassword(
            $UserName, $EMail, $EMailAgain, $EMailSubject, $EMailBody)
    {
        # make sure e-mail addresses match
        if ($EMail != $EMailAgain) 
        {  
            $this->Result = U_EMAILSDONTMATCH;
            return $this->Result;  
        }

        # make sure e-mail address looks valid
        if ($this->IsValidLookingEMailAddress($EMail) == FALSE) 
        {  
            $this->Result = U_ILLEGALEMAIL;
            return $this->Result;  
        }

        # generate random password
        $Password = $this->GetRandomPassword();

        # attempt to create new user with password
        $Result = $this->CreateNewUser($UserName, $Password, $Password);

        # if user creation failed
        if ($Result != U_OKAY)
        {
            # report error result to caller
            return $Result;
        }
        # else
        else
        {
            # set e-mail address in user record
            $this->Set("EMail", $EMail);

            # plug appropriate values into subject and body of e-mail message
            $EMailSubject = str_replace("X-USERNAME-X", $UserName, $EMailSubject);
            $EMailBody = str_replace("X-USERNAME-X", $UserName, $EMailBody);
            $EMailBody = str_replace("X-PASSWORD-X", $Password, $EMailBody);

            # send out e-mail message with new account info
            $Result = mail($EMail, $EMailSubject, $EMailBody);

            # if mailing attempt failed
            if ($Result != TRUE)
            {
                # report error to caller
                $this->Result = U_MAILINGERROR;
                return $this->Result;
            }
            # else
            else
            {
                # report success to caller
                $this->Result = U_OKAY;
                return $this->Result;
            }
        }
    }

    function Get($FieldName)
    {
        # make sure a user is logged in
        if ($this->IsLoggedIn() == FALSE) 
        {  
            $this->Result = U_NOTLOGGEDIN;
            return $this->Result;  
        }

        # retrieve specified value from database
        $this->DB->query(sprintf("SELECT %s FROM APUsers WHERE UserId=%d",
                $FieldName, $this->UserId));
        $Record = $this->DB->FetchNextRowArray();

        # return value to caller
        return $Record[$FieldName];
    }

    function GetDate($FieldName, $Format = "")
    {
        # make sure a user is logged in
        if ($this->IsLoggedIn() == FALSE) 
        {  
            $this->Result = U_NOTLOGGEDIN;
            return $this->Result;
        }

        # retrieve specified value from database
        if (strlen($Format) > 0)
        {
            $this->DB->query(sprintf("SELECT DATE_FORMAT(%s, '%s') AS %s FROM APUsers WHERE UserId=%d",
                    $FieldName, $Format, $FieldName, $this->UserId));
        }
        else
        {
            $this->DB->query(sprintf("SELECT %s FROM APUsers WHERE UserId=%d",
                    $FieldName, $this->UserId));
        }
        $Record = $this->DB->FetchNextRowArray();

        # return value to caller
        return $Record[$FieldName];
    }

    function Set($FieldName, $NewValue)
    {
        # make sure a user is logged in
        if ($this->IsLoggedIn() == FALSE) 
        {  
            $this->Result = U_NOTLOGGEDIN;
            return $this->Result;  
        }

        # set specified value in database
        $this->DB->query(sprintf("UPDATE APUsers SET %s='%s' WHERE UserId=%d",
                $FieldName, $NewValue, $this->UserId));

        # report success to caller
        $this->Result = U_OKAY;
        return $this->Result;
    }

    function IsLoggedIn()
    {
        if (isset($this->UserId)) {  return TRUE;  } else {  return FALSE;  }
    }

    function IsNotLoggedIn()
    {
        if (isset($this->UserId)) {  return FALSE;  } else {  return TRUE;  }
    }

    function HasPriv($Privilege, $Privilege2 = NULL, $Privilege3 = NULL, $Privilege4 = NULL, $Privilege5 = NULL, $Privilege6 = NULL)
    {
        # make sure a user is logged in (no privileges if not logged in)
        if ($this->IsLoggedIn() == FALSE) {  return FALSE;  }

        # build database query to check privileges
        $Query = sprintf("SELECT COUNT(*) AS PrivEntries FROM APUserPrivileges "
                        ."WHERE UserId=%d AND (Privilege=%d",
                $this->UserId, $Privilege);
        if ($Privilege2 != NULL) {  $Query .= " OR Privilege=".$Privilege2;  }
        if ($Privilege3 != NULL) {  $Query .= " OR Privilege=".$Privilege3;  }
        if ($Privilege4 != NULL) {  $Query .= " OR Privilege=".$Privilege4;  }
        if ($Privilege5 != NULL) {  $Query .= " OR Privilege=".$Privilege5;  }
        if ($Privilege6 != NULL) {  $Query .= " OR Privilege=".$Privilege6;  }
        $Query .= ")";

        # look for privilege in database
        $this->DB->query($Query);
        $Record = $this->DB->FetchNextRowArray();

        # return value to caller
        return ($Record["PrivEntries"] > 0) ? TRUE : FALSE;
    }

    function GivePriv($Privilege)
    {
        # if user does not already have privilege
        $this->DB->query(sprintf("SELECT COUNT(*) AS PrivEntries FROM APUserPrivileges WHERE UserId=%d AND Privilege=%d",
                $this->UserId, $Privilege));
        $Record = $this->DB->FetchNextRowArray();
        if ($Record["PrivEntries"] == 0)
        {
            # add privilege for this user to database
            $this->DB->query(sprintf("INSERT INTO APUserPrivileges "
                    ."(UserId, Privilege) VALUES "
                    ."(%d, %d)",
                    $this->UserId, $Privilege));
        }

        # report success to caller
        $this->Result = U_OKAY;
        return $this->Result;
    }

    function RevokePriv($Privilege)
    {
        # remove privilege from database (if present)
        $this->DB->Query("DELETE FROM APUserPrivileges "
                         ."WHERE UserId = '".$this->UserId."' AND Privilege = '".$Privilege."'");

        # report success to caller
        $this->Result = U_OKAY;
        return $this->Result;
    }

    function GetPrivList()
    {
        # for each privilege set for this user
        $PrivList = array();
        $this->DB->query(sprintf(
                "SELECT * FROM APUserPrivileges WHERE UserId=%d", $this->UserId));
        while ($Record = $this->DB->FetchNextRowArray())
        {
            # add privilege value to array
            $PrivList[] = $Record["Privilege"];
        }

        # return array to caller
        return $PrivList;
    }

    function SetPrivList($NewSettings)
    {
        # clear old priv list values
        $this->DB->query(sprintf(
                "DELETE FROM APUserPrivileges WHERE UserId=%d", $this->UserId));

        # for each priv value passed in
        foreach ($NewSettings as $Privilege)
        {
            # set priv for user
            $this->GivePriv($Privilege);
        }
    }

    function Status()
    {
        return $this->Result;
    }

    function StatusMessage()
    {
        global $APUserStatusMessages;
        return $APUserStatusMessages[$this->Result];
    }

    function Delete()
    {
        # delete user record from database
        $this->DB->Query("DELETE FROM APUsers WHERE UserId = '".$this->UserId."'");

        # report to caller that everything succeeded
        $this->Result = U_OKAY;
        return $this->Result;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    # handle to SQL database we use to store user information
    var $DB;

    # session to use in storing persistent information
    var $Session;

    # user ID number for reference into database
    var $UserId;

    # result of last operation
    var $Result;

    # check whether a user name is valid  (alphanumeric string of 2-24 chars)
    function IsValidUserName($UserName)
    {
        if (preg_match("/^[a-zA-Z0-9]{2,24}$/", $UserName)) {  return TRUE;  } else {  return FALSE;  }
    }

    # check whether a password is valid  (at least 6 characters)
    function IsValidPassword($Password)
    {
        if (strlen($Password) < 6) {  return FALSE;  } else {  return TRUE;  }
    }

    # check whether an e-mail address looks valid
    function IsValidLookingEMailAddress($EMail)
    {
        if (preg_match("/^[a-zA-Z0-9._\-]+@[a-zA-Z0-9._\-]+\.[a-zA-Z]{2,3}$/", $EMail)) {  return TRUE;  } else {  return FALSE;  }
    }

    # generate random password
    function GetRandomPassword($PasswordLength = 6)
    {
        # seed random number generator
        mt_srand((double)microtime() * 1000000);

        # generate password of requested length
        return mt_rand(pow(10, ($PasswordLength - 1)),
                (pow(10, $PasswordLength) - 1));
    }
};

# define return values
define("U_OKAY",                0);
define("U_ERROR",               1);
define("U_BADPASSWORD",         2);
define("U_NOSUCHUSER",          3);
define("U_PASSWORDSDONTMATCH",  4);
define("U_EMAILSDONTMATCH",     5);
define("U_DUPLICATEUSERNAME",   6);
define("U_ILLEGALUSERNAME",     7);
define("U_ILLEGALPASSWORD",     8);
define("U_ILLEGALEMAIL",        9);
define("U_NOTLOGGEDIN",         10);
define("U_MAILINGERROR",        11);

# text messages corresponding to possible User status codes
# (These messages should be accessed via the StatusMessage() method.)
$APUserStatusMessages = array(
        U_OKAY                => "The operation was successful.",
        U_ERROR               => "This has been an error.",
        U_BADPASSWORD         => "The password you entered was incorrect.",
        U_NOSUCHUSER          => "No such user name was found.",
        U_PASSWORDSDONTMATCH  => "The new passwords you entered do not match.",
        U_EMAILSDONTMATCH     => "The e-mail addresses you entered do not match.",
        U_DUPLICATEUSERNAME   => "The user name you requested is already in use.",
        U_ILLEGALUSERNAME     => "The user name you requested is too short, too long, or contains illegal characters.",
        U_ILLEGALPASSWORD     => "The new password you requested is too short, too long, or contains illegal characters.",
        U_ILLEGALEMAIL        => "The e-mail address you entered appears to be invalid.",
        U_NOTLOGGEDIN         => "The user is not logged in.",
        U_MAILINGERROR        => "An error occurred while attempting to send e-mail.  Please notify the system administrator."
        );


?>
