<?PHP

#
#   FILE:  Scout--OAIServer.php
#
#   METHODS PROVIDED:
#       OAIServer()
#           - constructor
#
#   AUTHOR:  Edward Almasy
#
#   Copyright 2002-2003 Internet Scout Project
#   http://scout.wisc.edu
#

require_once("Axis--Database.php");


class OAIServer {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function OAIServer(&$DB, $RepDescr, &$ItemFactory)
    {
        # save DB handle for our use
        $this->DB =& $DB;

        # save repository description
        $this->RepDescr = $RepDescr;

        # normalize repository description values
        $this->RepDescr["IDPrefix"] = 
                    preg_replace("/[^0-9a-z]/i", "", $this->RepDescr["IDPrefix"]);

        # save item factory
        $this->ItemFactory =& $ItemFactory;

        # load OAI request type and arguments
        $this->LoadArguments();

        # set default indent size
        $this->IndentSize = 4;

        # start with empty list of formats
        $this->FormatDescrs = array();

        # initialize description of mandatory format
        $OaidcNamespaceList = array(
            "oai_dc" => "http://www.openarchives.org/OAI/2.0/oai_dc/",
            "dc" => "http://purl.org/dc/elements/1.1/",
            );
        $OaidcElements = array(
            "dc:title",
            "dc:creator",
            "dc:subject",
            "dc:description",
            "dc:publisher",
            "dc:contributor",
            "dc:date",
            "dc:type",
            "dc:format",
            "dc:identifier",
            "dc:source",
            "dc:language",
            "dc:relation",
            "dc:coverage",
            "dc:rights",
            );
        $OaidcQualifiers = array();
        $this->AddFormat("oai_dc", "oai_dc:dc", 
                         "http://www.openarchives.org/OAI/2.0/oai_dc/"
                         ." http://www.openarchives.org/OAI/2.0/oai_dc.xsd",
                         NULL,
                         $OaidcNamespaceList, $OaidcElements, $OaidcQualifiers);
    }

    # add metadata format to export
    function AddFormat($Name, $TagName, $SchemaLocation, $SchemaVersion, $NamespaceList, $ElementList, $QualifierList)
    {
        # find highest current format ID
        $HighestFormatId = 0;
        foreach ($this->FormatDescrs as $FormatName => $FormatDescr)
        {
            if ($FormatDescr["FormatId"] > $HighestFormatId)
            {
                $HighestFormatId = $FormatDescr["FormatId"];
            }
        }

        # set new format ID to next value
        $this->FormatDescrs[$Name]["FormatId"] = $HighestFormatId + 1;
        
        # store values
        $this->FormatDescrs[$Name]["TagName"] = $TagName;
        $this->FormatDescrs[$Name]["SchemaLocation"] = $SchemaLocation;
        $this->FormatDescrs[$Name]["SchemaVersion"] = $SchemaVersion;
        $this->FormatDescrs[$Name]["ElementList"] = $ElementList;
        $this->FormatDescrs[$Name]["QualifierList"] = $QualifierList;
        $this->FormatDescrs[$Name]["NamespaceList"] = $NamespaceList;
        
        # start out with empty mappings list
        if (!isset($this->FieldMappings[$Name]))
        {
            $this->FieldMappings[$Name] = array();
        }
    }

    # return list of formats
    function FormatList()
    {
        $FList = array();
        foreach ($this->FormatDescrs as $FormatName => $FormatDescr)
        {
            $FList[$FormatDescr["FormatId"]] = $FormatName;
        }
        return $FList;
    }

    # return list of elements for a given format
    function FormatElementList($FormatName)
    {
        return $this->FormatDescrs[$FormatName]["ElementList"];
    }
    
    # return list of qualifiers for a given format
    function FormatQualifierList($FormatName)
    {
        return $this->FormatDescrs[$FormatName]["QualifierList"];
    }
    
    # get/set mapping of local field to OAI field
    function GetFieldMapping($FormatName, $LocalFieldName)
    {
        # return stored value
        if (isset($this->FieldMappings[$FormatName][$LocalFieldName]))
        {
            return $this->FieldMappings[$FormatName][$LocalFieldName];
        }
        else
        {
            return NULL;
        }
    }
    function SetFieldMapping($FormatName, $LocalFieldName, $OAIFieldName)
    {
        $this->FieldMappings[$FormatName][$LocalFieldName] = $OAIFieldName;
    }

    # get/set mapping of local qualifier to OAI qualifier
    function GetQualifierMapping($FormatName, $LocalQualifierName)
    {
        # return stored value
        if (isset($this->QualifierMappings[$FormatName][$LocalQualifierName]))
        {
            return $this->QualifierMappings[$FormatName][$LocalQualifierName];
        }
        else
        {
            return NULL;
        }
    }
    function SetQualifierMapping($FormatName, $LocalQualifierName, $OAIQualifierName)
    {
        $this->QualifierMappings[$FormatName][$LocalQualifierName] = $OAIQualifierName;
    }

    function GetResponse()
    {
        # call appropriate method based on request type
        switch (strtoupper($this->Args["verb"]))
        {
            case "IDENTIFY":
                $Response = $this->ProcessIdentify();
                break;

            case "GETRECORD":
                $Response = $this->ProcessGetRecord();
                break;

            case "LISTIDENTIFIERS":
                $Response = $this->ProcessListRecords(FALSE);
                break;

            case "LISTRECORDS":
                $Response = $this->ProcessListRecords(TRUE);
                break;

            case "LISTMETADATAFORMATS":
                $Response = $this->ProcessListMetadataFormats();
                break;

            case "LISTSETS":
                $Response = $this->ProcessListSets();
                break;

            default:
                # return "bad argument" response
                $Response = $this->GetResponseBeginTags();
                $Response .= $this->GetRequestTag();
                $Response .= $this->GetErrorTag("badVerb", "Bad or unknown request type.");
                $Response .= $this->GetResponseEndTags();
                break;
        }

        # return generated response to caller
        return $Response;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------
    
    var $DB;
    var $Args;
    var $RepDescr;
    var $ItemFactory;
    var $FormatDescrs;
    var $FormatFields;
    var $FieldMappings;
    var $QualifierMappings;
    var $IndentSize;

    
    # ---- response generation methods

    function ProcessIdentify()
    {
        # initialize response
        $Response = $this->GetResponseBeginTags();

        # add request info tag
        $Response .= $this->GetRequestTag("Identify");

        # open response type tag
        $Response .= $this->FormatTag("Identify");

        # add repository info tags
        $Response .= $this->FormatTag("repositoryName", $this->RepDescr["Name"]);
        $Response .= $this->FormatTag("baseURL", $this->RepDescr["BaseURL"]);
        $Response .= $this->FormatTag("protocolVersion", "2.0");
        foreach ($this->RepDescr["AdminEmail"] as $AdminEmail)
        {
            $Response .= $this->FormatTag("adminEmail", $AdminEmail);
        }
        $Response .= $this->FormatTag("earliestDatestamp", $this->RepDescr["EarliestDate"]);
        $Response .= $this->FormatTag("deletedRecord", "no");
        $Response .= $this->FormatTag("granularity", 
                                      (strtoupper($this->RepDescr["DateGranularity"]) == "DATETIME")
                                          ? "YYYY-MM-DDThh:mm:ssZ" : "YYYY-MM-DD");

        # add repository description section
        $Response .= $this->FormatTag("description");
        $Attribs = array(
                "xmlns" => "http://www.openarchives.org/OAI/2.0/oai-identifier",
                "xmlns:xsi" => "http://www.w3.org/2001/XMLSchema-instance",
                "xsi:schemaLocation" => "http://www.openarchives.org/OAI/2.0/oai-identifier http://www.openarchives.org/OAI/2.0/oai-identifier.xsd",
                );
        $Response .= $this->FormatTag("oai-identifier", NULL, $Attribs);
        $Response .= $this->FormatTag("scheme", "oai");
        $Response .= $this->FormatTag("repositoryIdentifier", $this->RepDescr["IDDomain"]);
        $Response .= $this->FormatTag("delimiter", ":");
        $Response .= $this->FormatTag("sampleIdentifier", $this->EncodeIdentifier("12345"));
        $Response .= $this->FormatTag();
        $Response .= $this->FormatTag();

        # close response type tag
        $Response .= $this->FormatTag();

        # close out response
        $Response .= $this->GetResponseEndTags();

        # return response to caller
        return $Response;
    }

    function ProcessGetRecord()
    {
        # initialize response
        $Response = $this->GetResponseBeginTags();
        
        # if arguments were bad
        $ItemId = $this->DecodeIdentifier($this->Args["identifier"]);
        $MetadataFormat = $this->Args["metadataPrefix"];
        if (($ItemId == NULL) || !is_array($this->FieldMappings[$MetadataFormat]))
        {
            # add request info tag with no attributes
            $Response .= $this->GetRequestTag("GetRecord");
            
            # add error tag
            $Response .= $this->GetErrorTag("badArgument", "Bad argument found.");
        }
        else
        {
            # add request info tag
            $ReqArgList = array("identifier", "metadataPrefix");
            $Response .= $this->GetRequestTag("GetRecord", $ReqArgList);

            # attempt to load item corresponding to record
            $Item = $this->ItemFactory->GetItem($ItemId);

            # if no item found
            if ($Item == NULL)
            {
                # add error tag
                $Response .= $this->GetErrorTag("idDoesNotExist", "No item found for specified ID.");
            }
            else
            {
                # open response type tag
                $Response .= $this->FormatTag("GetRecord");

                # add tags for record
                $Response .= $this->GetRecordTags($Item, $MetadataFormat);

                # close response type tag
                $Response .= $this->FormatTag();
            }
        }

        # close out response
        $Response .= $this->GetResponseEndTags();

        # return response to caller
        return $Response;
    }

    function ProcessListRecords($IncludeMetadata)
    {
        # set request type
        if ($IncludeMetadata)
        {
            $Request = "ListRecords";
        }
        else
        {
            $Request = "ListIdentifiers";
        }
        
        # initialize response
        $Response = $this->GetResponseBeginTags();
        
        # if resumption token supplied
        if (isset($this->Args["resumptionToken"]))
        {
            # set expected argument lists
            $ReqArgList = array("resumptionToken");
            $OptArgList = NULL;
            
            # parse into list parameters
            $Args = $this->DecodeResumptionToken($this->Args["resumptionToken"]);
        }
        else
        {
            # set expected argument lists
            $ReqArgList = array("metadataPrefix");
            $OptArgList = array("from", "until", "set");
            
            # get list parameters from incoming arguments
            $Args = $this->Args;
            
            # set list starting point to beginning
            $Args["ListStartPoint"] = 0;
        }

        # if resumption token was supplied and was bad
        if ($Args == NULL)
        {
            # add request info tag
            $Response .= $this->GetRequestTag($Request, $ReqArgList, $OptArgList);

            # add error tag indicating bad resumption token
            $Response .= $this->GetErrorTag("badResumptionToken", "Bad resumption token.");
            
            # if other parameter also supplied
            if (count($this->Args) > 2)
            {
                # add error tag indicating exclusive argument error
                $Response .= $this->GetErrorTag("badArgument", "Resumption token is exclusive argument.");
            }
        }
        # else if resumption token supplied and other arguments also supplied
        elseif (isset($this->Args["resumptionToken"]) && (count($this->Args) > 2))
        {
            # add error tag indicating exclusive argument error
            $Response .= $this->GetErrorTag("badArgument", "Resumption token is exclusive argument.");
        }
        # else if metadata format was not specified
        elseif (empty($Args["metadataPrefix"]))
        {
            # add request info tag with no attributes
            $Response .= $this->GetRequestTag($Request);
            
            # add error tag indicating bad argument
            $Response .= $this->GetErrorTag("badArgument", "No metadata format specified.");
        }
        # else if from or until date is specified but bad
        elseif ((isset($Args["from"]) && $this->DateIsInvalid($Args["from"]))
                || (isset($Args["until"]) && $this->DateIsInvalid($Args["until"])))
        {
            # add request info tag with no attributes
            $Response .= $this->GetRequestTag($Request);
            
            # add error tag indicating bad argument
            $Response .= $this->GetErrorTag("badArgument", "Bad date format.");
        }
        else
        {
            # add request info tag
            $Response .= $this->GetRequestTag($Request, $ReqArgList, $OptArgList);

            # if set requested
            if (isset($Args["set"]))
            {
                # add error tag indicating that we don't support sets
                $Response .= $this->GetErrorTag("noSetHierarchy", "This repository does not support sets.");
            }
            # else if requested metadata format is not supported
            elseif (empty($this->FormatDescrs[$Args["metadataPrefix"]]))
            {
                # add error tag indicating that format is not supported
                $Response .= $this->GetErrorTag("cannotDisseminateFormat", "Metadata format \"".$Args["metadataPrefix"]."\" not supported by this repository.");
            }
            else
            {
                # get list of items that matches incoming criteria
                $ItemIds = $this->ItemFactory->GetItems(
                    (isset($Args["from"]) ? $Args["from"] : NULL),
                    (isset($Args["until"]) ? $Args["until"] : NULL));
                
                # if no items found
                if (count($ItemIds) == 0)
                {
                    # add error tag indicating that no records found that match spec
                    $Response .= $this->GetErrorTag("noRecordsMatch", "No records were found that match the specified parameters.");
                }
                else
                {
                    # open response type tag
                    $Response .= $this->FormatTag($Request);
                    
                    # initialize count of processed items
                    $ListIndex = 0;
                    
                    # for each item
                    foreach ($ItemIds as $ItemId)
                    {
                        # if item is within range
                        if ($ListIndex >= $Args["ListStartPoint"])
                        {
                            # retrieve item
                            $Item = $this->ItemFactory->GetItem($ItemId);

                            # add record for item
                            $Response .= $this->GetRecordTags($Item, $Args["metadataPrefix"], $IncludeMetadata);
                        }
                        
                        # increment count of processed items
                        $ListIndex++;
                        
                        # stop processing if we have processed max number of items in a pass
                        $MaxItemsPerPass = 20;
                        if (($ListIndex - $Args["ListStartPoint"]) >= $MaxItemsPerPass) {  break;  }
                    }
                    
                    # if items left unprocessed
                    if ($ListIndex < count($ItemIds))
                    {
                        # add resumption token tag
                        $Token = $this->EncodeResumptionToken((isset($Args["from"]) ? $Args["from"] : NULL), 
                                                              (isset($Args["until"]) ? $Args["until"] : NULL), 
                                                              (isset($Args["metadataPrefix"]) ? $Args["metadataPrefix"] : NULL), 
                                                              (isset($Args["set"]) ? $Args["set"] : NULL), 
                                                              $ListIndex);
                        $Response .= $this->FormatTag("resumptionToken", $Token);
                    }
                    else
                    {
                        # if we started with a resumption token tag
                        if (isset($this->Args["resumptionToken"]))
                        {
                            # add empty resumption token tag to indicate end of set
                            $Response .= $this->FormatTag("resumptionToken", "");
                        }
                    }

                    # close response type tag
                    $Response .= $this->FormatTag();
                }
            }
        }

        # close out response
        $Response .= $this->GetResponseEndTags();

        # return response to caller
        return $Response;
    }

    function ProcessListMetadataFormats()
    {
        # initialize response
        $Response = $this->GetResponseBeginTags();

        # if arguments were bad
        $Arg = isset($this->Args["identifier"]) ? $this->Args["identifier"] : NULL;
        $ItemId = $this->DecodeIdentifier($Arg);
        if (isset($this->Args["identifier"]) && ($ItemId == NULL))
        {
            # add error tag
            $Response .= $this->GetErrorTag("idDoesNotExist", "Identifier unknown or illegal.");
        }
        else
        {
            # add request info tag
            $OptArgList = array("identifier");
            $Response .= $this->GetRequestTag("ListMetadataFormats", NULL, $OptArgList);

            # open response type tag
            $Response .= $this->FormatTag("ListMetadataFormats");

            # for each supported format
            foreach ($this->FormatDescrs as $FormatName => $FormatDescr)
            {
                # open format tag
                $Response .= $this->FormatTag("metadataFormat");

                # add tags describing format
                $Response .= $this->FormatTag("metadataPrefix", $FormatName);
                $Pieces = preg_split("/[\s]+/", $FormatDescr["SchemaLocation"]);
                $Response .= $this->FormatTag("schema", $Pieces[1]);
                $Response .= $this->FormatTag("metadataNamespace", $FormatDescr["NamespaceList"][$FormatName]);

                # close format tag
                $Response .= $this->FormatTag();
            }

            # close response type tag
            $Response .= $this->FormatTag();
        }

        # close out response
        $Response .= $this->GetResponseEndTags();

        # return response to caller
        return $Response;
    }

    function ProcessListSets()
    {
        # initialize response
        $Response = $this->GetResponseBeginTags();

        # add request info tag
        $OptArgList = array("resumptionToken");
        $Response .= $this->GetRequestTag("ListSets", NULL, $OptArgList);

        # add error tag indicating that we do not support sets
        $Response .= $this->GetErrorTag("noSetHierarchy", "This repository does not support sets.");

        # close out response
        $Response .= $this->GetResponseEndTags();

        # return response to caller
        return $Response;
    }

    
    # ---- common private methods
    
    function GetResponseBeginTags()
    {
        # start with XML declaration
        $Tags = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n";

        # add OAI-PMH root element begin tag
        $Tags .= "<OAI-PMH xmlns=\"http://www.openarchives.org/OAI/2.0/\"\n"
                ."        xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n"
                ."        xsi:schemaLocation=\"http://www.openarchives.org/OAI/2.0/\n"
                ."            http://www.openarchives.org/OAI/2.0/OAI-PMH.xsd\">\n";

        # add response timestamp
        $Tags .= "    <responseDate>".date("Y-m-d\\TH:i:s\\Z")."</responseDate>\n";

        # return tags to caller
        return $Tags;
    }

    function GetResponseEndTags()
    {
        # close out OAI-PMH root element
        $Tags = "</OAI-PMH>\n";

        # return tags to caller
        return $Tags;
    }

    function GetRequestTag($RequestType = NULL, $ReqArgList = NULL, $OptArgList = NULL)
    {
        # build attribute array
        if ($RequestType !== NULL)
        {
            $AttributeList["verb"] = $RequestType;
        }
        if ($ReqArgList != NULL)
        {
            foreach ($ReqArgList as $ArgName)
            {
                $AttributeList[$ArgName] = $this->Args[$ArgName];
            }
        }
        if ($OptArgList != NULL)
        {
            foreach ($OptArgList as $ArgName)
            {
                if (isset($this->Args[$ArgName]))
                {
                    $AttributeList[$ArgName] = $this->Args[$ArgName];
                }
            }
        }

        # generate formatted tag
        $Tag = $this->FormatTag("request",
                                $this->RepDescr["BaseURL"],
                                $AttributeList);

        # return tag to caller
        return $Tag;
    }

    function GetErrorTag($ErrorCode, $ErrorMessage)
    {
        return $this->FormatTag("error", $ErrorMessage, array("code" => $ErrorCode));
    }

    function GetRecordTags($Item, $MetadataFormat, $IncludeMetadata = TRUE)
    {
        # if more than identifiers requested
        if ($IncludeMetadata)
        {
            # open record tag
            $Tags = $this->FormatTag("record");
        }
        else
        {   $Tags = "";
        }

        # add header with identifier and datestamp tags
        $Tags .= $this->FormatTag("header");
        $Tags .= $this->FormatTag("identifier", 
                                  $this->EncodeIdentifier($Item->GetId()));
        $Tags .= $this->FormatTag("datestamp", $Item->GetDatestamp());
        $Tags .= $this->FormatTag();

        # if more than identifiers requested
        if ($IncludeMetadata)
        {
            # open metadata tag
            $Tags .= $this->FormatTag("metadata");

            # set up attributes for metadata format tag
            $MFAttribs["xsi:schemaLocation"] = $this->FormatDescrs[$MetadataFormat]["SchemaLocation"];
            if (strlen($this->FormatDescrs[$MetadataFormat]["SchemaVersion"]) > 0)
            {
                $MFAttribs["schemaVersion"] = $this->FormatDescrs[$MetadataFormat]["SchemaVersion"];
            }
            $MFAttribs["xmlns:xsi"] = "http://www.w3.org/2001/XMLSchema-instance";
            foreach ($this->FormatDescrs[$MetadataFormat]["NamespaceList"] as $NamespaceName => $NamespaceURI)
            {
                $MFAttribs["xmlns:".$NamespaceName] = $NamespaceURI;
            }

            # open metadata format tag
            $Tags .= $this->FormatTag($this->FormatDescrs[$MetadataFormat]["TagName"], NULL, $MFAttribs);

            # for each field mapping for this metadata format
            foreach ($this->FieldMappings[$MetadataFormat] as $LocalFieldName => $OAIFieldName)
            {
                # if field looks like it has been mapped
                if (strlen($OAIFieldName) > 0)
                {
                    # retrieve content for field
                    $Content = $Item->GetValue($LocalFieldName);
                    
                    # retrieve qualifiers for content
                    $Qualifier = $Item->GetQualifier($LocalFieldName);

                    # if content is array
                    if (is_array($Content))
                    {
                        # for each element of array
                        foreach ($Content as $ContentIndex => $ContentValue)
                        {
                            # if element has content
                            if (strlen($ContentValue) > 0)
                            {
                                # generate tag for element
                                if (strlen($Qualifier[$ContentIndex]) > 0)
                                {
                                    if (isset($this->QualifierMappings[$MetadataFormat][$Qualifier[$ContentIndex]])
                                        && (strlen($this->QualifierMappings[$MetadataFormat][$Qualifier[$ContentIndex]]) > 0))
                                    {
                                        $ContentAttribs["xsi:type"] = $this->QualifierMappings[$MetadataFormat][$Qualifier[$ContentIndex]];
                                    }
                                }
                                else
                                {
                                    $ContentAttribs = NULL;
                                }
                                $Tags .= $this->FormatTag($OAIFieldName, 
                                                          utf8_encode(htmlspecialchars(preg_replace("/[\\x00-\\x1F]+/", "", $ContentValue))),
                                                          $ContentAttribs);
                            }
                        }
                    }
                    else
                    {
                        # if field has content
                        if (strlen($Content) > 0)
                        {
                            # generate tag for field
                            if (strlen($Qualifier) > 0)
                            {
                                if (strlen($this->QualifierMappings[$MetadataFormat][$Qualifier]) > 0)
                                {
                                    $ContentAttribs["xsi:type"] = $this->QualifierMappings[$MetadataFormat][$Qualifier];
                                }
                            }
                            else
                            {
                                $ContentAttribs = NULL;
                            }
                            $Tags .= $this->FormatTag($OAIFieldName, 
                                                      utf8_encode(htmlspecialchars(preg_replace("/[\\x00-\\x1F]+/", "", $Content))),
                                                      $ContentAttribs);
                        }
                    }
                }
            }

            # close metadata format tag
            $Tags .= $this->FormatTag();

            # close metadata tag
            $Tags .= $this->FormatTag();
        }

        # if more than identifiers requested
        if ($IncludeMetadata)
        {
            # close record tag
            $Tags .= $this->FormatTag();
        }

        # return tags to caller
        return $Tags;
    }

    function EncodeIdentifier($ItemId)
    {
        # return encoded value to caller
        return "oai:".$this->RepDescr["IDDomain"]
                .":".$this->RepDescr["IDPrefix"]."-".$ItemId;
    }

    function DecodeIdentifier($Identifier)
    {
        # assume that decode will fail
        $Id = NULL;

        # split ID into component pieces
        $Pieces = split(":", $Identifier);

        # if pieces look okay
        if (($Pieces[0] == "oai") && ($Pieces[1] == $this->RepDescr["IDDomain"]))
        {
            # split final piece
            $Pieces = split("-", $Pieces[2]);

            # if identifier prefix looks okay
            if ($Pieces[0] == $this->RepDescr["IDPrefix"])
            {
                # decoded value is final piece
                $Id = $Pieces[1];
            }
        }

        # return decoded value to caller
        return $Id;
    }
    
    function EncodeResumptionToken($StartingDate, $EndingDate, $MetadataFormat, $SetSpec, $ListStartPoint)
    {
        # concatenate values to create token
        $Token = $StartingDate."-_-".$EndingDate."-_-".$MetadataFormat."-_-"
                .$SetSpec."-_-".$ListStartPoint;
        
        # return token to caller
        return $Token;
    }
    
    function DecodeResumptionToken($ResumptionToken)
    {
        # split into component pieces
        $Pieces = preg_split("/-_-/", $ResumptionToken);
        
        # if we were unable to split token
        if (count($Pieces) != 5)
        {
            # return NULL list
            $Args = NULL;
        }
        else
        {
            # assign component pieces to list parameters
            if (strlen($Pieces[0]) > 0) {  $Args["from"] = $Pieces[0];  }
            if (strlen($Pieces[1]) > 0) {  $Args["until"] = $Pieces[1];  }
            if (strlen($Pieces[2]) > 0) {  $Args["metadataPrefix"] = $Pieces[2];  }
            if (strlen($Pieces[3]) > 0) {  $Args["set"] = $Pieces[3];  }
            if (strlen($Pieces[4]) > 0) {  $Args["ListStartPoint"] = $Pieces[4];  }
        }
        
        # return list parameter array to caller
        return $Args;
    }
    
    function DateIsInvalid($Date)
    {
        # if date is null or matches required format
        if (empty($Date) || preg_match("/^[0-9]{4}-[0-9]{2}-[0-9]{2}$/", $Date))
        {
            # date is okay
            return FALSE;
        }
        else
        {
            # date is not okay
            return TRUE;
        }
    }

    function FormatTag($Name = NULL, $Content = NULL, $Attributes = NULL, $NewIndentLevel = NULL)
    {
        static $IndentLevel = 1;
        static $OpenTagStack = array();

        # reset indent level if requested
        if ($NewIndentLevel !== NULL)
        {
            $IndentLevel = $NewIndentLevel;
        }

        # if tag name supplied
        if ($Name !== NULL)
        {
            # start out with appropriate indent
            $Tag = str_repeat(" ", ($IndentLevel * $this->IndentSize));

            # open begin tag
            $Tag .= "<".$Name;

            # if attributes supplied
            if ($Attributes !== NULL)
            {
                # add attributes
                foreach ($Attributes as $AttributeName => $AttributeValue)
                {
                    $Tag .= " ".$AttributeName."=\"".$AttributeValue."\"";
                }
            }

            # if content supplied
            if ($Content !== NULL)
            {
                # close begin tag
                $Tag .= ">";

                # add content
                $Tag .= $Content;

                # add end tag
                $Tag .= "</".$Name.">\n";
            }
            else
            {
                # close begin tag
                $Tag .= ">\n";

                # increase indent level
                $IndentLevel++;

                # add tag to open tag stack
                array_push($OpenTagStack, $Name);
            }
        }
        else
        {
            # decrease indent level
            if ($IndentLevel > 0) {  $IndentLevel--;  }

            # pop last entry off of open tag stack
            $LastName = array_pop($OpenTagStack);

            # start out with appropriate indent
            $Tag = str_repeat(" ", ($IndentLevel * $this->IndentSize));

            # add end tag to match last open tag
            $Tag .= "</".$LastName.">\n";
        }

        # return formatted tag to caller
        return $Tag;
    }

    function LoadArguments()
    {
        global $HTTP_GET_VARS;
        global $HTTP_POST_VARS;

        # if request type available via POST variables
        if (isset($HTTP_POST_VARS["verb"]))
        {
            # retrieve arguments from POST variables
            $this->Args = $HTTP_POST_VARS;
        }
        # else if request type available via GET variables
        elseif (isset($HTTP_GET_VARS["verb"]))
        {
            # retrieve arguments from GET variables
            $this->Args = $HTTP_GET_VARS;
        }
        else
        {
            # ERROR OUT
            # ???
        }
    }
}

class OAIItemFactory {
    
    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function OAIItemFactory()
    {
    }
    
    function GetItem($ItemId) {  exit("OAIItemFactory method GetItem() not implemented");  }
    function GetItems($StartingDate = NULL, $EndingDate = NULL)
    {  
        exit("OAIItemFactory method GetItems() not implemented");  
    }
}

class OAIItem {
    
    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function OAIItem($ItemId)
    {
    }
    
    function GetId()                     {  exit("OAIItem method GetId() not implemented");  }
    function GetDatestamp()              {  exit("OAIItem method GetDatestamp() not implemented");  }
    function GetValue($ElementName)      {  exit("OAIItem method GetValue() not implemented");  }
    function GetQualifier($ElementName)  {  exit("OAIItem method GetQualifiers() not implemented");  }
    function Status()                    {  exit("OAIItem method Status() not implemented");  }
}


?>
