<?PHP

#
#   FILE:  SPT--ItemFactory.php
#
#   METHODS PROVIDED:
#       ItemFactory()
#           - constructor
#       Status()
#           - return current error status
#       GetLastTempItem()
#           - retrieve temp item based on user ID and last modified date
#                     (returns NULL if no temp item found for that ID)
#       CleanOutStaleTempItems()
#           - clear out any temp items more than one week old
#
#   NOTES:
#       - for a derived class to use the temp methods the item record in the
#             database must include "DateLastModified" and "LastModifiedById"
#             fields, and the item object must include a "Delete()" method
#
#   AUTHOR:  Edward Almasy
#
#   Part of the Scout Portal Toolkit
#   Copyright 2003 Internet Scout Project
#   http://scout.wisc.edu
#

require_once("include/SPT--SPTDatabase.php");
require_once("include/SPT--Common.php");        # (to make sure we have $Session and $User global vars)


class ItemFactory {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function ItemFactory()
    {
		# grab our own database handle
		$this->DB =& new SPTDatabase();
        
        # assume everything will be okay
        $this->ErrorStatus = 0;
    }
    
    # return current error status
    function Status() {  return $this->ErrorStatus;  }
    
    # get ID of currently edited item
    function GetCurrentEditedItemId()
    {
        # if ID available in session variable
        global $Session;
        if ($EditedIds = $Session->Get($this->ItemClassName."EditedIds"))
        {
            # look up value in session variable
            $ItemId = $EditedIds[0];
        }
        else
        {
            # attempt to look up last temp item ID
            $ItemId = $this->GetLastTempItemId();

            # store it in session variable
            $EditedIds = array($ItemId);
            $Session->RegisterVariable($this->ItemClassName."EditedIds", $EditedIds);
        }

        # return ID (if any) to caller
        return $ItemId;
    }

    # set ID of currently edited item
    function SetCurrentEditedItemId($NewId)
    {
        # if edited ID array already stored for session
        global $Session;
        if ($EditedIds = $Session->Get($this->ItemClassName."EditedIds"))
        {
            # prepend new value to array
            array_unshift($EditedIds, $NewId);
        }
        else
        {
            # start with fresh array
            $EditedIds = array($NewId);
        }

        # save in session variable
        $Session->RegisterVariable($this->ItemClassName."EditedIds", $EditedIds);
    }

    # clear currently edited item ID
    function ClearCurrentEditedItemId()
    {
        # if edited item IDs available in a session variable
        global $Session;
        $SessionVarName = $this->ItemClassName."EditedIds";
        if ($EditedIds = $Session->Get($SessionVarName))
        {
            # remove current item from edited item ID array
            array_shift($EditedIds);

            # if no further edited items
            if (count($EditedIds) < 1)
            {
                # destroy session variable
                $Session->UnregisterVariable($SessionVarName);
            }
            else
            {
                # save new shorter edited item ID array to session variable
                $Session->RegisterVariable($SessionVarName, $EditedIds);
            }
        }
    }

    # clear currently edited item ID and item
    function ClearCurrentEditedItem()
    {
        # if current edited item is temp item
        $CurrentEditedItemId = $this->GetCurrentEditedItemId();
        if ($CurrentEditedItemId < 0)
        {
            # delete temp item from DB
            $this->DB->Query("DELETE FROM ".$this->ItemTableName
                             ." WHERE ".$this->ItemIdFieldName." = ".$CurrentEditedItemId);
        }
        
        # clear current edited item ID
        $this->ClearCurrentEditedItemId();
    }

    # clear out any temp items more than one week old
    function CleanOutStaleTempItems()
    {
        # load array of stale items
        $this->DB->Query("SELECT ".$this->ItemIdFieldName." FROM ".$this->ItemTableName
                   ." WHERE ".$this->ItemIdFieldName." < 0"
                   ." AND DateLastModified < DATE_SUB(NOW(), INTERVAL 7 DAY)");
        $ItemIds = $this->DB->FetchColumn($this->ItemIdFieldName);
        
        # delete stale items
        foreach ($ItemIds as $ItemId)
        {
            $Item =& new $this->ItemClassName($ItemId);
            $Item->Delete();
        }
    }
    
    # retrieve most recent temp item ID based on user ID
    # (returns NULL if no temp item found for that user ID)
    function GetLastTempItemId()
    {
        # retrieve ID of most recently modified temp item for this user
        global $User;
        $ItemId = $this->DB->Query("SELECT ".$this->ItemIdFieldName." FROM ".$this->ItemTableName
                                 ." WHERE LastModifiedById = '".$User->Get("UserId")."'"
                                 ." AND ".$this->ItemIdFieldName." < 0"
                                 ." ORDER BY ".$this->ItemIdFieldName." ASC"
                                 ." LIMIT 1",
                                 $this->ItemIdFieldName);
        
        # return item to caller (or NULL if none found)
        return $ItemId;
    }
    
    # return next item ID
    function GetNextItemId()
    {
        $HighestItemId = $this->DB->Query("SELECT ".$this->ItemIdFieldName
                                          ." FROM ".$this->ItemTableName
                                          ." ORDER BY ".$this->ItemIdFieldName
                                          ." DESC LIMIT 1", 
                                          $this->ItemIdFieldName);
        if ($HighestItemId <= 0)
        {
            $ItemId = 1;
        }
        else
        {
            $ItemId = $HighestItemId + 1;
        }
        return $ItemId;
    }
    
    # return next temp item ID
    function GetNextTempItemId()
    {
        $LowestItemId = $this->DB->Query("SELECT ".$this->ItemIdFieldName
                                         ." FROM ".$this->ItemTableName
                                         ." ORDER BY ".$this->ItemIdFieldName
                                         ." ASC LIMIT 1", 
                                         $this->ItemIdFieldName);
        if ($LowestItemId > 0)
        {
            $ItemId = -1;
        }
        else
        {
            $ItemId = $LowestItemId - 1;
        }
        return $ItemId;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $ItemClassName;
    var $ItemTableName;
    var $ItemIdFieldName;
    var $DB;
    var $ErrorStatus;
    
}


?>
