<?PHP

#
#   Axis--User.php
#   An Object for Handling User Information
#
#   Copyright 1999-2001 Axis Data
#   This code is free software that can be used or redistributed under the
#   terms of Version 2 of the GNU General Public License, as published by the
#   Free Software Foundation (http://www.fsf.org).
#
#   Author:  Edward Almasy (almasy@axisdata.com)
#
#   Part of the AxisPHP library v1.2.4
#   For more information see http://www.axisdata.com/AxisPHP/
#

require_once("Axis--Session.php");


# status values (error codes)
define("U_OKAY",                0);
define("U_ERROR",               1);
define("U_BADPASSWORD",         2);
define("U_NOSUCHUSER",          3);
define("U_PASSWORDSDONTMATCH",  4);
define("U_EMAILSDONTMATCH",     5);
define("U_DUPLICATEUSERNAME",   6);
define("U_ILLEGALUSERNAME",     7);
define("U_EMPTYUSERNAME",       8);
define("U_ILLEGALPASSWORD",     9);
define("U_ILLEGALPASSWORDAGAIN",10);
define("U_EMPTYPASSWORD",       11);
define("U_EMPTYPASSWORDAGAIN",  12);
define("U_ILLEGALEMAIL",        13);
define("U_ILLEGALEMAILAGAIN",   14);
define("U_EMPTYEMAIL",          15);
define("U_EMPTYEMAILAGAIN",     16);
define("U_NOTLOGGEDIN",         17);
define("U_MAILINGERROR",        18);
define("U_TEMPLATENOTFOUND",    19);
define("U_DUPLICATEEMAIL",      20);


class User {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    function User(&$SessionOrDb, $UserInfo=NULL)
    {
        # assume constructor will succeed and user is not logged in
        $this->Result = U_OKAY;
        $this->LoggedIn = FALSE;

        # if a session was passed in
        if (is_object($SessionOrDb) && method_exists($SessionOrDb, "Session"))
        {
            # save pointer to session
            $this->Session =& $SessionOrDb;

            # swipe database handle from session
            $this->DB =& $this->Session->DB;

            # if user ID is available from session
            if ($this->Session->Get("APUserId") !== NULL)
            {
                # save user ID
                $this->UserId = $this->Session->Get("APUserId");

                # set flag indicating user is currently logged in
                $this->LoggedIn = TRUE;
            }
        }
        # else if database handle was passed in
        elseif (is_object($SessionOrDb) 
                && method_exists($SessionOrDb, "Database"))
        {
            # save database handle
            $this->DB =& $SessionOrDb;

            # if user ID was passed in
            if (is_int($UserInfo))
            {
                # save user ID
                $this->UserId = $UserInfo;
            }
            # else if user name was passed in
            elseif (is_string($UserInfo))
            {
                # look up user ID in database
                $this->DB->Query("SELECT UserId FROM APUsers"
                        ." WHERE UserName='".addslashes($UserInfo)."'");
                if ($this->DB->NumRowsSelected() > 0)
                {
                    $this->UserId = $this->DB->FetchField("UserId");
                }
                else
                {
                    $this->Result = U_NOSUCHUSER;
                }
            }
        }
        else
        {
            # error out
            $this->Result = U_ERROR;
            exit("ERROR: User object creation attempted without DB or session");
        }
    }

    function Status()
    {
        return $this->Result;
    }

    # return text message corresponding to current (or specified) status code
    function StatusMessage($StatusCode = NULL)
    {
        $APUserStatusMessages = array(
                U_OKAY                => "The operation was successful.",
                U_ERROR               => "There has been an error.",
                U_BADPASSWORD         => "The password you entered was"
                                            ." incorrect.",
                U_NOSUCHUSER          => "No such user name was found.",
                U_PASSWORDSDONTMATCH  => "The new passwords you entered do"
                                            ." not match.",
                U_EMAILSDONTMATCH     => "The e-mail addresses you entered"
                                            ." do not match.",
                U_DUPLICATEUSERNAME   => "The user name you requested is"
                                            ." already in use.",
                U_ILLEGALUSERNAME     => "The user name you requested is too"
                                            ." short, too long, or contains"
                                            ." illegal characters.",
                U_ILLEGALPASSWORD     => "The new password you requested is"
                                            ." too short, too long, or"
                                            ." contains illegal characters.",
                U_ILLEGALEMAIL        => "The e-mail address you entered"
                                            ." appears to be invalid.",
                U_NOTLOGGEDIN         => "The user is not logged in.",
                U_MAILINGERROR        => "An error occurred while attempting"
                                            ." to send e-mail.  Please notify"
                                            ." the system administrator.",
                U_TEMPLATENOTFOUND    => "An error occurred while attempting"
                                            ." to generate e-mail.  Please"
                                            ." notify the system administrator.",
                U_DUPLICATEEMAIL      => "The e-mail address you supplied already"
                                            ." has an account associated with it.",
                );

        return ($StatusCode === NULL) ? $APUserStatusMessages[$this->Result]
                : $APUserStatusMessages[$StatusCode];
    }

    function Delete()
    {
        # clear priv list values
        $this->DB->Query("DELETE FROM APUserPrivileges WHERE UserId = '".$this->UserId."'");

        # delete user record from database
        $this->DB->Query("DELETE FROM APUsers WHERE UserId = '".$this->UserId."'");

        # report to caller that everything succeeded
        $this->Result = U_OKAY;
        return $this->Result;
    }


    # ---- Getting/Setting Values --------------------------------------------

    function Id()
    {
        return $this->UserId;
    }
    function Name()
    {
        return $this->Get("UserName");
    }
    function LastLocation($NewLocation = NULL)
    {
        if ($NewLocation)
        {
            $this->DB->Query("UPDATE APUsers SET"
                    ." LastLocation = '".addslashes($NewLocation)."',"
                    ." LastActiveDate = NOW(),"
                    ." LastIPAddress = '".$_SERVER["REMOTE_ADDR"]."'"
                    ." WHERE UserId = '".addslashes($this->UserId)."'");
            if (isset($this->DBFields))
            {
                $this->DBFields["LastLocation"] = $NewLocation;
                $this->DBFields["LastActiveDate"] = date("Y-m-d H:i:s");
            }
        }
        return $this->Get("LastLocation");
    }
    function LastActiveDate()
    {
        return $this->Get("LastActiveDate");
    }
    function LastIPAddress()
    {
        return $this->Get("LastIPAddress");
    }

    # get value from specified field
    function Get($FieldName)
    {
        return $this->UpdateValue($FieldName);
    }

    # get value (formatted as a date) from specified field
    function GetDate($FieldName, $Format = "")
    {
        # make sure a user is logged in
        if ($this->IsLoggedIn() == FALSE) 
        {  
            $this->Result = U_NOTLOGGEDIN;
            return $this->Result;
        }

        # retrieve specified value from database
        if (strlen($Format) > 0)
        {
            $this->DB->Query("SELECT DATE_FORMAT(`".addslashes($FieldName)."`, '".addslashes($Format)."') AS `".addslashes($FieldName)."` FROM APUsers WHERE UserId='".$this->UserId."'");
        }
        else
        {
            $this->DB->Query("SELECT `".addslashes($FieldName)."` FROM APUsers WHERE UserId='".$this->UserId."'");
        }
        $Record = $this->DB->FetchNextRowArray();

        # return value to caller
        return $Record[$FieldName];
    }

    # set value in specified field
    function Set($FieldName, $NewValue)
    {
        $this->UpdateValue($FieldName, $NewValue);
        $this->Result = U_OKAY;
        return $this->Result;
    }


    # ---- Login Functions ---------------------------------------------------

    function Login($UserName, $Password)
    {
        global $APUserId;

        # error out if we are not part of a session
        if (!isset($this->Session))
        {
            exit("ERROR: User->Login() called on object without session");
        }

        # if user not found in DB
        $this->DB->Query("SELECT * FROM APUsers"
                ." WHERE UserName = '"
                        .addslashes($this->NormalizeUserName($UserName))."'");
        if ($this->DB->NumRowsSelected() < 1)
        {
            # result is no user by that name
            $this->Result = U_NOSUCHUSER;
        }
        else
        {
            # grab password from DB
            $Record = $this->DB->FetchNextRowArray();
            $StoredPassword = $Record["UserPassword"];

            # if supplied password matches encrypted password
            $EncryptedPassword = crypt($Password, $StoredPassword);
            if ($EncryptedPassword == $StoredPassword)
            {
                # result is success
                $this->Result = U_OKAY;

                # store user ID for session
                $this->UserId = $Record["UserId"];
                $APUserId = $this->UserId;
                $this->Session->RegisterVariable("APUserId");

                # update last login date
                $this->DB->Query("UPDATE APUsers SET LastLoginDate = NOW() "
                        ."WHERE UserId = '".$this->UserId."'");

                # set flag to indicate we are logged in
                $this->LoggedIn = TRUE;
            }
            else
            {
                # result is bad password
                $this->Result = U_BADPASSWORD;
            }
        }

        # return result to caller
        return $this->Result;
    }

    # log this user out
    function Logout()
    {
        # if we are part of a session
        if (isset($this->Session))
        {
            # clear user ID for session
            $this->Session->UnregisterVariable("APUserId");
        }

        # set flag to indicate user is no longer logged in
        $this->LoggedIn = FALSE;
    }

    # report whether this user is or is not currently logged in
    function IsLoggedIn() {  return $this->LoggedIn;  }
    function IsNotLoggedIn() {  return !$this->LoggedIn;  }


    # ---- Password Functions ------------------------------------------------
 
    # set new password (with checks against old password)
    function ChangePassword($OldPassword, $NewPassword, $NewPasswordAgain)
    {
        # if we are part of a session make sure a user is logged in
        if (isset($this->Session) && ($this->IsLoggedIn() == FALSE)) 
        {  
            $this->Result = U_NOTLOGGEDIN;  
            return $this->Result;
        }

        # if old password is not correct
        $StoredPassword = $this->DB->Query("SELECT UserPassword FROM APUsers"
                ." WHERE UserId='".$this->UserId."'", "UserPassword");
        $EncryptedPassword = crypt($OldPassword, $StoredPassword);
        if ($EncryptedPassword != $StoredPassword) 
        {  
            # set status to indicate error
            $this->Result = U_BADPASSWORD;
        }
        # else if new password is not legal
        elseif (!$this->IsValidPassword($NewPassword)) 
        {  
            # set status to indicate error
            $this->Result = U_ILLEGALPASSWORD;
        }
        # else if both instances of new password do not match
        elseif ($this->NormalizePassword($NewPassword) 
                != $this->NormalizePassword($NewPasswordAgain))
        {  
            # set status to indicate error
            $this->Result = U_PASSWORDSDONTMATCH;
        }
        else
        {
            # set new password
            $this->SetPassword($NewPassword);

            # set status to indicate password successfully changed
            $this->Result = U_OKAY;
        }

        # report to caller that everything succeeded
        return $this->Result;
    }

    # set new password
    function SetPassword($NewPassword)
    {
        # generate encrypted password
        $EncryptedPassword = crypt($this->NormalizePassword($NewPassword));

        # save encrypted password
        $this->UpdateValue("UserPassword", $EncryptedPassword);
    }

    function CreateNewUserWithEMailedPassword(
            $UserName, $EMail, $EMailAgain, 
            $TemplateFile = "Axis--User--EMailTemplate.txt")
    {
        return CreateNewUserAndMailPasswordFromFile(
                $UserName, $EMail, $EMailAgain, $TemplateFile);
    }

    function CreateNewUserAndMailPasswordFromFile(
            $UserName, $EMail, $EMailAgain, 
            $TemplateFile = "Axis--User--EMailTemplate.txt")
    {
        # load e-mail template from file (first line is subject)
        $Template = file($TemplateFile, 1);
        $EMailSubject = array_shift($Template);
        $EMailBody = join("", $Template);

        return CreateNewUserAndMailPassword(
                $UserName, $EMail, $EMailAgain, $EMailSubject, $EMailBody);
    }

    function CreateNewUserAndMailPassword(
            $UserName, $EMail, $EMailAgain, $EMailSubject, $EMailBody)
    {
        # make sure e-mail addresses match
        if ($EMail != $EMailAgain) 
        {  
            $this->Result = U_EMAILSDONTMATCH;
            return $this->Result;  
        }

        # make sure e-mail address looks valid
        if ($this->IsValidLookingEMailAddress($EMail) == FALSE) 
        {  
            $this->Result = U_ILLEGALEMAIL;
            return $this->Result;  
        }

        # generate random password
        $Password = $this->GetRandomPassword();

        # attempt to create new user with password
        $Result = $this->CreateNewUser($UserName, $Password, $Password);

        # if user creation failed
        if ($Result != U_OKAY)
        {
            # report error result to caller
            return $Result;
        }
        # else
        else
        {
            # set e-mail address in user record
            $this->Set("EMail", $EMail);

            # plug appropriate values into subject and body of e-mail message
            $EMailSubject = str_replace("X-USERNAME-X", $UserName, $EMailSubject);
            $EMailBody = str_replace("X-USERNAME-X", $UserName, $EMailBody);
            $EMailBody = str_replace("X-PASSWORD-X", $Password, $EMailBody);

            # send out e-mail message with new account info
            $Result = mail($EMail, $EMailSubject, $EMailBody);

            # if mailing attempt failed
            if ($Result != TRUE)
            {
                # report error to caller
                $this->Result = U_MAILINGERROR;
                return $this->Result;
            }
            # else
            else
            {
                # report success to caller
                $this->Result = U_OKAY;
                return $this->Result;
            }
        }
    }

    # get code for user to submit to confirm registration
    function GetActivationCode()
    {
        # code is MD5 sum based on user name and encrypted password
        $ActivationCodeLength = 6;
        return $this->GetUniqueCode("Activation", $ActivationCodeLength);
    }

    # check whether confirmation code is valid
    function IsActivationCodeGood($Code)
    {
        return (strtoupper(trim($Code)) == $this->GetActivationCode())
                ? TRUE : FALSE;
    }

    # get/set whether user registration has been confirmed
    function IsActivated($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("RegistrationConfirmed", $NewValue);
    }

    # get code for user to submit to confirm password reset
    function GetResetCode()
    {
        # code is MD5 sum based on user name and encrypted password
        $ResetCodeLength = 10;
        return $this->GetUniqueCode("Reset", $ResetCodeLength);
    }

    # check whether password reset code is valid
    function IsResetCodeGood($Code)
    {
        return (strtoupper(trim($Code)) == $this->GetResetCode())
                ? TRUE : FALSE;
    }

    # send e-mail to user (returns TRUE on success)
    function SendEMail(
            $TemplateTextOrFileName, $FromAddress = NULL, $MoreSubstitutions = NULL)
    {
        # if template is file name
        if (@is_file($TemplateTextOrFileName))
        {
            # load in template from file
            $Template = file($TemplateTextOrFileName, 1);

            # report error to caller if template load failed
            if ($Template == FALSE)
            {
                $this->Status = U_TEMPLATENOTFOUND;
                return $this->Status;
            }

            # join into one text block
            $TemplateTextOrFileName = join("", $Template);
        }

        # split template into lines
        $Template = explode("\n", $TemplateTextOrFileName);

        # strip any comments out of template
        $FilteredTemplate = array();
        foreach ($Template as $Line)
        {
            if (!preg_match("/^[\\s]*#/", $Line))
            {
                $FilteredTemplate[] = $Line;
            }
        }

        # split subject line out of template (first non-comment line in file)
        $EMailSubject = array_shift($FilteredTemplate);
        $EMailBody = join("\n", $FilteredTemplate);

        # set up our substitutions
        $Substitutions = array(
                "X-USERNAME-X" => $this->Get("UserName"),
                "X-EMAILADDRESS-X" => $this->Get("EMail"),
                "X-ACTIVATIONCODE-X" => $this->GetActivationCode(),
                "X-RESETCODE-X" => $this->GetResetCode(),
                "X-IPADDRESS-X" => @$_SERVER["REMOTE_ADDR"],
                );

        # if caller provided additional substitutions
        if (is_array($MoreSubstitutions))
        {
            # add in entries from caller to substitution list
            $Substitutions = array_merge(
                    $Substitutions, $MoreSubstitutions);
        }

        # perform substitutions on subject and body of message
        $EMailSubject = str_replace(array_keys($Substitutions), 
                array_values($Substitutions), $EMailSubject);
        $EMailBody = str_replace(array_keys($Substitutions), 
                array_values($Substitutions), $EMailBody);

        # if caller provided "From" address
        if ($FromAddress)
        {
            # prepend "From" address onto message
            $AdditionalHeaders = "From: ".$FromAddress;
        }

        # send out mail message
        if (isset($AdditionalHeaders))
        {
            $Result = mail($this->Get("EMail"), $EMailSubject, 
                    $EMailBody, $AdditionalHeaders);
        }
        else
        {
            $Result = mail($this->Get("EMail"), $EMailSubject, $EMailBody);
        }

        # report result of mailing attempt to caller
        $this->Status = ($Result == TRUE) ? U_OKAY : U_MAILINGERROR;
        return ($this->Status == U_OKAY);
    }


    # ---- Privilege Functions -----------------------------------------------

    function HasPriv($Privilege, $Privilege2 = NULL, $Privilege3 = NULL, 
            $Privilege4 = NULL, $Privilege5 = NULL, $Privilege6 = NULL)
    {
        # make sure a user is logged in (no privileges if not logged in)
        if ($this->IsLoggedIn() == FALSE) {  return FALSE;  }

        # build database query to check privileges
        $Query = "SELECT COUNT(*) AS PrivCount FROM APUserPrivileges "
                        ."WHERE UserId='".$this->UserId."'"
                            ." AND (Privilege='".$Privilege."'";
        if ($Privilege2 != NULL)
                { $Query .= " OR Privilege='".$Privilege2."'";  }
        if ($Privilege3 != NULL)
                {  $Query .= " OR Privilege='".$Privilege3."'";  }
        if ($Privilege4 != NULL)
                {  $Query .= " OR Privilege='".$Privilege4."'";  }
        if ($Privilege5 != NULL)
                {  $Query .= " OR Privilege='".$Privilege5."'";  }
        if ($Privilege6 != NULL)
                {  $Query .= " OR Privilege='".$Privilege6."'";  }
        $Query .= ")";

        # look for privilege in database
        $PrivCount = $this->DB->Query($Query, "PrivCount");

        # return value to caller
        return ($PrivCount > 0) ? TRUE : FALSE;
    }

    function GrantPriv($Privilege)
    {
        # if user does not already have privilege
        $PrivCount = $this->DB->Query("SELECT COUNT(*) AS PrivCount"
                ." FROM APUserPrivileges"
                ." WHERE UserId='".$this->UserId."'"
                ." AND Privilege='".$Privilege."'",
                "PrivCount");
        if ($PrivCount == 0)
        {
            # add privilege for this user to database
            $this->DB->Query("INSERT INTO APUserPrivileges"
                    ." (UserId, Privilege) VALUES"
                    ." ('".$this->UserId."', ".$Privilege.")");
        }

        # report success to caller
        $this->Result = U_OKAY;
        return $this->Result;
    }

    function RevokePriv($Privilege)
    {
        # remove privilege from database (if present)
        $this->DB->Query("DELETE FROM APUserPrivileges"
                         ." WHERE UserId = '".$this->UserId."'"
                         ." AND Privilege = '".$Privilege."'");

        # report success to caller
        $this->Result = U_OKAY;
        return $this->Result;
    }

    function GetPrivList()
    {
        # read privileges from database and return array to caller
        $this->DB->Query("SELECT Privilege FROM APUserPrivileges"
                ." WHERE UserId='".$this->UserId."'");
        return $this->DB->FetchColumn("Privilege");
    }

    function SetPrivList($NewPrivileges)
    {
        # clear old priv list values
        $this->DB->Query("DELETE FROM APUserPrivileges"
                ." WHERE UserId='".$this->UserId."'");

        # for each priv value passed in
        foreach ($NewPrivileges as $Privilege)
        {
            # set priv for user
            $this->GrantPriv($Privilege);
        }
    }


    # ---- Miscellaneous Functions -------------------------------------------

    # get unique alphanumeric code for user
    function GetUniqueCode($SeedString, $CodeLength)
    {
        return substr(strtoupper(md5(
                $this->Get("UserName").$this->Get("UserPassword").$SeedString)),
                0, $CodeLength);
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $DB;        # handle to SQL database we use to store user information
    var $Session;   # session to use in storing persistent information
    var $UserId;    # user ID number for reference into database
    var $Result;    # result of last operation
    var $LoggedIn;  # flag indicating whether user is logged in
    var $DBFields;  # used for caching user values

    # check whether a user name is valid  (alphanumeric string of 2-24 chars)
    function IsValidUserName($UserName)
    {
        if (preg_match("/^[a-zA-Z0-9]{2,24}$/", $UserName)) {  return TRUE;  } else {  return FALSE;  }
    }

    # check whether a password is valid  (at least 6 characters)
    function IsValidPassword($Password)
    {
        if (strlen(User::NormalizePassword($Password)) < 6)
                {  return FALSE;  } else {  return TRUE;  }
    }

    # check whether an e-mail address looks valid
    function IsValidLookingEMailAddress($EMail)
    {
        if (preg_match("/^[a-zA-Z0-9._\-]+@[a-zA-Z0-9._\-]+\.[a-zA-Z]{2,3}$/", $EMail)) {  return TRUE;  } else {  return FALSE;  }
    }

    # get normalized version of e-mail address
    # (may be called statically)
    function NormalizeEMailAddress($EMailAddress)
    {
        return strtolower(trim($EMailAddress));
    }

    # get normalized version of user name
    # (may be called statically)
    function NormalizeUserName($UserName)
    {
        return trim($UserName);
    }

    # get normalized version of password
    # (may be called statically)
    function NormalizePassword($Password)
    {
        return trim($Password);
    }

    # generate random password
    # generate random password
    function GetRandomPassword($PasswordMinLength = 6, $PasswordMaxLength = 8)
    {
        # seed random number generator
        mt_srand((double)microtime() * 1000000);

        # generate password of requested length
        return sprintf("%06d", mt_rand(pow(10, ($PasswordMinLength - 1)),
                (pow(10, $PasswordMaxLength) - 1)));
    }

    # convenience function to supply parameters to Database->UpdateValue()
    function UpdateValue($FieldName, $NewValue = DB_NOVALUE)
    {
        return $this->DB->UpdateValue("APUsers", $FieldName, $NewValue,
                "UserId = '".$this->UserId."'", $this->DBFields);
    }

    # methods for backward compatibility with earlier versions of User
    function GivePriv($Privilege) {  $this->GrantPriv($Privilege);  }
};


?>
