<?php

#
#   FILE:  SPT--FullRecord.php
#
#   FUNCTIONS PROVIDED:
#       PrintControlledNames()
#           - print list of controlled names associated with the record
#       PrintClassificationNames()
#           - print list of classification names associated with the record
#       UserAlreadyRatedResource()
#           - indicate whether logged in user has already rated resource
#       PrintUserRatingNumber()
#       PrintCumulativeRatingNumber()
#           - print raw rating number
#       PrintUserRatingGraphic()
#       PrintCumulativeRatingGraphic()
#           - call user-supplied routine to display appropriate rating graphic
#       PrintNumberOfRatings()
#           - print number of ratings used to calculate cumulative rating
#       PrintNumberOfRatings()
#           - print an "s" if there was more than one rating
#       ResourceHasBeenRated()
#           - returns TRUE if someone has rated this resource
#       PrintRateResourceLink()
#           - print URL (to go in form tag action field) for rating processing
#       ResourceCommentsAreAvailable()
#           - returns TRUE is this resource has comments
#       PrintResourceComments()
#           - calls PrintResourceComment() for each comment for this resource
#
#   FUNCTIONS EXPECTED:
#       PrintForumMessage($Subject, $Body, $DatePosted, $PosterName, 
#               $PosterEmail, $EditOkay, $EditLink, $DeleteLink)
#           - called by PrintResourceComments() for each comment
#       PrintControlledName()
#           - called by PrintControlledNames() for each controlled name
#       PrintCNStart()
#       PrintCNEnd()
#       PrintNewLine()
#       PrintNameType()
#       PrintClassStart()
#       PrintClassEnd()
#       PrintClassificationInfo()
#           - ???
#       PrintRatingGraphicNoRating()
#           - print graphic (or text) to indicate resource with no rating
#       PrintRatingGraphic0()
#       PrintRatingGraphic1()
#       PrintRatingGraphic2()
#       PrintRatingGraphic3()
#       PrintRatingGraphic4()
#       PrintRatingGraphic5()
#       PrintRatingGraphic6()
#       PrintRatingGraphic7()
#       PrintRatingGraphic8()
#       PrintRatingGraphic9()
#       PrintRatingGraphic10()
#           - print graphic (or text) to indicate specified rating
#        PrintTextField()
#
#   Part of the Scout Portal Toolkit
#   Copyright 2003 Internet Scout Project
#   http://scout.cs.wisc.edu
#

require_once("include/SPT--Common.php");
require_once("include/SPT--CommonSearch.php");
require_once("include/SPT--Resource.php");
require_once("include/SPT--MetadataSchema.php");
PageTitle("Full Resource Record Display");

# ----- EXPORTED FUNCTIONS ---------------------------------------------------

function PrintResourceId()
{
    global $ResourceId;

    print $ResourceId;
}

function PrintTitle()
{
    global $Resource;

    print GetHtmlEscapedString($Resource->Get("Title"));
}

function PrintFieldName()
{
    global $Field;

    print $Field->Name();
}

function PrintValue()
{
    global $Value;

    print stripslashes($Value);
}

function PrintQualifier()
{
    global $Field;
    global $Entry;
    global $Value;
    global $GenericId;

    if ($Field->UsesQualifiers())
    {
        $Qualifier = $Entry[$Field->Name()." Qualifier"];
        # normal field
        if (is_object($Qualifier))
        {
            $Name = $Qualifier->Name();
            if (!empty($Name))
                print "<small> (<a href=\"".$Qualifier->Url()."\">".
                    $Qualifier->Name()."</a>)</small>";
        }            
        # controlled name, options, or classifications, use global Qualifier
        else if (is_array($Qualifier))
        {
            # look up this qualifier
            foreach ($Qualifier as $ArrayId => $QualifierObj)
            {
                if ($ArrayId == $GenericId)
                    break;
            }
            if (is_object($QualifierObj))
            {
                $Name = $QualifierObj->Name();
                if (!empty($Name))
                    print "<small> (<a href=\"".$QualifierObj->Url()."\">".
                        $QualifierObj->Name()."</a>)</small>";
            }
        }
    }
}

function GetLink()
{
    global $Link;

    print $Link;
}

function PrintControlledNameLink()
{
    global $ControlledNameLink;

    print $ControlledNameLink;
}

function CumulativeRatingFieldEnabled()
{
    global $Schema;

    $Field = $Schema->GetFieldByName("Cumulative Rating");
    return $Field->Viewable();
}

function PrintAllFields()
{
    global $Entry, $Schema, $Field, $Value, $Link, $SysConfig, $DB, $Resource;
    global $G_User;
    
    # for each metadata field
    $Fields = $Schema->GetFields(NULL, MDFORDER_DISPLAY);
    foreach ($Fields as $Field)
    {
        $FieldName = $Field->Name();
        $Value = $Entry[$FieldName];
        if ($Resource->UserCanViewField($G_User, $Field) && !empty($Value))
        {
            switch($Field->Type())
            {
                case MDFTYPE_TEXT:
                    # special case to make Title hyperlink to URL
                    if ($FieldName == "Title" && strlen($Entry["Url"]) > 0)
                    {
                        $Link = $Entry["Url"]."\" class=\"ResourceTitle\"";
                        if ($SysConfig->ResourceLaunchesNewWindowEnabled())
                            $Link .= " target=\"_blank\"";
                        PrintLink();
                    }
                    else if ( $FieldName == "Url" )
                    {
                        $Link = $Value;
                        PrintLink();
                    }    
                    else
                        PrintRow();
                    break;

                case MDFTYPE_PARAGRAPH:
                    # translate newlines into <br> prior to printing
                    $Value = str_replace("\n", "<br>", $Value);
                    PrintParagraphField();
                    break;

                case MDFTYPE_NUMBER:
                    if ($FieldName != "Cumulative Rating")
                        PrintRow();
                    break;

                case MDFTYPE_DATE:
                	$Date = $Value;
			        $Value = $Date->Formatted();
                    if (!empty($Value))
                        PrintRow();
                    break;

                case MDFTYPE_TIMESTAMP:
                    PrintRow();
                    break;

                case MDFTYPE_FLAG:
			        if ($Value > 0)
				        $Value = $Field->FlagOnLabel();
			        else
				        $Value = $Field->FlagOffLabel();
				    PrintRow();
				    break;

                case MDFTYPE_TREE:
                    PrintClassificationNames();
                    break;

                case MDFTYPE_CONTROLLEDNAME:
                case MDFTYPE_OPTION:
                    PrintCNStart();
                    PrintControlledNames();
                    PrintCNEnd();
                    break;

                case MDFTYPE_USER:
                    if ($G_User->HasPriv(PRIV_RESOURCEADMIN)) 
                    {
                        $NewUser = $Entry[$FieldName];
                        # make sure UserId is valid
                        if (is_null($NewUser->Get("UserId")))
                            $Value = "Unknown";
                        else
                            $Value = $NewUser->Get("UserName");
                        PrintRow();
                    }
                    break;

                case MDFTYPE_IMAGE:
                    $SPTImage = $Resource->GetByField($Field, TRUE);
                    $ResourceId = $Resource->Id();
                    if ($SPTImage && (is_readable($SPTImage->PreviewUrl())))
                    {
                        PrintImageField($FieldName, 
                        "SPT--FullImage.php?ResourceId=${ResourceId}&FieldName=${FieldName}&edit=0",
                        $SPTImage->Height(),
                        $SPTImage->Width(),
                        $SPTImage->PreviewUrl(),
                        $SPTImage->PreviewHeight(),
                        $SPTImage->PreviewWidth(),
                        $SPTImage->AltText());
                    }
                    break;

                case MDFTYPE_FILE:
                    if ($Resource->UserCanViewField($G_User, $Field))
                    {
                        PrintFileField($Field, $Resource);
                    }
                    break;
            }
        }
    }
}

# display controlled names for this record
function PrintControlledNames()
{
    global $Schema;
    global $Field;
    global $Link;
    global $Value;
    global $Resource;    
    global $GenericId;
    
    # retrieve list of entries for field
    $Names = $Resource->GetByField($Field);
            
    # print field name
    PrintNameType($Field->Name());
                
    # for each entry
    foreach ($Names as $GenericId => $Value)
    {
        # build link to controlled name search
        if ($Field->Type() == MDFTYPE_OPTION)
        {
            $Link = "SPT--AdvancedSearch.php?Q=Y&"
                        ."G".$Field->Id()."=".$GenericId;
        }
        else
        {
            $Link = "SPT--AdvancedSearch.php?Q=Y&"
                        ."F".$Field->Id()."=".urlencode("=".$Value);
        }

         # print entry
         PrintLinkedValue();
    }
}
 
 # display classifications for this record
function PrintClassificationNames()
{
    global $Schema;
    global $Link;
    global $Field;
    global $Resource;
    global $Value;
    global $GenericId;

    $Names = $Resource->Classifications();

    if (count($Names) > 0)
        PrintClassStart();
    foreach ($Names as $ClassificationType => $ClassificationTypes)    
    {
        if ($ClassificationType == $Field->Name())
        {
            foreach ($ClassificationTypes as $GenericId => $Classification)
            {
                $Value = stripslashes($Classification);

                if (!empty($Classification) && isset($Field) 
                    && $Field->Enabled() && $Field->Viewable())
                {
                    $Link = "SPT--AdvancedSearch.php?Q=Y&"
                        ."G".$Field->Id()."=".$GenericId;
                    PrintLinkedValue();
                }
            }
            if (count($Names) > 0)
                PrintClassEnd();
        }
    }
}

function UserAlreadyRatedResource()
{
    global $Resource;
    return ($Resource->Rating() == NULL) ? FALSE : TRUE;
}

function PrintUserRatingNumber()
{
    global $Resource;
    print($Resource->Rating());
}

function PrintCumulativeRatingNumber()
{
    global $Resource;
    print($Resource->CumulativeRating());
}

function PrintUserRatingGraphic()
{
    global $Resource;
    PrintRatingGraphic($Resource->Rating());
}

function PrintCumulativeRatingGraphic()
{
    global $Resource;
    PrintRatingGraphic($Resource->CumulativeRating());
}

function PrintNumberOfRatings()
{
    global $Resource;
    print($Resource->NumberOfRatings());
}

function PrintNumberOfRatingsPlural()
{
    global $Resource;
    if ($Resource->NumberOfRatings() > 1) {  print("s");  }
}

function ResourceHasBeenRated()
{
    global $Resource;
    return ($Resource->NumberOfRatings() > 0) ? TRUE : FALSE;
}

function PrintRateResourceLink()
{
    global $ResourceId;
    print("SPT--RateResource.php?F_ResourceId=${ResourceId}");
}

function ResourceCommentsAreAvailable()
{
    global $Resource;
    return ($Resource->NumberOfComments() > 0) ? TRUE : FALSE;
}

function CheckForEdit($PosterId)
{
    global $User;

    if (($User->Get("UserId") == $PosterId &&
        $User->HasPriv(PRIV_POSTCOMMENTS)) ||
        $User->HasPriv(PRIV_SYSADMIN))
        return true;
    else
        return false;
}

function PrintResourceComments()
{
    global $User;
    global $Resource;
    global $ResourceId;

    # retrieve comments
    $Comments = $Resource->Comments();

    # for each comment
    foreach ($Comments as $Comment)
    {
        $EditOkay = CheckForEdit($Comment->PosterId());
        $MessageId = $Comment->MessageId();
        $EditLink = "SPT--AddResourceComment.php?ResourceId=".
            "$ResourceId&messageid=$MessageId";
        $DeleteLink = "SPT--AddResourceComment.php?ResourceId=".
            "$ResourceId&messageid=$MessageId";
        # print comment
        PrintForumMessage(
                $Comment->Subject(), 
                $Comment->Body(),
                $Comment->DatePosted(), 
                $Comment->PosterName(),
                $Comment->PosterEmail(),
                $EditOkay,
                $EditLink,
                $DeleteLink,
                NULL,
                TRUE);
    }
}

# ----- LOCAL FUNCTIONS ------------------------------------------------------

function PrintRatingGraphic($Rating)
{
    global $Resource;
    if (($Rating === NULL) || ($Resource->NumberOfRatings() < 1))
    {
        PrintRatingGraphicNoRating();
    }
    else
    {
        switch (intval(($Rating + 5) / 10))
        {
            case 0:  PrintRatingGraphic0();  break;
            case 1:  PrintRatingGraphic1();  break;
            case 2:  PrintRatingGraphic2();  break;
            case 3:  PrintRatingGraphic3();  break;
            case 4:  PrintRatingGraphic4();  break;
            case 5:  PrintRatingGraphic5();  break;
            case 6:  PrintRatingGraphic6();  break;
            case 7:  PrintRatingGraphic7();  break;
            case 8:  PrintRatingGraphic8();  break;
            case 9:  PrintRatingGraphic9();  break;
            case 10:  PrintRatingGraphic10();  break;
        }
    }
}



# ----- MAIN -----------------------------------------------------------------

# grab entry information from database
$DB =& new SPTDatabase();

# Get the schema
$Schema = & new MetadataSchema();

# Get the fields for the schema
$Fields = $Schema->GetFields();

# load resource
if (isset($_GET["ResourceId"]))
    $ResourceId = intval($_GET["ResourceId"]);
else
    $ResourceId = $Session->Get("ResourceId");

$Resource = & new Resource($ResourceId);

# load resource values
$Entry = $Resource->GetAsArray();

# cache various values
if (isset($Entry["DateLastVerified"]))
    $DateLastVerified = $Entry["DateLastVerified"];

include("include/SPT--StandardHtmlPageLoad.php");

?>
