<?PHP

#
#   FILE:  SPT--MetadataField.php
#
#   METHODS PROVIDED:
#       Type($NewValue = DB_NOVALUE) 
#           - get/set type of field as enumerated value
#       TypeAsName()
#           - get type of field as type name (string)
#       GetAllowedConversionTypes()
#           - get associative array (enumeration => string) containing field types we can convert to
#       Name($NewValue = DB_NOVALUE) 
#           - get/set name of field
#       HasItemLevelQualifiers($NewValue = DB_NOVALUE)
#           - get/set whether field uses item-level qualifiers
#       AssociatedQualifierList()
#           - get list of qualifiers associated with field
#       UnassociatedQualifierList()
#           - get list of qualifiers not associated with field
#       AssociateWithQualifier($QualifierIdOrObject)
#           - add qualifier association
#       UnassociateWithQualifier($QualifierIdOrObject)
#           - delete qualifier association
#       Status()
#           - get current error status of object
#       Id()
#       DBFieldName()
#           - get field attributes
#       Description($NewValue = DB_NOVALUE)
#       RequiredBySPT($NewValue = DB_NOVALUE)
#       Enabled($NewValue = DB_NOVALUE)
#       Optional($NewValue = DB_NOVALUE)
#       Viewable($NewValue = DB_NOVALUE)
#       AllowMultiple($NewValue = DB_NOVALUE)
#       IncludeInKeywordSearch($NewValue = DB_NOVALUE)
#       IncludeInAdvancedSearch($NewValue = DB_NOVALUE)
#       TextFieldSize($NewValue = DB_NOVALUE)
#       MaxLength($NewValue = DB_NOVALUE)
#       ParagraphRows($NewValue = DB_NOVALUE)
#       ParagraphCols($NewValue = DB_NOVALUE)
#       DefaultValue($NewValue = DB_NOVALUE)
#       MinValue($NewValue = DB_NOVALUE)
#       MaxValue($NewValue = DB_NOVALUE)
#       FlagOnLabel($NewValue = DB_NOVALUE)
#       FlagOffLabel($NewValue = DB_NOVALUE)
#       DateFormat($NewValue = DB_NOVALUE)
#       SearchWeight($NewValue = DB_NOVALUE)
#       MaxHeight($NewValue = DB_NOVALUE)
#       MaxWidth($NewValue = DB_NOVALUE)
#       MaxPreviewHeight($NewValue = DB_NOVALUE)
#       MaxPreviewWidth($NewValue = DB_NOVALUE)
#       MaxThumbnailHeight($NewValue = DB_NOVALUE)
#       MaxThumbnailWidth($NewValue = DB_NOVALUE)
#       DefaultAltText($NewValue = DB_NOVALUE)
#       UsesQualifiers($NewValue = DB_NOVALUE)
#       DefaultQualifier($NewValue = DB_NOVALUE)
#           - get/set field attributes
#
#   AUTHOR:  Edward Almasy
#
#   Part of the Scout Portal Toolkit
#   Copyright 2002-2003 Internet Scout Project
#   http://scout.wisc.edu
#

require_once(dirname(__FILE__)."/SPT--SPTDatabase.php");
require_once(dirname(__FILE__)."/SPT--MetadataSchema.php");
require_once(dirname(__FILE__)."/SPT--Qualifier.php");
require_once(dirname(__FILE__)."/SPT--QualifierFactory.php");
require_once(dirname(__FILE__)."/SPT--ControlledNameFactory.php");
require_once(dirname(__FILE__)."/SPT--ClassificationFactory.php");


# metadata field types 
# (must parallel MetadataFields.FieldType declaration in SPT--CreateTables.sql 
#        and $FieldTypeDBEnums declaration below)
define("MDFTYPE_TEXT",           1);
define("MDFTYPE_PARAGRAPH",      2);
define("MDFTYPE_NUMBER",         4);
define("MDFTYPE_DATE",           8);
define("MDFTYPE_TIMESTAMP",      16);
define("MDFTYPE_FLAG",           32);
define("MDFTYPE_TREE",           64);
define("MDFTYPE_CONTROLLEDNAME", 128);
define("MDFTYPE_OPTION",         256);
define("MDFTYPE_USER",           512);
define("MDFTYPE_IMAGE",          1024);
define("MDFTYPE_FILE",           2048);

# error status codes
define("MDFSTAT_OK",                1);
define("MDFSTAT_DUPLICATENAME",     2);
define("MDFSTAT_DUPLICATEDBCOLUMN", 4);
define("MDFSTAT_ILLEGALNAME",       8);
define("MDFSTAT_FIELDDOESNOTEXIST", 16);


class MetadataField {

    # ---- PUBLIC INTERFACE --------------------------------------------------
    
    # get current error status of object
    function Status() {  return $this->ErrorStatus;  }

    # get/set type of field as enumerated value
    function Type($NewValue = DB_NOVALUE) 
    {
        global $FieldTypePHPEnums;
        
        # if new value supplied
        if (($NewValue != DB_NOVALUE)
             && ($NewValue != $FieldTypePHPEnums[$this->DBFields["FieldType"]]))
        {
            # update database fields and store new type
            $this->ModifyField(NULL, $NewValue);
        }
        
        # return type to caller
        return $FieldTypePHPEnums[$this->DBFields["FieldType"]];
    }
    
    # get type of field as type name (string)
    function TypeAsName()
    {
        return $this->DBFields["FieldType"];
    }
    
    # get/set name of field
    function Name($NewName = DB_NOVALUE) 
    {  
        # if new name specified
        if (($NewName != DB_NOVALUE) 
            && (trim($NewName) != $this->DBFields["FieldName"]))
        {
            # if field name is invalid
            $NewName = trim($NewName);
            if (strlen($this->NormalizeFieldNameForDB($NewName)) < 1)
            {
                # set error status to indicate illegal name
                $this->ErrorStatus = MDFSTAT_ILLEGALNAME;
            }
            else
            {
                # check for duplicate name
                $DuplicateCount = $this->DB->Query("SELECT COUNT(*) AS RecordCount FROM MetadataFields "
                                             ."WHERE FieldName = '".addslashes($NewName)."'", 
                                             "RecordCount");

                # if field name is duplicate
                if ($DuplicateCount > 0)
                {
                    # set error status to indicate duplicate name
                    $this->ErrorStatus = MDFSTAT_DUPLICATENAME;
                }
                else
                {
                    # modify database declaration to reflect new field name
                    $this->ErrorStatus = MDFSTAT_OK;
                    $this->ModifyField($NewName);
                }
            }
        }
        
        # return value to caller
        return $this->DBFields["FieldName"];
    }

    # get associative array (enumeration => string) containing field types we can convert to
    function GetAllowedConversionTypes()
    {
        # determine type list based on our type
        switch ($this->Type())
        {
            case MDFTYPE_TEXT:
            case MDFTYPE_PARAGRAPH:
            case MDFTYPE_NUMBER:
            case MDFTYPE_FLAG:
                $AllowedTypes = array(
                        MDFTYPE_TEXT             => "Text",
                        MDFTYPE_PARAGRAPH         => "Paragraph",
                        MDFTYPE_NUMBER             => "Number",
                        MDFTYPE_FLAG             => "Flag",
                        );
                break;
                
            case MDFTYPE_CONTROLLEDNAME:
            case MDFTYPE_OPTION:
                $AllowedTypes = array(
                        MDFTYPE_CONTROLLEDNAME     => "ControlledName",
                        MDFTYPE_OPTION             => "Option",
                        );
                break;
                
            case MDFTYPE_DATE:
                $AllowedTypes = array(
                        MDFTYPE_TEXT             => "Text",
                        MDFTYPE_DATE             => "Date",
                        );
                break;
                
            case MDFTYPE_IMAGE:
                $AllowedTypes = array(
                        MDFTYPE_TEXT             => "Text",
                        MDFTYPE_IMAGE             => "Still Image",
                        );
                break;
                
            case MDFTYPE_TIMESTAMP:
            case MDFTYPE_TREE:
            case MDFTYPE_USER:
            case MDFTYPE_FILE:
            default:
                $AllowedTypes = array();
                break;
        }
        
        # return type list to caller
        return $AllowedTypes;
    }
    

    # get/set whether item is temporary instance
    function IsTempItem($NewSetting = NULL)
    {
        $ItemTableName = "MetadataFields";
        $ItemIdFieldName = "FieldId";
        $ItemFactoryObjectName = "MetadataSchema";
        $ItemAssociationTables = array(
                "FieldQualifierInts",
                );
        $ItemAssociationFieldName = "MetadataFieldId";
        
        # if new temp item setting supplied
        if ($NewSetting !== NULL)
        {
            # if caller requested to switch
            if ((($this->Id() < 0) && ($NewSetting == FALSE))
                    || (($this->Id() >= 0) && ($NewSetting == TRUE)))
            {
                # if field name is invalid
                if (strlen($this->NormalizeFieldNameForDB($this->Name())) < 1)
                {
                    # set error status to indicate illegal name
                    $this->ErrorStatus = MDFSTAT_ILLEGALNAME;
                }
                else
                {
                    # lock DB tables to prevent next ID from being grabbed
                    $DB =& $this->DB;
                    $DB->Query("LOCK TABLES ".$ItemTableName." WRITE,".
                            "APSessions WRITE, APSessionData WRITE");

                    # get next temp item ID
                    $OldItemId = $this->Id();
                    $Factory =& new $ItemFactoryObjectName();
                    if ($NewSetting == TRUE)
                    {
                        $this->Id = $Factory->GetNextTempItemId();
                    }
                    else
                    {
                        $this->Id = $Factory->GetNextItemId();
                    }

                    # change item ID
                    $DB->Query("UPDATE ".$ItemTableName." SET ".$ItemIdFieldName." = ".
                        $this->Id.  " WHERE ".$ItemIdFieldName." = ".$OldItemId);

                    # release DB tables
                    $DB->Query("UNLOCK TABLES");

                    # change associations
                    foreach ($ItemAssociationTables as $TableName)
                    {
                        $DB->Query("UPDATE ".$TableName." SET ".$ItemAssociationFieldName." = ".
                                $this->Id.  " WHERE ".$ItemAssociationFieldName." = ".$OldItemId);
                    }

                    # if changing item from temp to non-temp
                    if ($NewSetting == FALSE)
                    {
                        # add any needed database fields and/or entries
                        $this->AddDatabaseFields();
                    }
                }
            }
        }
        
        # report to caller whether we are a temp item
        return ($this->Id() < 0) ? TRUE : FALSE;
    }
    
    # get field attributes
    function Id() {  return $this->DBFields["FieldId"];  }
    function DBFieldName() {  return $this->DBFields["DBFieldName"];  }
    
    # get/set field attributes
    function Description($NewValue = DB_NOVALUE) {  return $this->UpdateValue("Description", $NewValue);  }
    function RequiredBySPT($NewValue = DB_NOVALUE) {  return $this->UpdateValue("RequiredBySPT", $NewValue);  }
    function Enabled($NewValue = DB_NOVALUE) {  return $this->UpdateValue("Enabled", $NewValue);  }
    function Optional($NewValue = DB_NOVALUE) {  return $this->UpdateValue("Optional", $NewValue);  }
    function Viewable($NewValue = DB_NOVALUE) {  return $this->UpdateValue("Viewable", $NewValue);  }
    function AllowMultiple($NewValue = DB_NOVALUE) {  return $this->UpdateValue("AllowMultiple", $NewValue);  }
    function IncludeInKeywordSearch($NewValue = DB_NOVALUE) {  return $this->UpdateValue("IncludeInKeywordSearch", $NewValue);  }
    function IncludeInAdvancedSearch($NewValue = DB_NOVALUE) {  return $this->UpdateValue("IncludeInAdvancedSearch", $NewValue);  }
    function IncludeInRecommenderSystem($NewValue = DB_NOVALUE) {  return $this->UpdateValue("IncludeInRecommenderSystem", $NewValue);  }
    function TextFieldSize($NewValue = DB_NOVALUE) {  return $this->UpdateValue("TextFieldSize", $NewValue);  }
    function MaxLength($NewValue = DB_NOVALUE) {  return $this->UpdateValue("MaxLength", $NewValue);  }
    function ParagraphRows($NewValue = DB_NOVALUE) {  return $this->UpdateValue("ParagraphRows", $NewValue);  }
    function ParagraphCols($NewValue = DB_NOVALUE) {  return $this->UpdateValue("ParagraphCols", $NewValue);  }
    function DefaultValue($NewValue = DB_NOVALUE) {  return $this->UpdateValue("DefaultValue", $NewValue);  }
    function MinValue($NewValue = DB_NOVALUE) {  return $this->UpdateValue("MinValue", $NewValue);  }
    function MaxValue($NewValue = DB_NOVALUE) {  return $this->UpdateValue("MaxValue", $NewValue);  }
    function FlagOnLabel($NewValue = DB_NOVALUE) {  return $this->UpdateValue("FlagOnLabel", $NewValue);  }
    function FlagOffLabel($NewValue = DB_NOVALUE) {  return $this->UpdateValue("FlagOffLabel", $NewValue);  }
    function DateFormat($NewValue = DB_NOVALUE) {  return $this->UpdateValue("DateFormat", $NewValue);  }
    function SearchWeight($NewValue = DB_NOVALUE) {  return $this->UpdateValue("SearchWeight", $NewValue);  }
    function RecommenderWeight($NewValue = DB_NOVALUE) {  return $this->UpdateValue("RecommenderWeight", $NewValue);  }
    function MaxHeight($NewValue = DB_NOVALUE) {  return $this->UpdateValue("MaxHeight", $NewValue);  }
    function MaxWidth($NewValue = DB_NOVALUE) {  return $this->UpdateValue("MaxWidth", $NewValue);  }
    function MaxPreviewHeight($NewValue = DB_NOVALUE) {  return $this->UpdateValue("MaxPreviewHeight", $NewValue);  }
    function MaxPreviewWidth($NewValue = DB_NOVALUE) {  return $this->UpdateValue("MaxPreviewWidth", $NewValue);  }
    function MaxThumbnailHeight($NewValue = DB_NOVALUE) {  return $this->UpdateValue("MaxThumbnailHeight", $NewValue);  }
    function MaxThumbnailWidth($NewValue = DB_NOVALUE) {  return $this->UpdateValue("MaxThumbnailWidth", $NewValue);  }
    function DefaultAltText($NewValue = DB_NOVALUE) {  return $this->UpdateValue("DefaultAltText", $NewValue);  }
    function UsesQualifiers($NewValue = DB_NOVALUE) {  return $this->UpdateValue("UsesQualifiers", $NewValue);  }
    function DefaultQualifier($NewValue = DB_NOVALUE) {  return $this->UpdateValue("DefaultQualifier", $NewValue);  }
    function UseForOaiSets($NewValue = DB_NOVALUE) {  return $this->UpdateValue("UseForOaiSets", $NewValue);  }
    function ViewingPrivilege($NewValue = DB_NOVALUE) {  return $this->UpdateValue("ViewingPrivilege", $NewValue);  }
    function AuthoringPrivilege($NewValue = DB_NOVALUE) {  return $this->UpdateValue("AuthoringPrivilege", $NewValue);  }
    function EditingPrivilege($NewValue = DB_NOVALUE) {  return $this->UpdateValue("EditingPrivilege", $NewValue);  }

    # get possible values (only meaningful for Trees, Controlled Names, Options, Flags)
    # (index for returned array is IDs for values)
    function GetPossibleValues($MaxNumberOfValues = NULL)
    {
        # retrieve values based on field type
        switch ($this->Type())
        {
            case MDFTYPE_TREE:
                $QueryString = "SELECT ClassificationId,ClassificationName"
                        ." FROM Classifications WHERE FieldId = ".$this->Id()
                        ." ORDER BY ClassificationName";
                if ($MaxNumberOfValues) 
                        {  $QueryString .= " LIMIT ".intval($MaxNumberOfValues);  }
                $this->DB->Query($QueryString);
                $PossibleValues = $this->DB->FetchColumn(
                        "ClassificationName", "ClassificationId");
                break;

            case MDFTYPE_CONTROLLEDNAME:
            case MDFTYPE_OPTION:
                $QueryString = "SELECT ControlledNameId,ControlledName"
                        ." FROM ControlledNames WHERE FieldId = ".$this->Id()
                        ." ORDER BY ControlledName";
                if ($MaxNumberOfValues) 
                        {  $QueryString .= " LIMIT ".intval($MaxNumberOfValues);  }
                $this->DB->Query($QueryString);
                $PossibleValues = $this->DB->FetchColumn(
                        "ControlledName", "ControlledNameId");
                break;

            case MDFTYPE_FLAG:
                $PossibleValues[0] = $this->FlagOffLabel();
                $PossibleValues[1] = $this->FlagOnLabel();
                break;

            default:
                # for everything else return an empty array
                $PossibleValues = array();
                break;
        }

        # return array of possible values to caller
        return $PossibleValues;
    }

    # get count of possible values (only meaningful for Trees, Controlled Names, Options)
    function GetCountOfPossibleValues()
    {
        # retrieve values based on field type
        switch ($this->Type())
        {
            case MDFTYPE_TREE:
                $Count = $this->DB->Query("SELECT count(*) AS ValueCount"
                        ." FROM Classifications WHERE FieldId = ".$this->Id(), 
                        "ValueCount");
                break;

            case MDFTYPE_CONTROLLEDNAME:
            case MDFTYPE_OPTION:
                $Count = $this->DB->Query("SELECT count(*) AS ValueCount"
                        ." FROM ControlledNames WHERE FieldId = ".$this->Id(),
                        "ValueCount");
                break;

            case MDFTYPE_FLAG:
                $Count = 2;
                break;

            default:
                # for everything else return an empty array
                $Count = 0;
                break;
        }

        # return count of possible values to caller
        return $Count;
    }

    # get ID for specified value (only meaningful for Trees / Controlled Names / Options)
    # (returns NULL if value not found)
    function GetIdForValue($Value)
    {
        # retrieve ID based on field type
        switch ($this->Type())
        {
            case MDFTYPE_TREE:
                $Id = $this->DB->Query("SELECT ClassificationId FROM Classifications"
                        ." WHERE ClassificationName = '".addslashes($Value)."'"
                        ." AND FieldId = ".$this->Id(),
                        "ClassificationId");
                break;

            case MDFTYPE_CONTROLLEDNAME:
            case MDFTYPE_OPTION:
                $Id = $this->DB->Query("SELECT ControlledNameId FROM ControlledNames"
                        ." WHERE ControlledName = '".addslashes($Value)."'"
                        ." AND FieldId = ".$this->Id(),
                        "ControlledNameId");
                break;

            default:
                # for everything else return NULL
                $Id = NULL;
                break;
        }

        # return ID for value to caller
        return $Id;
    }

    # get value for specified ID (only meaningful for Trees / Controlled Names / Options)
    # (returns NULL if ID not found)
    function GetValueForId($Id)
    {
        # retrieve ID based on field type
        switch ($this->Type())
        {
            case MDFTYPE_TREE:
                $Value = $this->DB->Query("SELECT ClassificationName FROM Classifications"
                        ." WHERE ClassificationId = '".intval($Id)."'"
                        ." AND FieldId = ".$this->Id(),
                        "ClassificationName");
                break;

            case MDFTYPE_CONTROLLEDNAME:
            case MDFTYPE_OPTION:
                $Value = $this->DB->Query("SELECT ControlledName FROM ControlledNames"
                        ." WHERE ControlledNameId = '".intval($Id)."'"
                        ." AND FieldId = ".$this->Id(),
                        "ControlledName");
                break;

            default:
                # for everything else return NULL
                $Value = NULL;
                break;
        }

        # return ID for value to caller
        return $Value;
    }



    # get/set whether field uses item-level qualifiers
    function HasItemLevelQualifiers($NewValue = DB_NOVALUE)
    {
        # if value provided different from present value
        if (($NewValue != DB_NOVALUE)
            && ($NewValue != $this->DBFields["HasItemLevelQualifiers"]))
        {
            # check if qualifier column currently exists
            $QualColName = $this->DBFieldName()."Qualifier";
            $QualColExists = $this->DB->FieldExists("Resources", $QualColName);

            # if new value indicates qualifiers should now be used
            if ($NewValue == TRUE)
            {
                # if qualifier column does not exist in DB for this field
                if ($QualColExists == FALSE)
                {
                    # add qualifier column in DB for this field
                    $this->DB->Query("ALTER TABLE Resources ADD COLUMN `"
                                     .$QualColName."` INT");
                }
            }
            else
            {
                # if qualifier column exists in DB for this field
                if ($QualColExists == TRUE)
                {
                    # remove qualifier column from DB for this field
                    $this->DB->Query("ALTER TABLE Resources DROP COLUMN `"
                                     .$QualColName."`");
                }
            }
        }

        return $this->UpdateValue("HasItemLevelQualifiers", $NewValue);  
    }
    
    # get list of qualifiers associated with field
    function AssociatedQualifierList()
    {
        # start with empty list
        $List = array();
        
        # for each associated qualifier
        $this->DB->Query("SELECT QualifierId FROM FieldQualifierInts"
                     ." WHERE MetadataFieldId = ".$this->DBFields["FieldId"]);
        while ($Record = $this->DB->FetchRow())
        {
            # load qualifier object
            $Qual =& new Qualifier($Record["QualifierId"]);
            
            # add qualifier ID and name to list
            $List[$Qual->Id()] = $Qual->Name();
        }
        
        # return list to caller
        return $List;
    }
    
    # get list of qualifiers not associated with field
    function UnassociatedQualifierList()
    {
        # grab list of associated qualifiers
        $AssociatedQualifiers = $this->AssociatedQualifierList();
        
        # get list of all qualifiers
        $QFactory =& new QualifierFactory();
        $AllQualifiers = $QFactory->QualifierList();
        
        # return list of unassociated qualifiers
        return array_diff($AllQualifiers, $AssociatedQualifiers);
    }
    
    # add qualifier association
    function AssociateWithQualifier($QualifierIdOrObject)
    {
        # if qualifier object passed in
        if (is_object($QualifierIdOrObject))
        {
            # grab qualifier ID from object
            $QualifierIdOrObject = $QualifierIdOrObject->Id();
        }
        
        # if not already associated
        $RecordCount = $this->DB->Query(
            "SELECT COUNT(*) AS RecordCount FROM FieldQualifierInts"
            ." WHERE QualifierId = ".$QualifierIdOrObject
            ." AND MetadataFieldId = ".$this->Id(), "RecordCount");
        if ($RecordCount < 1)
        {
            # associate field with qualifier
            $this->DB->Query("INSERT INTO FieldQualifierInts SET"
                             ." QualifierId = ".$QualifierIdOrObject.","
                             ." MetadataFieldId = ".$this->Id());
        }
    }
    
    # delete qualifier association
    function UnassociateWithQualifier($QualifierIdOrObject)
    {
        # if qualifier object passed in
        if (is_object($QualifierIdOrObject))
        {
            # grab qualifier ID from object
            $QualifierIdOrObject = $QualifierIdOrObject->Id();
        }
        
        # delete intersection record from database
        $this->DB->Query("DELETE FROM FieldQualifierInts WHERE QualifierId = "
                         .$QualifierIdOrObject." AND MetadataFieldId = ".
                         $this->Id());
    }

    # retrieve item factory object for this field
    function GetFactory()
    {
        switch ($this->Type())
        {
            case MDFTYPE_TREE:
                $Factory = new ClassificationFactory($this->Id());
                break;

            case MDFTYPE_CONTROLLEDNAME:
            case MDFTYPE_OPTION:
                $Factory = new ControlledNameFactory($this->Id());
                break;

            default:
                $Factory = NULL;
                break;
        }

        return $Factory;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------
    
    var $DB;
    var $DBFields;
    var $ErrorStatus;

    # object constructor (only for use by MetadataSchema object)
    function MetadataField($FieldId, $FieldName = NULL, $FieldType = NULL, 
                           $Optional = TRUE, $DefaultValue = NULL)
    {
        global $FieldTypeDBEnums;
        global $FieldTypePHPEnums;
        
        # assume everything will be okay
        $this->ErrorStatus = MDFSTAT_OK;
        
        # grab our own database handle
        $this->DB =& new SPTDatabase();
        $DB =& $this->DB;

        # if field ID supplied
        if ($FieldId != NULL)
        {
            # look up field in database
            $DB->Query("SELECT * FROM MetadataFields WHERE FieldId = ".intval($FieldId));
            $Record = $DB->FetchRow();
        }
        
        # if no field ID supplied or if record not found in database
        if (($FieldId == NULL) || ($Record == NULL))
        {
            # error out if valid field type not supplied
            if (empty($FieldTypeDBEnums[$FieldType]))
            {
                $this->ErrorStatus = MDFSTAT_FIELDDOESNOTEXIST;
                return;
            }
            
            # if field name supplied
            $FieldName = trim($FieldName);
            if (strlen($FieldName) > 0)
            {
                # error out if field name is duplicate
                $DuplicateCount = $DB->Query("SELECT COUNT(*) AS RecordCount FROM MetadataFields "
                                             ."WHERE FieldName = '".addslashes($FieldName)."'", 
                                             "RecordCount");
                if ($DuplicateCount > 0)
                {
                    $this->ErrorStatus = MDFSTAT_DUPLICATENAME;
                    return;
                }
            }
            
            # grab current user ID
            global $User;
            $UserId = $User->Get("UserId");

            # lock DB tables and get next temporary field ID
            $Schema =& new MetadataSchema();
            $DB->Query("LOCK TABLES MetadataFields WRITE");
            $FieldId = $Schema->GetNextTempItemId();

            # add field to MDF table in database
            $DB->Query("INSERT INTO MetadataFields "
                  ."(FieldId, FieldName, FieldType, Optional, DefaultValue, LastModifiedById) VALUES "
                  ."('".$FieldId."', "
                  ."'".addslashes($FieldName)."', "
                  ."'".$FieldTypeDBEnums[$FieldType]."', "
                  .($Optional ? 1 : 0).", "
                  ."'".$DefaultValue."',"
                  ."'".$UserId."')");
            
            # release DB tables
            $DB->Query("UNLOCK TABLES");

            # re-read record from database
            $DB->Query("SELECT * FROM MetadataFields WHERE FieldId = $FieldId");
            $this->DBFields = $DB->FetchRow();
            $this->DBFields["DBFieldName"] = 
                    $this->NormalizeFieldNameForDB($this->DBFields["FieldName"]);
            
            # set field order values for new field
            $FieldCount = $DB->Query("SELECT COUNT(*) AS FieldCount FROM MetadataFields", "FieldCount");
            $this->OrderPosition(MDFORDER_DISPLAY, ($FieldCount + 1));
            $this->OrderPosition(MDFORDER_EDITING, ($FieldCount + 1));
        }
        else
        {
            # save values locally
            $this->DBFields = $Record;
            $this->DBFields["DBFieldName"] = 
                    $this->NormalizeFieldNameForDB($Record["FieldName"]);
        }
    }

    # remove field from database (only for use by MetadataSchema object)
    function Drop()
    {
        global $FieldTypeDBEnums;
        global $FieldTypePHPEnums;
        
        # clear other database entries as appropriate for field type
        $DB =& $this->DB;
        $DBFieldName = $this->DBFields["DBFieldName"];
        switch ($FieldTypePHPEnums[$this->DBFields["FieldType"]])
        {
            case MDFTYPE_TEXT:
            case MDFTYPE_PARAGRAPH:
            case MDFTYPE_NUMBER:
            case MDFTYPE_USER:
            case MDFTYPE_IMAGE:
            case MDFTYPE_TIMESTAMP:
                # remove field from resources table
                if ($DB->FieldExists("Resources", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE Resources DROP COLUMN `".$DBFieldName."`");
                }
                break;
                
            case MDFTYPE_FLAG:
                # remove field from resources table
                if ($DB->FieldExists("Resources", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE Resources DROP COLUMN `".$DBFieldName."`");
                }
                
                # remove field from saved user searches table
                if ($DB->FieldExists("UserSearch", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE UserSearch DROP COLUMN `".$DBFieldName."`");
                }
                break;
                
            case MDFTYPE_DATE:
                # remove fields from resources table
                if ($DB->FieldExists("Resources", $DBFieldName."Begin"))
                {
                    $DB->Query("ALTER TABLE Resources DROP COLUMN `".$DBFieldName."Begin`");
                    $DB->Query("ALTER TABLE Resources DROP COLUMN `".$DBFieldName."End`");
                    $DB->Query("ALTER TABLE Resources DROP COLUMN `".$DBFieldName."Precision`");
                }
                break;
                
            case MDFTYPE_TREE:
                $DB->Query("SELECT ClassificationId FROM Classifications "
                           ."WHERE FieldId = ".$this->Id());
                $TempDB =& new SPTDatabase();
                while ($ClassificationId = $DB->FetchField("ClassificationId"))
                {
                    # remove any resource / name intersections
                    $TempDB->Query("DELETE FROM ResourceClassInts WHERE "
                                   ."ClassificationId = ".$ClassificationId);
                    
                    # remove controlled name
                    $TempDB->Query("DELETE FROM Classifications WHERE "
                                   ."ClassificationId = ".$ClassificationId);
                }
                break;

            case MDFTYPE_CONTROLLEDNAME:
            case MDFTYPE_OPTION:
                $DB->Query("SELECT ControlledNameId FROM ControlledNames "
                           ."WHERE FieldId = ".$this->Id());
                $TempDB =& new SPTDatabase();
                while ($ControlledNameId = $DB->FetchField("ControlledNameId"))
                {
                    # remove any resource / name intersections
                    $TempDB->Query("DELETE FROM ResourceNameInts WHERE "
                                   ."ControlledNameId = ".$ControlledNameId);
                    
                    # remove any variant names
                    $TempDB->Query("DELETE FROM VariantNames WHERE "
                                   ."ControlledNameId = ".$ControlledNameId);
                    
                    # remove controlled name
                    $TempDB->Query("DELETE FROM ControlledNames WHERE "
                                   ."ControlledNameId = ".$ControlledNameId);
                }

                # drop entry from saved user searches table (if type is option)
                if ($DB->FieldExists("UserSearch", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE UserSearch DROP COLUMN `".$DBFieldName."`");
                }
                break;

            case MDFTYPE_FILE:
                # for each file associated with this field
                $DB->Query("SELECT FileId FROM Files WHERE FieldId = '".$this->Id()."'");
                while ($FileId = $DB->FetchRow())
                {
                    # delete file
                    $File =& new File(intval($FileId));
                    $File->Delete();
                }
                break;
        }
        
        # remove field from database
        $DB->Query("DELETE FROM MetadataFields "
                   ."WHERE FieldId = '".$this->DBFields["FieldId"]."'");
        
        # remove any qualifier associations
        $DB->Query("DELETE FROM FieldQualifierInts WHERE MetadataFieldId = '"
                   .$this->DBFields["FieldId"]."'");
    }

    # modify any database fields
    function ModifyField($NewName = NULL, $NewType = NULL)
    {
        global $FieldTypeDBEnums;
        global $FieldTypePHPEnums;
        
        # grab old DB field name
        $OldDBFieldName = $this->DBFields["DBFieldName"];
        $OldFieldType = NULL;
        
        # if new field name supplied
        if ($NewName != NULL)
        {
            # cache the old name for options and controllednames below
            $OldName = $this->DBFields["FieldName"];

            # store new name
            $this->UpdateValue("FieldName", $NewName);
            
            # determine new DB field name
            $NewDBFieldName = $this->NormalizeFieldNameForDB($NewName);
            
            # store new database field name
            $this->DBFields["DBFieldName"] = $NewDBFieldName;
        }
        else
        {
            # set new field name equal to old field name
            $NewDBFieldName = $OldDBFieldName;
        }
        
        # if new type supplied
        if ($NewType != NULL)
        {
            # grab old field type
            $OldFieldType = $FieldTypePHPEnums[$this->DBFields["FieldType"]];
            
            # store new field type
            $this->UpdateValue("FieldType", $FieldTypeDBEnums[$NewType]);
        }

        # if this is not a temporary field
        if ($this->Id() >= 0)
        {
            # modify field in DB as appropriate for field type
            $DB =& $this->DB;
            $FieldType = $FieldTypePHPEnums[$this->DBFields["FieldType"]];
            switch ($FieldType)
            {
                case MDFTYPE_TEXT:
                case MDFTYPE_PARAGRAPH:
                    # alter field declaration in Resources table
                    $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                               .$OldDBFieldName."` `"
                               .$NewDBFieldName."` TEXT "
                               .($this->DBFields["Optional"] ? "" : "NOT NULL"));
                    break;

                case MDFTYPE_NUMBER:
                case MDFTYPE_USER:
                    # alter field declaration in Resources table
                    $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                               .$OldDBFieldName."` `"
                               .$NewDBFieldName."` INT "
                               .($this->DBFields["Optional"] ? "" : "NOT NULL"));
                    break;

                case MDFTYPE_FILE:
                    # alter field declaration in Resources table
                    $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                               .$OldDBFieldName."` `"
                               .$NewDBFieldName."` TEXT");
                    break;

                case MDFTYPE_IMAGE:
                    # alter field declaration in Resources table
                    $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                               .$OldDBFieldName."` `"
                               .$NewDBFieldName."` INT");
                    break;

                case MDFTYPE_FLAG:
                    # alter field declaration in Resources table
                    $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                               .$OldDBFieldName."` `"
                               .$NewDBFieldName."` INT "
                               .($this->DBFields["Optional"] ? "" : "NOT NULL"));

                    # if old type was flag
                    if ($OldFieldType == MDFTYPE_FLAG)
                    {
                        # alter field declaration in saved user searches table
                        $DB->Query("ALTER TABLE UserSearch CHANGE COLUMN `"
                                   .$OldDBFieldName."` `"
                                   .$NewDBFieldName."` INT "
                                   .($this->DBFields["Optional"] ? "" : "NOT NULL"));
                    }
                    break;

                case MDFTYPE_DATE:
                    # if new type supplied and new type is different from old
                    if (($NewType != NULL) && ($NewType != $OldFieldType))
                    {
                        # if old type was time stamp
                        if ($OldFieldType == MDFTYPE_TIMESTAMP)
                        {
                            # change time stamp field in resources table to begin date
                            $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                                       .$OldDBFieldName."` `"
                                       .$NewDBFieldName."Begin` DATE "
                                       .($this->DBFields["Optional"] ? "" : "NOT NULL"));

                            # add end date and precision fields
                            $DB->Query("ALTER TABLE Resources ADD COLUMN `".$NewDBFieldName."End"
                                       ."` DATE ".($Optional ? "" : "NOT NULL"));
                            $DB->Query("ALTER TABLE Resources ADD COLUMN `".$NewDBFieldName."Precision`"
                                       ." INT ".($Optional ? "" : "NOT NULL"));

                            # set precision to reflect time stamp content
                            $DB->Query("UPDATE Resources SET `".$NewDBFieldName."Precision` = "
                                       .(DATEPRE_BEGINYEAR|DATEPRE_BEGINMONTH|DATEPRE_BEGINDAY));
                        }
                        else
                        {
                            exit("<br>ERROR:  Attempt to convert metadata field to date from type other than timestamp<br>\n");
                        }
                    }
                    else
                    {
                        # change name of fields
                        $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                                   .$OldDBFieldName."Begin` `"
                                   .$NewDBFieldName."Begin` DATE "
                                   .($this->DBFields["Optional"] ? "" : "NOT NULL"));
                        $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                                   .$OldDBFieldName."End` `"
                                   .$NewDBFieldName."End` DATE "
                                   .($this->DBFields["Optional"] ? "" : "NOT NULL"));
                        $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                                   .$OldDBFieldName."Precision` `"
                                   .$NewDBFieldName."Precision` INT "
                                   .($this->DBFields["Optional"] ? "" : "NOT NULL"));
                    }
                    break;

                case MDFTYPE_TIMESTAMP:
                    # if new type supplied and new type is different from old
                    if (($NewType != NULL) && ($NewType != $OldFieldType))
                    {
                        # if old type was date
                        if ($OldFieldType == MDFTYPE_DATE)
                        {
                            # change begin date field in resource table to time stamp
                            $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                                       .$OldDBFieldName."Begin` `"
                                       .$NewDBFieldName."` DATETIME "
                                       .($this->DBFields["Optional"] ? "" : "NOT NULL"));

                            # drop end date and precision fields
                            $DB->Query("ALTER TABLE Resources DROP COLUMN `"
                                       .$OldDBFieldName."End`");
                            $DB->Query("ALTER TABLE Resources DROP COLUMN `"
                                       .$OldDBFieldName."Precision`");
                        }
                        else
                        {
                            exit("<br>ERROR:  Attempt to convert metadata field to time stamp from type other than date<br>\n");
                        }
                    }
                    else
                    {
                        # change name of field
                        $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                                   .$OldDBFieldName."` `"
                                   .$NewDBFieldName."` DATETIME "
                                   .($this->DBFields["Optional"] ? "" : "NOT NULL"));
                    }
                    break;

                case MDFTYPE_TREE:
                    break;

                case MDFTYPE_CONTROLLEDNAME:
                    if ($OldFieldType == MDFTYPE_OPTION)
                    {
                        $DB->Query("ALTER TABLE UserSearch DROP COLUMN `".
                            $OldDBFieldName."`");
                    }
                    break;

                case MDFTYPE_OPTION:
                    if ($OldFieldType == MDFTYPE_CONTROLLEDNAME)
                    {
                        if (!$DB->FieldExists("UserSearch", $NewDBFieldName))
                        {
                            $DB->Query("ALTER TABLE UserSearch ADD COLUMN `".
                                $NewDBFieldName."` TEXT ".
                                ($this->DBFields["Optional"] ? "" : "NOT NULL"));
                        }
                    }
                    else
                    {
                        # alter field declaration in saved user searches table
                        $DB->Query("ALTER TABLE UserSearch CHANGE COLUMN `"
                           .$OldDBFieldName."` `"
                           .$NewDBFieldName."` TEXT "
                           .($this->DBFields["Optional"] ? "" : "NOT NULL"));
                        break;
                    }
            }

            # if qualifier DB field exists
            if ($DB->FieldExists("Resources", $OldDBFieldName."Qualifier"))
            {
                # rename qualifier DB field
                $DB->Query("ALTER TABLE Resources CHANGE COLUMN `"
                           .$OldDBFieldName."Qualifier` `"
                           .$NewDBFieldName."Qualifier` INT ");
            }
        }
    }

    # convenience function to supply parameters to Database->UpdateValue()
    function UpdateValue($FieldName, $NewValue)
    {
        return $this->DB->UpdateValue("MetadataFields", $FieldName, $NewValue,
                               "FieldId = ".$this->DBFields["FieldId"],
                               $this->DBFields);
    }

    # normalize field name for use as database field name
    function NormalizeFieldNameForDB($Name)
    {
        return preg_replace("/[^a-z0-9]/i", "", $Name);
    }
    
    # add any needed database fields and/or entries
    function AddDatabaseFields()
    {
        # grab values for common use
        $DB =& $this->DB;
        $FieldName = $this->Name();
        $DBFieldName = $this->DBFieldName();
        $Optional = $this->Optional();
        $DefaultValue = $this->DefaultValue();
        
        # set up field(s) based on field type
        switch ($this->Type())
        {
            case MDFTYPE_TEXT:
            case MDFTYPE_PARAGRAPH:
                # add field to resources table (if not already present)
                if (!$DB->FieldExists("Resources", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `".$DBFieldName
                               ."` TEXT ".($Optional ? "" : "NOT NULL"));
                }

                # if default value supplied
                if ($DefaultValue != NULL)
                {
                    # set all existing records to default value
                    $DB->Query("UPDATE Resources SET `"
                               .$DBFieldName."` = '".addslashes($DefaultValue)."'");
                }
                break;

            case MDFTYPE_NUMBER:
                # add field to resources table (if not already present)
                if (!$DB->FieldExists("Resources", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `".$DBFieldName
                               ."` INT ".($Optional ? "" : "NOT NULL"));
                }

                # if default value supplied
                if ($DefaultValue != NULL)
                {
                    # set all existing records to default value
                    $DB->Query("UPDATE Resources SET `"
                               .$DBFieldName."` = '".addslashes($DefaultValue)."'");
                }
                break;

            case MDFTYPE_FLAG:
                # add field to resources table (if not already present)
                if (!$DB->FieldExists("Resources", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `".$DBFieldName
                               ."` INT ".($Optional ? "" : "NOT NULL"));
                }

                # add field to saved user searches table (if not already present)
                if (!$DB->FieldExists("UserSearch", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE UserSearch ADD COLUMN `".$DBFieldName
                               ."` INT ".($Optional ? "" : "NOT NULL"));
                }

                # if default value supplied
                if ($DefaultValue != NULL)
                {
                    # set all existing records to default value
                    $DB->Query("UPDATE Resources SET `"
                               .$DBFieldName."` = '".addslashes($DefaultValue)."'");
                }
                break;

            case MDFTYPE_USER:
                # add field to resources table (if not already present)
                if (!$DB->FieldExists("Resources", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `".$DBFieldName
                               ."` INT ".($Optional ? "" : "NOT NULL"));
                }
                break;

            case MDFTYPE_FILE:
                # add fields to resources table (if not already present)
                if (!$DB->FieldExists("Resources", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `"
                               .$DBFieldName."` TEXT");
                }
                break;

            case MDFTYPE_IMAGE:
                # add fields to resources table (if not already present)
                if (!$DB->FieldExists("Resources", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `"
                               .$DBFieldName."` INT");
                }
                break;

            case MDFTYPE_DATE:
                # add fields to resources table (if not already present)
                if (!$DB->FieldExists("Resources", $DBFieldName."Begin"))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `".$DBFieldName."Begin`"
                               ." DATE ".($Optional ? "" : "NOT NULL"));
                }
                if (!$DB->FieldExists("Resources", $DBFieldName."End"))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `".$DBFieldName."End`"
                               ." DATE ".($Optional ? "" : "NOT NULL"));
                }
                if (!$DB->FieldExists("Resources", $DBFieldName."Precision"))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `".$DBFieldName."Precision`"
                               ." INT ".($Optional ? "" : "NOT NULL"));
                }
                break;

            case MDFTYPE_TIMESTAMP:
                # add fields to resources table (if not already present)
                if (!$DB->FieldExists("Resources", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE Resources ADD COLUMN `".$DBFieldName
                               ."` DATETIME ".($Optional ? "" : "NOT NULL"));
                }
                break;

            case MDFTYPE_TREE:
                break;

            case MDFTYPE_CONTROLLEDNAME:
                break;

            case MDFTYPE_OPTION:
                # add field to saved user searches table (if not already present)
                if (!$DB->FieldExists("UserSearch", $DBFieldName))
                {
                    $DB->Query("ALTER TABLE UserSearch ADD COLUMN `".$DBFieldName
                               ."` TEXT ".($Optional ? "" : "NOT NULL"));
                }
                break;

            default:
                exit("<br>ERROR:  Attempt to add database fields for illegal metadata field type<br>\n");
                break;
        }
    }

    # get/set field order positions
    function OrderPosition($OrderType, $NewValue = DB_NOVALUE)
    {  
        switch ($OrderType)
        {
            case MDFORDER_DISPLAY:
                return $this->UpdateValue("DisplayOrderPosition", $NewValue);  
                break;

            case MDFORDER_EDITING:
                return $this->UpdateValue("EditingOrderPosition", $NewValue);  
                break;

            default:
                exit("invalid order type passed to MetadataField::OrderPosition");
                break;
        }
    }
}

# field type DB/PHP enum translations
$FieldTypeDBEnums = array(
        MDFTYPE_TEXT             => "Text",
        MDFTYPE_PARAGRAPH        => "Paragraph",
        MDFTYPE_NUMBER           => "Number",
        MDFTYPE_DATE             => "Date",
        MDFTYPE_TIMESTAMP        => "TimeStamp",
        MDFTYPE_FLAG             => "Flag",
        MDFTYPE_TREE             => "Tree",
        MDFTYPE_CONTROLLEDNAME   => "ControlledName",
        MDFTYPE_OPTION           => "Option",
        MDFTYPE_USER             => "User",
        MDFTYPE_IMAGE            => "Still Image",
        MDFTYPE_FILE             => "File",
        );
$FieldTypeDBAllowedEnums = array(
        MDFTYPE_TEXT             => "Text",
        MDFTYPE_PARAGRAPH        => "Paragraph",
        MDFTYPE_NUMBER           => "Number",
        MDFTYPE_DATE             => "Date",
        MDFTYPE_FLAG             => "Flag",
        MDFTYPE_TREE             => "Tree",
        MDFTYPE_CONTROLLEDNAME   => "ControlledName",
        MDFTYPE_OPTION           => "Option",
        MDFTYPE_IMAGE            => "Still Image",
        MDFTYPE_FILE             => "File",
        );
$FieldTypePHPEnums = array(
        "Text"                   => MDFTYPE_TEXT,
        "Paragraph"              => MDFTYPE_PARAGRAPH,
        "Number"                 => MDFTYPE_NUMBER,
        "Date"                   => MDFTYPE_DATE,
        "TimeStamp"              => MDFTYPE_TIMESTAMP,
        "Flag"                   => MDFTYPE_FLAG,
        "Tree"                   => MDFTYPE_TREE,
        "ControlledName"         => MDFTYPE_CONTROLLEDNAME,
        "Option"                 => MDFTYPE_OPTION,
        "User"                   => MDFTYPE_USER,
        "Still Image"            => MDFTYPE_IMAGE,
        "File"                   => MDFTYPE_FILE,
        );
        

?>
