<?PHP

/**
* Base class for all plugins.
*/
abstract class Plugin {

    # ----- PUBLIC INTERFACE -------------------------------------------------

    /**
    * Set the plugin attributes.  At minimum this method MUST set $this->Name
    * and $this->Version.  This is called when the plugin is initially loaded.
    */
    abstract function Register();

    /**
    * Initialize the plugin.  This is called after all plugins have been loaded
    * but before any methods for this plugin have been called.
    * @return NULL if initialization was successful, otherwise a string containing
    *       an error message indicating why initialization failed.
    */
    function Initialize()
    {
        return NULL;
    }

    /**
    * Hook methods to be called when specific events occur.
    * For events declared by other plugins the name string should start with
    * the plugin base (class) name followed by "::" and then the event name.
    * @return Array of method names to hook indexed by the event constants
    *       or names to hook them to.
    */
    function HookEvents()
    {
        return array();
    }

    /**
    * Declare events triggered by this plugin.
    * @return Array with event names for the index and event types for the values.
    */
    function DeclareEvents()
    {
        return array();
    }

    /**
    * Perform any work needed when the plugin is first installed (for example,
    * creating database tables).
    * @return NULL if installation succeeded, otherwise a string containing
    *       an error message indicating why installation failed.
    */
    function Install()
    {
        return NULL;
    }

    /**
    * Perform any work needed when the plugin is upgraded to a new version
    * (for example, adding fields to database tables).
    * @param PreviousVersion The version number of this plugin that was
    *       previously installed.
    * @return NULL if upgrade succeeded, otherwise a string containing
    *       an error message indicating why upgrade failed.
    */
    function Upgrade($PreviousVersion)
    {
        return NULL;
    }

    /**
    * Perform any work needed when the plugin is uninstalled.
    * @param RemoveData If TRUE, any data created by the plugin should
    *       be removed.
    * @return NULL if uninstall succeeded, otherwise a string containing
    *       an error message indicating why uninstall failed.
    */
    function Uninstall($RemoveData)
    {
        return NULL;
    }

    /**
    * Retrieve plugin information.
    * @return Array of attribute values indexed by attribute names. 
    */
    function GetAttributes()
    {
        return array(
                "Name" => $this->Name,
                "Version" => $this->Version,
                "Description" => $this->Description,
                "Author" => $this->Author,
                "Url" => $this->Url,
                "Email" => $this->Email,
                "EnabledByDefault" => $this->EnabledByDefault,
                "Requires" => $this->Requires,
                );
    }


    # ----- PRIVATE INTERFACE ------------------------------------------------

    /** Proper (human-readable) name of plugin. */
    protected $Name = NULL;
    /** Version number of plugin in the format X.X.X (for example: 1.2.12). */
    protected $Version = NULL;
    /** Text description of the plugin. */
    protected $Description = NULL;
    /** Name of the plugin's author. */
    protected $Author = NULL;
    /** Web address for more information about the plugin. */
    protected $Url = NULL;
    /** Contact email for the plugin's author. */
    protected $Email = NULL;
    /** Whether the plugin should be enabled by default when installed */
    protected $EnabledByDefault = FALSE;

    /**
    * Array with plugin base (class) names for the index and minimum version
    * numbers for the values.
    */
    protected $Requires = array();
}

?>
