<?PHP
  /**
     The DrupalSync plugin provides login integration between a CWIS
     and a Drupal site.  There is a companion cwis_user plugin which
     must be installed on the Drupal site for any of this to work.
     The Drupal side of the plugin will need to be configured with the
     database information for the CWIS installation.  The CWIS side
     needs no additional configuration once the plugin is installed.
     
     User creation, deletion, modifiction, login, and logout are
     mirrored from one site to the other.
     
     CWIS has more stringent username requirements than Drupal does,
     so usernames created on Drupal are normalzed to meet CWIS'
     standards before creating a companion CWIS user.
   */

require_once("sync_common.php");

final class DrupalSync extends Plugin
{
    function Register()
    {
        $this->Name = "Drupal/CWIS Login Synchronization";
        $this->Version = "1.0.0";
        $this->Description = "Enables users to register or log in"
                ." seamlessly on Drupal or CWIS and be registered"
                ." or logged in on the other package as well.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
            "CWISCore" => "2.0.0"
            );
    }

    function Install()
    {
        $DB = new Database();
        $DB->Query("CREATE TABLE DrupalToCwisUserSync ( "
                   ."UserName TEXT, "
                   ."Command TEXT,"
                   ."Password TEXT, "
                   ."Email TEXT, "
                   ."Token INT );");
        $DB->Query("CREATE TABLE CwisToDrupalUserSync ( "
                   ."UserName TEXT, "
                   ."Command TEXT,"
                   ."Password TEXT, "
                   ."Email TEXT, "
                   ."Token INT );");
        $DB->Query("CREATE TABLE UserNameMap ( "
                   ."CwisName TEXT, "
                   ."DrupalName TEXT)");
    }

    function HookEvents()
    {
        return array(
            "EVENT_PHP_PAGE_LOAD"         => "CheckLoginCookie",
            "EVENT_USER_ADDED"            => "DrupalMirrorUserAdd",
            "EVENT_USER_DELETED"          => "DrupalMirrorUserDel",
            "EVENT_USER_PASSWORD_CHANGED" => "DrupalMirrorUserUpdate",
            "EVENT_USER_LOGIN"            => "DrupalMirrorUserLogin",
            "EVENT_USER_LOGOUT"           => "DrupalMirrorUserLogout"
            );
    }

    function Initialize()
    {
        $this->DB = new Database();
        return NULL;
    }

    function GetUserData($UserId)
    {
        $this->DB->Query("SELECT UserName, EMail FROM APUsers WHERE "
                   ."UserId = ".intval($UserId));
        return $this->DB->FetchRow();
    }

    function CheckLoginCookie($PageName)
    {
        global $G_User;

        $Helper = new SyncHelper($this->DB);

        $UserFactory = new UserFactory($this->DB);

        // Handle pending user creation requested by Drupal
        $UsersToCreate = $Helper->UsersToCreate();
        foreach ($UsersToCreate as $UserData)
        {
            $TmpUser = new SPTUser($UserData["UserName"]);
            if ($TmpUser->Status() == U_NOSUCHUSER)
            {         
                // If we've gotten this far, then the user has already
                // passed Drupal's password strenght and email format
                // requirements. We don't want to fail to mirrow a
                // user that's valid on the Drupal side because they
                // have different PW/Email format requirements than we
                // do. Therefore, ignore errors stemming from that.
                $TmpUser = $UserFactory->CreateNewUser(
                    $UserData["UserName"],
                    $UserData["Password"], $UserData["Password"],
                    $UserData["Email"], $UserData["Email"],
                    array(U_ILLEGALPASSWORD, U_ILLEGALPASSWORDAGAIN,
                          U_ILLEGALEMAIL, U_ILLEGALEMAILAGAIN));
                $TmpUser->GivePriv(PRIV_POSTTOFORUMS);
                $TmpUser->GivePriv(PRIV_POSTCOMMENTS);
            }

            if ($TmpUser->Status() == U_OKAY)
            {
                $Helper->UserCreated($UserData["UserName"]);
            }
        }

        // Handle pending user deletions requested by Drupal
        $UsersToDelete = $Helper->UsersToDelete();
        foreach ($UsersToDelete as $UserData)
        {
            $TmpUser = new SPTUser($UserData["UserName"]);
            $TmpUser->Delete();
            $Helper->UserDeleted($UserData["UserName"]);
        }

        // Handle pending user update requests by Drupal
        $UsersToUpdate = $Helper->UsersToUpdate();
        foreach ($UsersToUpdate as $UserData)
        {
            $TmpUser = new SPTUser($UserData["UserName"]);
            if (is_object($TmpUser) && $TmpUser->Status() == U_OKAY )
            {
                if (strlen($UserData["Password"]))
                {
                    $TmpUser->SetPassword($UserData["Password"]);
                }

                if (strlen($UserData["Email"]))
                {
                    $TmpUser->Set("EMail",$UserData["Email"]);
                }

                $Helper->UserUpdated($UserData["UserName"]);
            }
        }

        // Check for requests dealing with this user
        $Cmd = $Helper->CheckForCommand( $G_User->Name() );
        if ($Cmd["Command"]==="Login")
        {
            if ($G_User->Login($Cmd["UserName"],$Cmd["Password"])==U_OKAY)
            {
                $Helper->UserLoggedIn($Cmd["UserName"]);
            }
        }
        else if ($Cmd["Command"]==="Logout")
        {
            $G_User->Logout();
            $Helper->UserLoggedOut($Cmd["UserName"]);
        }
    }

    function DrupalMirrorUserAdd($UserId, $Password)
    {
        $Helper = new SyncHelper($this->DB);
        $Res = $this->GetUserData($UserId);
        $Helper->SignalCommand(
            "Create", $Res["UserName"], $Password, $Res["EMail"]);
    }

    function DrupalMirrorUserDel($UserId)
    {
        $Helper = new SyncHelper($this->DB);
        $Res = $this->GetUserData($UserId);
        $Helper->SignalCommand("Delete", $Res["UserName"]);
    }

    function DrupalMirrorUserUpdate($UserId,$OldPass,$NewPass)
    {
        $Helper = new SyncHelper($this->DB);
        $Res = $this->GetUserData($UserId);
        $Helper->SignalCommand(
            "Update",$Res["UserName"], $NewPass, $Res["EMail"]);
    }

    function DrupalMirrorUserLogin($UserId, $Password)
    {
        $Helper = new SyncHelper($this->DB);
        $Res = $this->GetUserData($UserId);
        $Helper->SignalCommand(
            "Login", $Res["UserName"], $Password, $Res["EMail"]);
    }

    function DrupalMirrorUserLogout($UserId)
    {
        $Helper = new SyncHelper($this->DB);
        $Res = $this->GetUserData($UserId);
        $Helper->SignalCommand("Logout", $Res["UserName"]);
    }

    private $DB;
};
?>
