<?PHP
#
#   FILE:  SPT--CommonSearch.php
#
#   FUNCTIONS PROVIDED:
#
#       GetMinForColumn()
#           - get the minimum value for the specified column and table
#
#       GetMaxForColumn()
#           - get the maximum value for the specified column and table
#
#       DisplayControlledNameListForName()
#           - display a ControlledName list as a multi option menu
#
#       GetSelectedListValue()
#           - get the selected values from a ControlledName multi option list
#
#       DisplayControlledNamesAsText()
#           - display list of ControlledNames as text boxes with labels
#
#       DisplayClassificationTypeList()
#           - display list of Classification Tree types
#
#       GetControlledNameIsList($ControlledNameTypeId)
#           - return a ControlledNameTypeIsList for a ControlledNameTypeId
#
#       GetControlledNameType($ControlledNameTypeId)
#           - look up and return ControlledNameTypeName based on 
#             ControlledNameTypeId
#
#       GetControlledNameTypeId()
#           - return a ControlledNameTypeId for a ControlledNameTypeName
#
#       GetControlledName()
#           - return a ControlledName for a ControlledNameId
#
#       CreateFrequencyOptionList()
#           - create frequency option list for Prefrences
#
#       SavedSearchesExist()
#           - returns true if saved searches exist for current user
#
#       CreateOptionListForAdvancedSearch()
#           - create an option list for the Advanced Search fields
#
#       UnquoteString()
#           - remove quotes from the beginning and end of a string
#
#       ConvertWildCardString()
#           - convert a wildcard string for metadata tool search;
#
#       BuildWhereStatement()
#           - build a where statement for the metadata tool search
#
#       BuildPhraseSearch()
#           - build a phrase query (quoted) for metadata tool search
#
#       PrintSearchHeader()
#           - print out header html as a result of a search
#
#       PrintArchiveEntry()
#           - print an individual resource record
#
#   FUNCTIONS EXPECTED:
#       None
#
#   Part of the Scout Portal Toolkit
#   Copyright 2003 Internet Scout Project
#   http://scout.cs.wisc.edu
# 

# get the minimum value for the specified column and table
function GetMinForColumn($ColumnName, $Table)
{
    $db =& new SPTDatabase();

    $query = "Select min($ColumnName) as Min from $Table";
    $db->Query($query);
    $Min = $db->FetchNextRowArray();

    return ($Min["Min"]);
}

# get the maximum value for the specified column and table
function GetMaxForColumn($ColumnName, $Table)
{
    $db =& new SPTDatabase();

    $query = "Select max($ColumnName) as Max from $Table";
    $db->Query($query);
    $Max = $db->FetchNextRowArray();

    return ($Max["Max"]);
}

function DisplayUserAgents()
{
    $db =& new SPTDatabase();
    global $User;
    global $SavedSearchCount;

    $Items = array();

    $UserId = $User->Get("UserId");

    $query = "Select SearchName from UserSearch where UserId=$UserId ".
                    "ORDER BY SearchName";
    $db->Query($query);

    $SavedSearchCount = 0;
    while($Entry = $db->FetchNextRowArray())
    {
        print "<tr><td align=\"right\">$Entry[SearchName]:</td>\n";
        print "<td>";
        CreateFrequencyOptionList($Entry[SearchName], $SavedSearchCount);
        print "</td></tr>";
        $SavedSearchCount++;
    }
}

# function to determine if saved searches exist for this user
function SavedSearchesExist()
{
    global $SavedSearchCount;

    return($SavedSearchCount > 0) ? true : false;
}

#  display a ControlledName list as a multi option menu 
function DisplayControlledNameListForName($VarName, $ControlledNameTypeName,
            $FirstEntry, $Selected, $Height, $Multi, $Search) 
{
    $db =& new SPTDatabase();

    $query = "Select * from ControlledNameTypes where ".
        "ControlledNameTypeName=\"".$ControlledNameTypeName."\"";
    $db->Query($query);

    # only retrieve controlled names if type found
    if($db->NumRowsSelected())
    {
        $Entry = $db->FetchNextRowArray();
        $query = "Select * from ControlledNames where ControlledNameTypeId=".
            $Entry["ControlledNameTypeId"]." order by ControlledName";
        $db->Query($query);
    }
    $Items = array();
    reset($Items);
    if(!empty($FirstEntry) && !$Multi)
        $Items[$FirstEntry] = $FirstEntry;

    # load up array with values 
    if($db->NumRowsSelected())
    {
        while($Entry = $db->FetchNextRowArray())
        {
            if($Search)
                $Items[$Entry["ControlledName"]] = $Entry["ControlledName"];
            else
                $Items[$Entry["ControlledNameId"]] = $Entry["ControlledName"];
        }
    }
    # convert varname into array varname if not searching
    if($Multi)
        $VarName = $VarName."[]";

    PrintOptionList($VarName, $Items, $Selected, "", $Height, 1, $Multi);
}

#  get the selected values from a ControlledName multi option list
function GetSelectedListValue($ResourceId, $ListTypeName)
{
    $db =& new SPTDatabase();

    $Selected = array();
    if(!empty($ListTypeName))
    {
        $query = "Select * from ControlledNameTypes where ".
            "ControlledNameTypeName=\"".$ListTypeName."\"";
        $db->Query($query);
        $Entry = $db->FetchNextRowArray();

        $query = "Select min(ControlledNameId), max(ControlledNameId) from ".
            "ControlledNames where ControlledNameTypeId = ".
            $Entry["ControlledNameTypeId"];
        $db->Query($query);

        # only do next query if ControlledName has entries
        if($db->NumRowsSelected())
        {
            $Entry = $db->FetchNextRowArray();
            $Min = $Entry[0];
            $Max = $Entry[1];

            # only do next query if valid min and max values found
            if(isset($Min) && isset($Max) && isset($ResourceId))
            {
                $query = "Select * from ResourceNameInts where ResourceId=".
                    $ResourceId." and ControlledNameId >= ".$Min.
                    " and ControlledNameId <= ".$Max;

                $db->Query($query);
                while($Entry = $db->FetchNextRowArray())
                {
                    $Selected[] = $Entry["ControlledNameId"];    
                }
            }
        }    
    }
    return $Selected;
}

#  display ControlledNames as a list of labels and text boxes
function DisplayControlledNamesAsText($VarName, $ControlledNameIsList)
{
    $db =& new SPTDatabase();

    $query = "Select * from ControlledNameTypes where ControlledNameIsList=".
        $ControlledNameIsList;
    $db->Query($query);

    $j = 1;
    while($Entry = $db->FetchNextRowArray())
    {
        echo "<tr class=\"lightBody\">\n";
        echo "<td>".$Entry[ControlledNameTypeName].":</td>\n";
        echo "<td><input type=\"text\" size=\"52\" name=\"".$VarName.$j."\" value=\"\"></td>\n";
        echo "</tr>\n";
        $j++;
    }
}

# functon to return a ControlledNameTypeName for a ControlledNameTypeId
function GetControlledNameType($ControlledNameTypeId)
{
    $db =& new SPTDatabase();

    $query = "Select * from ControlledNameTypes where ControlledNameTypeId =".
            $ControlledNameTypeId;
    $db->Query($query);

    $Entry = $db->FetchNextRowArray();

    return $Entry[ControlledNameTypeName];
}

# functon to return a ControlledNameTypeIsList for a ControlledNameTypeId
function GetControlledNameIsList($ControlledNameTypeId)
{
    $db =& new SPTDatabase();

    $query = "Select * from ControlledNameTypes where ControlledNameTypeId =".
            $ControlledNameTypeId;
    $db->Query($query);

    $Entry = $db->FetchNextRowArray();

    return $Entry[ControlledNameIsList];
}

#  return a ControlledNameTypeId for a ControlledNameTypeName
function GetControlledNameTypeId($ControlledNameTypeName)
{
    $db =& new SPTDatabase();

    $query = "Select * from ControlledNameTypes where ControlledNameTypeName =".
           "'".$ControlledNameTypeName."'";
    $db->Query($query);

    $Entry = $db->FetchNextRowArray();

    return $Entry[ControlledNameTypeId];
}

#  return a ClassificationTypeId for a ClassificationTypeName
function GetClassificationTypeId($ClassificationTypeName)
{
    $db =& new SPTDatabase();

    $query = "Select * from ClassificationTypes ".
            "where ClassificationTypeName =".
            "'".$ClassificationTypeName."'";
    $db->Query($query);

    $Entry = $db->FetchNextRowArray();

    return $Entry[ClassificationTypeId];
}

#  return a ClassificationTypeName for a ClassificationTypeId
function GetClassificationType($ClassificationTypeId)
{
    $db =& new SPTDatabase();

    $query = "Select * from ClassificationTypes ".
            "where ClassificationTypeId =".
            $ClassificationTypeId;
    $db->Query($query);

    $Entry = $db->FetchNextRowArray();

    return $Entry[ClassificationTypeName];
}

#  display a Classification list of values as an option menu
function DisplayClassificationTypeList($Selected, $SubmitOnChange = FALSE)
{
    $DB =& new SPTDatabase();

    $Query = "Select * from ClassificationTypes".
        " ORDER BY ClassificationTypeName";
    $DB->Query($Query);

    while($Entry = $DB->FetchNextRowArray())
    {
        $Items[$Entry["ClassificationTypeId"]] = 
            $Entry["ClassificationTypeName"];
    }

    if($SubmitOnChange)
        $FieldName = "Submit";
    else
        $FieldName = "F_ClassificationTypeId";
    PrintOptionList($FieldName, $Items, $Selected, $SubmitOnChange,
        1, NULL, false);
}

#  return a ControlledName for a ControlledNameId
function GetControlledName($ControlledNameId)
{
    $db =& new SPTDatabase();

    $query = "Select * from ControlledNames where ControlledNameId =".
           "'".$ControlledNameId."'";
    $db->Query($query);

    $Entry = $db->FetchNextRowArray();

    return $Entry[ControlledName];
}

# create frequency option list for Preferences
function CreateFrequencyOptionList($SearchName, $Suffix)
{
    $db =& new SPTDatabase();
    global $User;

    $UserId = $User->Get("UserId");

    if(isset($SearchName))
    {
        $query = "Select Frequency from UserSearch where UserId=$UserId ".
                "AND SearchName = \"$SearchName\"";
        $db->Query($query);
        $Entry = $db->FetchNextRowArray();
    }
    echo "<select name=\"F_Frequency_$Suffix\" size=1>\n";

    echo "<option";
    if(!isset($Entry[Frequency]) || $Entry[Frequency] == "Never")
        echo " selected";
    echo ">Never\n";

    echo "<option";
    if($Entry[Frequency] == "Once a day")
        echo " selected";
    echo ">Once a day\n";
    echo "<option";
    if($Entry[Frequency] == "Once a week")
        echo " selected";
    echo ">Once a week\n";
    echo "<option";
    if($Entry[Frequency] == "Bi-weekly")
        echo " selected";
    echo ">Bi-weekly\n";
    echo "<option";
    if($Entry[Frequency] == "Once a month")
        echo " selected";
    echo ">Once a month\n";
    echo "<option";
    if($Entry[Frequency] == "Quarterly")
        echo " selected";
    echo ">Quarterly\n";
    if ($User->HasPriv(PRIV_RESOURCEADMIN))
    {
        echo "<option";
        if($Entry[Frequency] == "Hourly")
            echo " selected";
        echo ">Hourly\n";
    }

    echo "</select>\n";
}

# create an option list for the Advanced Search fields
function CreateOptionListForAdvancedSearch($VarName, $Selected, $VarValue) 
{
    $Values = array();
    
    # Get the schema
    $Schema = & new MetadataSchema();

    # Get the fields for the schema
    $Fields = $Schema->GetFields(MDFTYPE_TEXT | MDFTYPE_PARAGRAPH);
    foreach($Fields as $Field)
        if($Field->Enabled() && $Field->IncludeInAdvancedSearch() &&
            ($Field->Viewable() || InMetadataTool()))
            $Values[$Field->Name()] = $Field->Name();
        
    # hard code Keyword as a search type
    $Values["Keyword"] = "Keyword";
    
    $Fields = $Schema->GetFields(MDFTYPE_CONTROLLEDNAME);
    $i = 0;
    foreach($Fields as $Field)
    {
        if($Field->Enabled() && $Field->IncludeInAdvancedSearch() &&
            ($Field->Viewable() || InMetadataTool()))
        {
            # use first controlled name as default if Selected is NULL  
            if($i == 0 && empty($Selected))
                $Selected = $Field->DBFieldName();
            $Values[$Field->Name()] = $Field->Name();
        }
        $i++;
    }

    $Fields = $Schema->GetFields(MDFTYPE_TREE);
    foreach($Fields as $Field)
        if($Field->Enabled() && $Field->IncludeInAdvancedSearch() &&
            ($Field->Viewable() || InMetadataTool()))
            $Values[$Field->Name()] = $Field->Name();

    $Fields = $Schema->GetFields(MDFTYPE_NUMBER);
    foreach($Fields as $Field)
        if($Field->Enabled() && $Field->IncludeInAdvancedSearch() &&
            ($Field->Viewable() || InMetadataTool()))
            $Values[$Field->Name()] = $Field->Name();
            
    $Fields = $Schema->GetFields(MDFTYPE_IMAGE);
    foreach($Fields as $Field)
        if($Field->Enabled() && $Field->IncludeInAdvancedSearch() &&
            ($Field->Viewable() || InMetadataTool()))
            $Values[$Field->Name()] = $Field->Name();

    $Fields = $Schema->GetFields(MDFTYPE_DATE | MDFTYPE_TIMESTAMP);
    foreach($Fields as $Field)
    {
        if($Field->Enabled() && $Field->IncludeInAdvancedSearch() &&
            ($Field->Viewable() || InMetadataTool()))
                $Values[$Field->Name()] = $Field->Name();
    }                
            
    echo "<select name=\"".$VarName."\" size=1>\n";
    $i=0;
    foreach($Values as $Value => $Name)
    {
        echo "<option value=\"".$Value."\"";
        if(isset($VarValue) && $VarValue == $Value)
            echo " selected";
        else if($Name == $Selected && !isset($VarValue))
            echo " selected";
        echo ">".$Name."</option>\n";
        $i++;
    }
    echo "</select>\n";
}


#  remove quotes from the beginning and end of a string
function UnquoteString($String)
{
    $String[0] = " ";
    $String[strlen($String) - 1] = " ";
    $String = trim($String);
    
    # escape embedded single quotes
    $String = addslashes($String);
    $String = strtolower($String);
    return $String;
}

#  convert a wildcard string for metadata tool search
function ConvertWildCardString($SearchString)
{
    # check for wild card matching
    if(strstr($SearchString, "*") || strstr($SearchString, "?"))
    {
        $SearchString = str_replace("*", "%", $SearchString);
        $SearchString = str_replace("?", "_", $SearchString);
    }
    # remove "^" at beginning of line
    if(strlen(strpos($SearchString, "^")))
        $SearchString = str_replace("^", "", $SearchString);
    # otherwise prepend "%" wildcard to match everything prior to string
    else
        $SearchString = "%".$SearchString;
    # remove "$" at end of line
    if(strpos($SearchString, "$") == (strlen($SearchString)-1))
        $SearchString = str_replace("$", "", $SearchString);
    # otherise append "%" wildcard to match everything after the string
    else
        $SearchString = $SearchString."%";

    # return properly formatted wildcard string
    return $SearchString;
}

#  build a where statement for the metadata tool search
function BuildWhereStatement($SearchField, $SearchString)
{
    # escape quotes
    $SearchString = addslashes($SearchString);

    # check for keyword searching
    if($SearchField == "Title, Description, Url")
    {
        # check for wild card matching
        if(strstr($SearchString, "*") || strstr($SearchString, "?") ||
            strlen($SearchString) < 4)
        {
            if(strlen($SearchString) < 4)
                $SearchString = "%".$SearchString."%";
            else
                $SearchString = ConvertWildCardString($SearchString);
            $Where = " (Title LIKE '$SearchString' OR ".
                      "Description LIKE '$SearchString' OR ".
                      "Url LIKE '$SearchString') ";
        }
        # normal keyword fulltext search
        else
            $Where = " (MATCH ($SearchField) AGAINST "."('$SearchString')) ";
        
    }
    # process everything else
    else
    {
        # check for wildcard matching
        if(strstr($SearchString, "*") || strstr($SearchString, "?") ||
            strlen($SearchString) < 4)
        {
            if(strlen($SearchString) < 4)
                $SearchString = "%".$SearchString."%";
            else
                $SearchString = ConvertWildCardString($SearchString);
            $Where = " ($SearchField) LIKE '$SearchString' ";
        }
        else
            $Where = " MATCH ($SearchField) AGAINST "."('$SearchString') ";
    }
    return $Where;
}

#  build a phrase query (quoted) for metadata tool search
function BuildPhraseSearch($SearchField, $SearchString)
{
    # remove leading and trailing whitespace
    $SearchString = trim($SearchString);

    # strip off backslashes
    $SearchString = stripslashes($SearchString);

    # check for ", ', or ` at beginning and end of string
    $FirstChar = substr($SearchString, 0, 1);
    $LastChar = substr($SearchString, strlen($SearchString) - 1, 1);

    if(($FirstChar == "\"" || $FirstChar == "'" || $FirstChar == "`") && 
       ($LastChar == "\"" || $LastChar == "'" || $LastChar == "`"))
#    if($FirstChar == "\"" || $FirstChar == "'" || $FirstChar == "`")
    {
        # remove backslashes and quotes
        $SearchString = UnquoteString($SearchString);

        # keyword search
        if($SearchField == "Title, Description, Url")
        {
            $WhereClause = " AND (LOCATE('".$SearchString."', ".
                "LOWER(Title)) > 0".
                " OR LOCATE('".$SearchString."', ".
                "LOWER(C.ControlledName)) > 0".        
                " OR LOCATE('".$SearchString."', ".
                "LOWER(Description)) > 0".
                " OR LOCATE('".$SearchString."', ".
                "LOWER(Url)) > 0) ";             
        }
        # all other fields
        else
        {
            $WhereClause = " AND LOCATE('".$SearchString."', LOWER(".
                $SearchField.")) > 0 ";

        }
    }
    else
        $WhereClause = "";

    return $WhereClause;
}

?>
