<?PHP

#
#   FILE:  SPT--InstallComplete.php
#
#   FUNCTIONS PROVIDED:
#       (none)
#
#   FUNCTIONS EXPECTED:
#       (none)
#
#   FORM FIELDS EXPECTED:
#       F_DBHost
#           - host name of computer housing database server
#       F_DBUserName
#       F_DBPassword
#           - user name and password needed to connect to database server
#       F_DBName
#           - name of database to use for storing SPT data
#
#   Part of the Scout Portal Toolkit
#   Copyright 2001-2003 Internet Scout Project
#   http://scout.wisc.edu
#

require_once("Axis--Database.php");
require_once("Axis--Image.php");
require_once("Axis--User.php");


# ----- EXPORTED FUNCTIONS ---------------------------------------------------

# SPT <--> CWIS version mapping
# (this must match the similar maps in:
#     SPT--InstallComplete.php,
#     include/SPT--Common.php,
#     SPT--BuildRelease, and
#     sptinstall)
$SPT_SPTCWISVersionMap = array(
        "1.1.4" => "0.9.0",
        "1.1.5" => "0.9.1",
        "1.1.6" => "0.9.2",
        "1.1.7" => "0.9.3",
        "1.1.8" => "0.9.4",
        "1.2.0" => "1.0.0",
        "1.2.1" => "1.0.1",
        "1.2.2" => "1.1.0",
        "1.2.3" => "1.1.1",
        "1.3.0" => "1.3.0",
        "1.3.1" => "1.3.1",
        "1.4.0" => "1.4.0",
        "1.4.1" => "1.4.1",
        );

function PrintErrorMessageList()
{
    global $ErrorMessageList;

    print("<ul>\n");
    foreach ($ErrorMessageList as $Message)
    {
        print("<li>".$Message."</li>\n");
    }
    print("</ul>\n");
}

function InstallOkay()
{
    global $EverythingOkay;

    return $EverythingOkay;
}

function PreviouslyInstalled()
{
    global $AlreadyInstalled;
    return $AlreadyInstalled;
}

function DoingUpgrade()
{
    global $DoingAnUpgrade;
    return $DoingAnUpgrade;
}

function PrintAppVersion()
{
    global $AppVersion;
    global $F_AppName;
    global $SPT_SPTCWISVersionMap;
    
    if ($F_AppName == "CWIS")
    {
        print($SPT_SPTCWISVersionMap[$AppVersion]);
    }
    else
    {
        print($AppVersion);
    }
}

function PrintAdminLogin()
{
    global $F_AdminLogin;
    print($F_AdminLogin);
}

function Debug()
{
    return isset($_POST["F_Debug"]) ? TRUE : FALSE;
}


# ----- LOCAL FUNCTIONS ------------------------------------------------------

# (THIS FUNCTION NOT YET USED)
function CreateDirectoriesAndLinks()
{
    # links needed
    $RequiredLinks = array(
            "SPT--Home.php"                      => "index.php",
            "../include"                         => "MetadataTool/include",
            "../images"                          => "MetadataTool/images",
            "../Axis--Database.php"              => "MetadataTool/Axis--Database.php",
            "../Axis--StandardLibrary.php"       => "MetadataTool/Axis--StandardLibrary.php",
            "../Axis--Session.php"               => "MetadataTool/Axis--Session.php",
            "../Axis--User.php"                  => "MetadataTool/Axis--User.php",
            "../Axis--UserFactory.php"           => "MetadataTool/Axis--UserFactory.php",
            "../Axis--Image.php"                 => "MetadataTool/Axis--Image.php",
            "../include"                         => "SPTUI--Default/MetadataTool/include",
            "../images"                          => "SPTUI--Default/MetadataTool/images",
            "../SPTUI--Default/MetadataTool"     => "MetadataTool/SPTUI--Default",
            "../include"                         => "SPTUI--CleanOrange/MetadataTool/include",
            "../images"                          => "SPTUI--CleanOrange/MetadataTool/images",
            "../SPTUI--CleanOrange/MetadataTool" => "MetadataTool/SPTUI--CleanOrange",
            "../include"                         => "SPTUI--Standards/MetadataTool/include",
            "../images"                          => "SPTUI--Standards/MetadataTool/images",
            "../SPTUI--Standards/MetadataTool"   => "MetadataTool/SPTUI--Standards",
            "../Scout--SearchEngine.php"         => "MetadataTool/Scout--SearchEngine.php",
            "../Scout--RSSClient.php"            => "MetadataTool/Scout--RSSClient.php",
            "../Scout--XMLParser.php"            => "MetadataTool/Scout--XMLParser.php",
            );
    # directories needed
    $RequiredDirectories = array(
            "ImageStorage",
            "ImageStorage/Previews",
            "ImageStorage/Thumbnails",
            "TempStorage",
            );

    # for each required directory
    foreach (@RequiredDirectories as $Dir => $Perm)
    {
    }
}

function CanonicalVersion($Version)
{
    # split into major/minor/rev
    $Pieces = explode(".", $Version);

    # calculate canonical version of version and return to caller
    return ($Pieces[0] * 10000) + ($Pieces[1] * 100) + $Pieces[2];
}

function CheckEnvironment()
{
    if (Debug()) {  print("IC: Checking operating environment...<br>\n");  }
    global $ErrorMessageList;

    # assume that everything will be okay
    $EverythingOkay = TRUE;

    # check whether PHP is configured to allow file uploads
    if (ini_get("file_uploads") != "1")
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Your web server does not have file uploads enabled (via the <i>file_uploads</i> PHP configuration directive), which are required for the software to operate correctly.";
    }

    # report back to caller whether everything was okay (TRUE if okay)
    if (Debug()) {  print("IC: ...operating environment checking completed.<br>\n");  }
    return $EverythingOkay;
}

function CheckValues($HostName, $UserName, $Password, $DBName, 
                     $AdminLogin, $AdminPassword, $AdminEmail, $AppName)
{
    if (Debug()) {  print("IC: Checking input values...<br>\n");  }
    global $ErrorMessageList;

    # assume that everything will be okay
    $EverythingOkay = TRUE;

    # check that DB host name looks plausible
    if (!preg_match("/^[a-zA-Z0-9.\-]+$/", $HostName))
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "The database host name you entered, '${HostName}', does not appear to be valid.";
    }

    # check that DB user name and password look plausible
    if (!preg_match("/^[a-zA-Z0-9_\-#$^]+$/", $UserName))
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "The database user name you entered, '${UserName}', does not appear to be valid.";
    }
    # (no test for valid-looking password, since it can contain almost anything)

    # check that DB name looks plausible
    if (!preg_match("/^[a-zA-Z0-9_\-#$^]+$/", $DBName))
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "The database name you entered, '${DBName}', does not appear to be valid.";
    }

    # check that config file template is readable
    if (! is_readable("include/SPT--Config.php.DIST"))
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Unable to read configuration file template.";
    }

    # check that new config file is writable
    if (! is_writable("include"))
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Unable to write configuration file.";
    }

    # check that DB create tables file is readable
    if (! is_readable("SPT--CreateTables.sql"))
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Unable to read create tables setup file.";
    }

    # check that create default schema file is readable
    $DefaultSchemaFileName = ($AppName == "CWIS") ? "CWIS--CreateDefaultSchema.sql" : "SPT--CreateDefaultSchema.sql";

    if (! is_readable($DefaultSchemaFileName))
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Unable to read default schema setup file.";
    }

    # check administrator user name and password and e-mail address
    if (! User::IsValidUserName($AdminLogin))
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Requested administrator user name is invalid (must contain only letters and/or numbers).";
    }
    if (! User::IsValidPassword($AdminPassword))
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Requested administrator password is invalid (must be at least six characters long).";
    }
    if (strlen(trim($AdminEmail)) < 1)
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "No administrator e-mail address entered.";
    }

    # test database parameters
    $DBHandle = mysql_connect($HostName, $UserName, $Password);
    if ($DBHandle == FALSE)
    {
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Unable to connect to database server on <i>${HostName}</i> with user name <i>${UserName}</i>.";
    }
    else
    {
        mysql_close($DBHandle);
    }

    # report back to caller whether everything was okay (TRUE if okay)
    if (Debug()) {  print("IC: ...input value checking completed.<br>\n");  }
    return $EverythingOkay;
}

function SetConfigFileValues($HostName, $UserName, $Password, $DBName, $AppName)
{
    if (Debug()) {  print("IC: Setting config file values...<br>\n");  }
    if (Debug()) {  print("IC:     HostName = \"$HostName\".<br>\n");  }
    if (Debug()) {  print("IC:     UserName = \"$UserName\".<br>\n");  }
    if (Debug()) {  print("IC:     Password = \"$Password\".<br>\n");  }
    if (Debug()) {  print("IC:       DBName = \"$DBName\".<br>\n");  }
    if (Debug()) {  print("IC:      AppName = \"$AppName\".<br>\n");  }
    global $ErrorMessageList;

    # set up config file substitution settings
    $Patterns = array(
            "/X-DBHOST-X/",
            "/X-DBUSER-X/",
            "/X-DBPASSWORD-X/",
            "/X-DBNAME-X/",
            "/X-DEFAULTUI-X/"
            );
    $Replacements = array(
            $HostName,
            $UserName,
            $Password,
            $DBName,
            (($AppName == "CWIS") ? "SPTUI--CWIS" : "SPTUI--Default")
            );

    # assume that everything will be okay
    $EverythingOkay = TRUE;

    # open config file template
    $InputFile = fopen("include/SPT--Config.php.DIST", "r");

    # open new config file (truncating any existing file content)
    $OutputFileName = "include/SPT--Config.php";
    if (is_file($OutputFileName)) {  chmod($OutputFileName, 0666);  }
    $OutputFile = fopen($OutputFileName, "w");

    # if attempt to open config file failed
    if ($OutputFile == FALSE)
    {
        # add error message
        $ErrorMessageList[] = "Unable to write configuration file.";

        # set flag to indicate that things did not go okay
        $EverythingOkay = FALSE;
    }
    else
    {
        # while lines left in template file
        while (! feof($InputFile))
        {
            # read in line from template file
            $Line = fgets($InputFile, 256);
    
            # perform any needed substitutions
            $Line = preg_replace($Patterns, $Replacements, $Line);
    
            # write line out to new config file
            fwrite($OutputFile, $Line);
        }
    
        # close new config file
        fclose($OutputFile);
    
        # close config file template
        fclose($InputFile);
    
        # remove write permissions on new config file
        chmod($OutputFileName, 0444);
    }

    # report back to caller whether everything was okay (TRUE if okay)
    if (Debug()) {  print("IC: ...config value setting completed.<br>\n");  }
    return $EverythingOkay;
}

function SetUpDatabase($HostName, $UserName, $Password, $DBName)
{
    if (Debug()) {  print("IC: Setting up database...<br>\n");  }

    global $ErrorMessageList;

    # assume that everything will be okay
    $EverythingOkay = TRUE;

    # open connection to database server
    $DBHandle = mysql_connect($HostName, $UserName, $Password);

    # if connection attempt failed
    if ($DBHandle == FALSE)
    {
        # add error message
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Unable to connect to database server on <i>${HostName}</i> with user name <i>${UserName}</i>.";
    }
    else
    {
        # look for database with specified name
        $Result = mysql_list_dbs($DBHandle);
        $FoundDB = FALSE;
        while ($DBInfo = mysql_fetch_object($Result)) 
        {
            if ($DBInfo->Database == $DBName) {  $FoundDB = TRUE;  }
        }

        # if database does not already exist
        if ($FoundDB == FALSE)
        {
            # create database
            mysql_query("CREATE DATABASE ".$DBName, $DBHandle);
        }

        # close connection to database server
        mysql_close($DBHandle);
    }

    # report back to caller whether everything was okay (TRUE if okay)
    if (Debug()) {  print("IC: ...database setup completed.<br>\n");  }
    return $EverythingOkay;
}

function ExecuteSQLCommands(&$DB, $SQLFileName)
{
    if (Debug()) {  print("IC: Executing SQL: $SQLFileName<br>\n");  }
    
    global $ErrorMessageList;

    # assume that everything will be okay
    $EverythingOkay = TRUE;

    # open DB setup file
    $InputFile = fopen($SQLFileName, "r");

    # while lines left in DB setup file and everything is going okay
    $CurrentCommand = "";
    $CurrentDisplay = "";
    while ((! feof($InputFile)) && ($EverythingOkay == TRUE))
    {
        # read in line from DB setup file
        $Line = fgets($InputFile, 4096);

        # trim whitespace from line
        $Line = trim($Line);

        # if line is not a comment
        if ((preg_match("/^#/", $Line) == FALSE) 
                && (preg_match("/^--/", $Line) == FALSE))
        {
            # add line to current command
            $CurrentCommand .= " ".$Line;
            $CurrentDisplay .= $Line."<br>\n";

            # if line ends command
            if (preg_match("/;$/", $Line) == TRUE)
            {
                # execute command
                $CurrentCommand = preg_replace("/;$/", "", $CurrentCommand);
                $Result = $DB->Query($CurrentCommand);

                # if command failed
                if ($Result === FALSE)
                {
                    # add error message
                    $EverythingOkay = FALSE;
                    $ErrorMessageList[] = "Could not successfully execute SQL command:<br>\n"
                            .$CurrentDisplay."<br>(SQL error message: <i>".mysql_error()."</i>)";
                }
                else
                {
                    # clear current command
                    $CurrentCommand = "";
                    $CurrentDisplay = "";
                }
            }
        }
    }

    # close DB setup file
    fclose($InputFile);

    # report back to caller whether everything was okay (TRUE if okay)
    if (Debug()) {  print("IC: ...execution of $SQLFileName completed.<br>\n");  }
    return $EverythingOkay;
}

# function to get list of matching files (used by ConvertImageFiles())
function ListDir($DirPath, $Pattern)
{
    static $ResultArray = array();  

    $Handle = opendir($DirPath);
    while (false !== ($File = readdir($Handle)))
    {
        if ($File == '.' || $File == '..')
            continue;
        if (is_dir($DirPath.$File))
            continue; 
        else if (preg_match($Pattern, $File))
            $ResultArray[]=$DirPath.$File;
    }
    closedir($Handle);
    return $ResultArray;
}

function ConvertImageFiles($HostName, $UserName, $Password, $DBName, $AppName)
{
    if (Debug()) {  print("IC: Converting image files...<br>\n");  }
    
    global $ErrorMessageList;

    $Success = true;
    $DB =& new Database($UserName, $Password, $DBName, $HostName);
    $DBTwo =& new Database($UserName, $Password, $DBName, $HostName);

    if (isset($_SERVER["PATH_TRANSLATED"])
             && strlen($_SERVER["PATH_TRANSLATED"]))
    {
        $ScriptFileName = $_SERVER["PATH_TRANSLATED"];
    }
    elseif (isset($_SERVER["SCRIPT_FILENAME"])
             && strlen($_SERVER["SCRIPT_FILENAME"]))
    {
        $ScriptFileName = $_SERVER["SCRIPT_FILENAME"];
    }   
    elseif (isset($_SERVER["PATH_INFO"])
             && strlen($_SERVER["PATH_INFO"]))
    {   
        $ScriptFileName = $_SERVER["PATH_INFO"];
    }   
    else
    {   
        exit("ERROR:  unable to determine script file name");
    }   

    $InputPath = substr($ScriptFileName, 0, strrpos($ScriptFileName, '/')).
        "/ImageStorage/";

    $ImagePaths = ListDir($InputPath, "/^Img*/");

    # for each metadata field
    $DB->Query("SELECT FieldName FROM MetadataFields WHERE FieldType = 'Still Image'");
    while ($FieldName = $DB->FetchField("FieldName"))
    {
        # create new ImageId field in Resources
        $DBFieldName = preg_replace("/[^a-z0-9]/i", "", $FieldName);
        $DBTwo->Query("ALTER TABLE Resources ADD COLUMN ".$DBFieldName.
                   " INT DEFAULT 0");
    }

    # initialize image ID counter
    $ImageId = 1;

    # for each image file
    foreach ($ImagePaths as $ImagePath)
    {
        $ImageName = basename($ImagePath);
        $PreviewName = str_replace("Img", "Preview", $ImageName);
        $ThumbName = str_replace("Img", "Thumb", $ImageName);

        $ImageExt = array_pop(explode('.', $ImageName));
        $PreviewExt = array_pop(explode('.', $PreviewName));
        $ThumbExt = array_pop(explode('.', $ThumbName));

        $ThumbPath = $InputPath."Thumbnails/".$ThumbName;
        $PreviewPath = $InputPath."Previews/".$PreviewName;

        if (Debug()) {  echo "ImagePath = $ImagePath<br>";  }
        if (Debug()) {  echo "PreviewPath = $PreviewPath<br>";  }
        if (Debug()) {  echo "ThumbPath = $ThumbPath<br>";  }

        # create new file names
        $NewUrl = $InputPath."Img--".sprintf("%08d", $ImageId).
                        ".".$ImageExt;
        $NewPreview = $InputPath."Previews/Preview--".sprintf("%08d", $ImageId).
                        ".".$PreviewExt;
        $NewThumb = $InputPath."Thumbnails/Thumb--".sprintf("%08d", $ImageId).
                        ".".$ThumbExt;      

        if (file_exists($ImagePath))
            $Image = & new Image($ImagePath);
        else
        {
            $ErrorMessageList[] = "Image doesn't exist: $ImagePath<br>";
            $Success = false;
        }

        if (file_exists($PreviewPath))
            $Preview = & new Image($PreviewPath);
        else
        {
            $ErrorMessageList[] = "Preview image doesn't exist: $PreviewPath<br>";
            $Success = false;
        }
    
        if (file_exists($ThumbPath))
            $Thumb = & new Image($ThumbPath);
        else
        {
            $ErrorMessageList[] = "Thumbnail Image doesn't exist: $ThumbPath<br>";
            $Success = false;
        }

        if (Debug()) {  echo "NewUrl=$NewUrl<br>";  }
        if (Debug()) {  echo "NewPreview=$NewPreview<br>";  }
        if (Debug()) {  echo "NewThumb=$NewThumb<br>";  }

        list($ResourceId, $FieldId) = sscanf($ImageName, "Img--%d--%d"); 

        if (Debug()) {  echo "ResourceId=$ResourceId, FieldId=$FieldId<br>";  }

        $FieldName = $DB->Query("SELECT FieldName from MetadataFields WHERE FieldId = '".$FieldId."'", "FieldName");
        $DBFieldName = preg_replace("/[^a-z0-9]/i", "", $FieldName);
    
        if (Debug()) {  echo "DBFieldName=$DBFieldName<br>";  }

        # get alttext and format directly from Resources
        $AltText = $DB->Query("Select ".$DBFieldName."AltText from Resources ".
                "WHERE ResourceId=$ResourceId", $DBFieldName."AltText"); 
        $Format = $DB->Query("Select ".$DBFieldName."Format from Resources ".
                "WHERE ResourceId=$ResourceId", $DBFieldName."Format"); 
    
        if (Debug()) {  echo "AltText=$AltText, Format=$Format<br>";  }
    
        # set ImageId for Resource
        $DB->Query("Update Resources set ".$DBFieldName." = ".$ImageId
                   ." WHERE ResourceId = ".$ResourceId);
    
        if (Debug()) {  echo "ImageX=".$Image->XSize()."<br>";  }
        if (Debug()) {  echo "ImageY=".$Image->YSize()."<br>";  }
        if (Debug()) {  echo "PreviewX=".$Preview->XSize()."<br>";  }
        if (Debug()) {  echo "PreviewY=".$Preview->YSize()."<br>";  }
        if (Debug()) {  echo "ThumbX=".$Thumb->XSize()."<br>";  }
        if (Debug()) {  echo "ThumbY=".$Thumb->YSize()."<br>";  }
    
        # create Image record
        $Query = "INSERT INTO Images (ImageId, Format, AltText, Width, Height, ".
                "PreviewWidth, PreviewHeight, ThumbnailWidth, ThumbnailHeight) VALUES (".
                "$ImageId, \"$Format\", \"$AltText\", ".$Image->XSize().", ".
                $Image->YSize().", ".$Preview->XSize().", ".$Preview->YSize().", ".
                $Thumb->XSize().", ".$Thumb->YSize().")";
    
        if (Debug()) {  echo "Query = $Query<br>";  }
    
        $DB->Query($Query);
    
        # rename the old files to the new files
        if (rename($ImagePath, $NewUrl) == false)
        {
            $ErrorMessageList[] = "Rename Failed: Src=$ImagePath, Dst=$NewUrl<br>";
            $Success = false;
        }
        if (rename($PreviewPath, $NewPreview) == false)
        {
            $ErrorMessageList[] = "Rename Failed: Src=$PreviewPath, Dst=$NewPreview<br.";
            $Success = false;
        }
        if (rename($ThumbPath, $NewThumb) == false)
        {
            $ErrorMessageList[] = "Rename Failed: Src=$ThumbPath, Dst=$NewThumb<br>";
            $Success = false;
        }
        $ImageId++;
    }
    
    # now drop the AltText and Format fields for each StillImage field
    $DB->Query("SELECT FieldName FROM MetadataFields WHERE FieldType = 'Still Image'");
    while ($FieldName = $DB->FetchField("FieldName"))
    {
        $DBFieldName = preg_replace("/[^a-z0-9]/i", "", $FieldName);
        $DBTwo->Query("ALTER TABLE Resources DROP COLUMN ".$DBFieldName."AltText");
        $DBTwo->Query("ALTER TABLE Resources DROP COLUMN ".$DBFieldName."Format");
        if (Debug()) {  echo "dropping column for $DBFieldName...<br>";  }
    }
    
    if (Debug()) {  print("IC: ...image file conversion completed.<br>\n");  }
    return $Success;
}    

function ConvertTypeTables($HostName, $UserName, $Password, $DBName, $AppName)
{
    global $ErrorMessageList;
    global $_SERVER;

    $Success = true;
    $DB =& new Database($UserName, $Password, $DBName, $HostName);
    $DBTwo =& new Database($UserName, $Password, $DBName, $HostName);
    
    # for controlled name and option fields
    $DB->Query("SELECT FieldName, FieldId FROM MetadataFields WHERE ".
                "FieldType = 'ControlledName' OR FieldType = 'Option'");
    
    while ($Entry = $DB->FetchNextRowArray())
    {
        $FieldName = $Entry["FieldName"];
        $FieldId = $Entry["FieldId"];

        if (Debug())
            echo "query=Select ControlledNameTypeId from ".
                "ControlledNameTypes where ControlledNameTypeName = \"".
                $FieldName."\"<br>";

        $TypeId = $DBTwo->Query("Select ControlledNameTypeId from ".
            "ControlledNameTypes where ControlledNameTypeName = \"".
            $FieldName."\"", "ControlledNameTypeId");

        if (Debug())
            echo "query=UPDATE ControlledNames SET ControlledNameTypeId = ".
                $FieldId." WHERE ControlledNameTypeId = $TypeId<br>";

        # this shouldn't happen unless the Type table is out of synch
        if (isset($TypeId))
            $DBTwo->Query("UPDATE ControlledNames SET ControlledNameTypeId = ".
                $FieldId." WHERE ControlledNameTypeId = $TypeId");
    }

    # change the field in ControlledNames
    $DB->Query("ALTER TABLE ControlledNames CHANGE COLUMN ".
                "ControlledNameTypeId FieldId INT"); 

    # should now be safe to drop the obsolete table
    $DB->Query("DROP TABLE ControlledNameTypes");

    # Get tree fields 
    $DB->Query("SELECT FieldName, FieldId FROM MetadataFields WHERE ".
                "FieldType = 'Tree'");
    
    while ($Entry = $DB->FetchNextRowArray())
    {
        $FieldName = $Entry["FieldName"];
        $FieldId = $Entry["FieldId"];

        if (Debug())
            echo "query=Select ClassificationTypeId from ".
            "ClassificationTypes where ClassificationTypeName = \"".
            $FieldName."\"<br>";

        $TypeId = $DBTwo->Query("Select ClassificationTypeId from ".
            "ClassificationTypes where ClassificationTypeName = \"".
            $FieldName."\"", "ClassificationTypeId");

        if (Debug())
            echo "UPDATE Classifications SET ClassificationTypeId = ".
                $FieldId." WHERE ClassificationTypeId = $TypeId<br>";
            
        # TypeId should be set unless type table is out of sync
        if (isset($TypeId))
            $DBTwo->Query("UPDATE Classifications SET ClassificationTypeId = ".
                $FieldId." WHERE ClassificationTypeId = $TypeId");
    }

    # update the field in the classifications table
    $DB->Query("ALTER TABLE Classifications CHANGE COLUMN ".
                "ClassificationTypeId FieldId INT"); 

    # now convert the default browsing field
    $FieldId = $DB->Query(
        "Select ClassificationTypeId from SystemConfiguration", 
        "ClassificationTypeId");

    $FieldName = $DB->Query(
        "Select ClassificationTypeName from ClassificationTypes ".
        "WHERE ClassificationTypeId = $FieldId", "ClassificationTypeName");

    $FieldId = $DB->Query(
        "Select FieldId from MetadataFields where FieldName = \"".
            $FieldName."\"", "FieldId");

    # store field id for browsing field id
    if (is_numeric($FieldId))
    {
        $DB->Query("Update SystemConfiguration Set ClassificationTypeId = ".
                    $FieldId);
    }
    
    # rename the field to something more meaningful
    $DB->Query("ALTER TABLE SystemConfiguration CHANGE COLUMN ".
                "ClassificationTypeId BrowsingFieldId INT");

    # convert the list of users to new browsing field id
    $DB->Query("Select UserId, ClassificationTypeId from APUsers");
    while ($Entry = $DB->FetchNextRowArray())
    {
        $FieldId = $Entry["ClassificationTypeId"];
        $UserId = $Entry["UserId"];

        if (Debug())
        {
            echo "UserId = $UserId, FieldId = $FieldId<br>";
        }

        # only process if this has been set
        if ($FieldId > 0)    
        {
            $FieldName = $DBTwo->Query(
                "Select ClassificationTypeName from ClassificationTypes ".
            "WHERE ClassificationTypeId = $FieldId", "ClassificationTypeName");

            if (Debug())
                echo "FieldName=$FieldName<br>";

            $FieldId = $DBTwo->Query(
                "Select FieldId from MetadataFields where FieldName = \"".
                $FieldName."\"", "FieldName");

            # now update the browsing field id to found medatafield id
            if (is_numeric($FieldId))
            {
                $UserToEdit = & new User($DB, 
                                intval($FoundUserArray["UserId"]));
                $UserToEdit->Set("ClassificationTypeId", $FieldId);
            }    
        }
    }

    # rename the field to something more meaningful
    $DB->Query("ALTER TABLE APUsers CHANGE COLUMN ".
                "ClassificationTypeId BrowsingFieldId INT");
    
    # should now be safe to drop the obsolete table
    $DB->Query("DROP TABLE ClassificationTypes");
}

function ConvertSavedSearches($HostName, $UserName, $Password, $DBName, $AppName)
{
    # search frequency mnemonics (must match those in SPT--SavedSearch.php)
    define("SEARCHFREQ_NEVER",     0);
    define("SEARCHFREQ_HOURLY",    1);
    define("SEARCHFREQ_DAILY",     2);
    define("SEARCHFREQ_WEEKLY",    3);
    define("SEARCHFREQ_BIWEEKLY",  4);
    define("SEARCHFREQ_MONTHLY",   5);
    define("SEARCHFREQ_QUARTERLY", 6);
    define("SEARCHFREQ_YEARLY",    7);
 
    # table for translating old search frequency strings into new search frequency mnemonics
    $FreqXlatTable = array(
            "Never"          => SEARCHFREQ_NEVER,
            "Hourly"         => SEARCHFREQ_HOURLY,
            "Once a day"     => SEARCHFREQ_DAILY,
            "Once a week"    => SEARCHFREQ_WEEKLY,
            "Bi-weekly"      => SEARCHFREQ_BIWEEKLY,
            "Once a month"   => SEARCHFREQ_MONTHLY,
            "Quarterly"      => SEARCHFREQ_QUARTERLY,
            "Yearly"         => SEARCHFREQ_YEARLY,
            );

    # load metadata field IDs
    $DB =& new Database($UserName, $Password, $DBName, $HostName);
    $DB->Query("SELECT FieldId, FieldName FROM MetadataFields");
    $FieldIds = $DB->FetchColumn("FieldId", "FieldName");
    $DB->Query("SELECT FieldId, FieldType FROM MetadataFields");
    $FieldTypes = $DB->FetchColumn("FieldType", "FieldId");

    # for each old saved search
    $InputDB =& new Database($UserName, $Password, $DBName, $HostName);
    $OutputDB =& new Database($UserName, $Password, $DBName, $HostName);
    $InputDB->Query("SELECT * FROM UserSearch");
    while ($OldSearch = $InputDB->FetchRow())
    {
        # add new saved search to database
        $UserId = $OldSearch["UserId"];
        if (isset($FreqXlatTable[$OldSearch["Frequency"]]))
        {
            $Freq = $FreqXlatTable[$OldSearch["Frequency"]];
        }
        else
        {
            $Freq = SEARCHFREQ_NEVER;
        }
        $OutputDB->Query("INSERT INTO SavedSearches"
                ." (SearchName, UserId, DateLastRun, Frequency) VALUES"
                ." ('".addslashes($OldSearch["SearchName"])."', ".$UserId.","
                ." '".$OldSearch."', ".$Freq.")");
        $NewSearchId = $OutputDB->LastInsertId("SavedSearches");

        # for each text search
        for ($Index = 1;  $Index <= 4;  $Index++)
        {
            # if text search has field and search string
            if (isset($OldSearch["SearchCat".$Index]) && strlen($OldSearch["SearchCat".$Index])
                    && isset($FieldIds[$OldSearch["SearchCat".$Index]]) && strlen($FieldIds[$OldSearch["SearchCat".$Index]])
                    && isset($OldSearch["SearchText".$Index]) && strlen($OldSearch["SearchText".$Index]))
            {
                # add new text parameter for new search
                $OutputDB->Query("INSERT INTO SavedSearchTextParameters"
                        ." (SearchId, FieldId, SearchText) VALUES"
                        ." (".$NewSearchId.","
                        ." ".$FieldIds[$OldSearch["SearchCat".$Index]].","
                        ." '".addslashes($OldSearch["SearchText".$Index])."')");
            }
        }

        # for each possible metadata field
        foreach ($FieldIds as $FieldName => $FieldId)
        {
            # if value parameter exists for field
            $DBFieldName = preg_replace("/[^a-z0-9]/i", "", $FieldName);
            if (isset($OldSearch[$DBFieldName]) 
                    && strlen($OldSearch[$DBFieldName]))
            {
                # for each value
                $Values = explode("|", $OldSearch[$DBFieldName]);
                foreach ($Values as $Value)
                {
                    # retrieve ID for this value
                    $ValueId = NULL;
                    switch ($FieldTypes[$FieldId])
                    {
                        case "Tree":
                            $ValueId = $DB->Query("SELECT ClassificationId"
                                        ." FROM Classifications"
                                        ." WHERE ClassificationName = '"
                                                .addslashes($Value)."'"
                                        ." AND FieldId = ".$FieldId,
                                    "ClassificationId");
                            break;
    
                        case "ControlledName":
                        case "Option":
                            $ValueId = $DB->Query("SELECT ControlledNameId"
                                        ." FROM ControlledNames"
                                        ." WHERE ControlledName = '"
                                                .addslashes($Value)."'"
                                        ." AND FieldId = ".$FieldId,
                                    "ControlledNameId");
                            break;
    
                        case "Flag":
                            if (($FieldName != "Release Flag") || ($Value == "0"))
                            {
                                $ValueId = $Value;
                            }
                            break;
                    }
    
                    # if value ID found
                    if ($ValueId !== NULL)
                    {
                        # add new value parameter for new search
                        $OutputDB->Query("INSERT INTO SavedSearchIdParameters"
                                ." (SearchId, FieldId, SearchValueId) VALUES"
                                ." (".$NewSearchId.", ".$FieldId.", ".$ValueId.")");
                    }
                }
            }
        }
    }
}

function ReadDirectory($Path, $PerlExp = "")
{
    # open directory
    $Dir = opendir($Path);

    # while file names left to read from directory
    $FileNames = array();
    while ($FileName = readdir($Dir))
    {
        # if name matches mask
        if (($PerlExp != "") && preg_match($PerlExp, $FileName))
        {
            # store file name in array
            $FileNames[count($FileNames)] = $FileName;
        }
    }

    # close directory
    closedir($Dir);

    # sort array of file names
    sort($FileNames);

    # return array to caller
    return $FileNames;
}

function SetUpDatabaseTables($HostName, $UserName, $Password, $DBName, $AppName, 
                             $AdminLogin, $AdminPassword, $AdminEmail)
{
    if (Debug()) {  print("IC: Setting up database tables...<br>\n");  }

    global $ErrorMessageList;

    # assume that everything will be okay
    $EverythingOkay = TRUE;

    # open connection to database server and select 
    $DB =& new Database($UserName, $Password, $DBName, $HostName);

    # if database selection failed
    # (this test needs to be improved -- need status reporting from Database object!)
    if (!is_object($DB))
    {
        # add error message
        $EverythingOkay = FALSE;
        $ErrorMessageList[] = "Unable to select database ${DBName} to create tables.";
    }
    else
    {
        # if we're supposed to upgrade
        if (file_exists("OLDVERSION"))
        {
            if (Debug()) {  print("IC: Install Type = UPGRADE<br>\n");  }
            
            # retrieve version of old install
            $InputFile = fopen("OLDVERSION", "r");
            $OldAppVersion = trim(fgets($InputFile, 256));
            fclose($InputFile);

            # for each available DB upgrade file
            $FileNames = ReadDirectory(".", "/SPT--Upgrade--.*--DBTables\.sql/");
            for ($Index = 0;  ($Index < count($FileNames)) && $EverythingOkay;  $Index++)
            {
                # parse out version number of upgrade file
                list($Dummy, $Dummy, $UpVer) = explode("--", $FileNames[$Index]);

                # if upgrade file version is greater than old SPT install version
                if (strcmp($UpVer, $OldAppVersion) > 0)
                {
                    # run commands from DB upgrade file
                    $EverythingOkay = ExecuteSQLCommands($DB, $FileNames[$Index]);
                }
            }

            # if upgrading to CWIS and no announcements in DB
            if (($AppName == "CWIS") && ($DB->Query("SELECT COUNT(*) AS AnnCount FROM Announcements", "AnnCount") == 0))
            {
                # add announcements to DB
                $DB->Query("INSERT INTO Announcements (AnnouncementId, AnnouncementHeading, AnnouncementText, DatePosted, PosterId) VALUES (1, 'Welcome', 'Welcome to your new online resource portal!  You may want to add some welcome text here.', NOW(), 1)");
                $DB->Query("INSERT INTO Announcements (AnnouncementId, AnnouncementHeading, AnnouncementText, DatePosted, PosterId) VALUES (2, 'News', 'News and announcements can also go in this area.', NOW(), 1)");
            }

            # if upgrading from a pre-1.1.4 version
            if (CanonicalVersion($OldAppVersion) < CanonicalVersion("1.1.4"))
            {
                # convert image file storage structure
                ConvertImageFiles($HostName, $UserName, $Password, $DBName, $AppName);
            }

            # if upgrading from a pre-1.1.8 version
            if (CanonicalVersion($OldAppVersion) < CanonicalVersion("1.1.8"))
            {
                if (Debug()) {  print("IC: updating user privileges<br>\n");  }
                
                # give all users ability to post resource comments and post to forums
                $DBTwo =& new Database($UserName, $Password, $DBName, $HostName);
                $DB->Query("SELECT UserId FROM APUsers");
                while ($UserId = $DB->FetchField("UserId"))
                {
                    $DBTwo->Query("INSERT INTO APUserPrivileges (UserId, Privilege) VALUES (".$UserId.", 9)");
                    $DBTwo->Query("INSERT INTO APUserPrivileges (UserId, Privilege) VALUES (".$UserId.", 10)");
                }
            }

            # if upgrading from a pre-1.2.0 version
            if (CanonicalVersion($OldAppVersion) < CanonicalVersion("1.2.0"))
            {
                # set field ordering to (hopefully) resemble the old order
                $DB->Query("UPDATE MetadataFields SET EditingOrderPosition = FieldId");
                $DB->Query("UPDATE MetadataFields SET DisplayOrderPosition = FieldId");
            }
            
            # if upgrading from a pre-1.2.3 version
            if (CanonicalVersion($OldAppVersion) < CanonicalVersion("1.2.3"))
            {
                # convert image file storage structure
                ConvertTypeTables($HostName, $UserName, $Password, $DBName, $AppName);
            }

            # if upgrading from a pre-1.4.0 version
            if (CanonicalVersion($OldAppVersion) < CanonicalVersion("1.4.0"))
            {
                # convert saved search storage structure
                ConvertSavedSearches($HostName, $UserName, $Password, $DBName, $AppName);
            }

            # if upgrading to CWIS
            if ($AppName == "CWIS")
            {
                if (Debug()) {  print("IC: setting CWIS-specific UI values<br>\n");  }
                
                # set active UI to CWIS
                $DB->Query("UPDATE SystemConfiguration SET DefaultActiveUI = 'SPTUI--CWIS'");
                
                # switch users from SPT distribution UIs to CWIS
                $DB->Query("UPDATE APUsers SET ActiveUI = 'SPTUI--CWIS'");
                
                # disable multiple interfaces
                $DB->Query("UPDATE SystemConfiguration SET AllowMultipleUIsEnabled = 0");
            }

            # clear last registration date
            $DB->Query("UPDATE SystemConfiguration SET LastRegisteredOn = NULL");
        }
        # else
        else
        {
            if (Debug()) {  print("IC: Install Type = NEW<br>\n");  }
            
            $DefaultSchemaFileName = ($AppName == "CWIS") ? "CWIS--CreateDefaultSchema.sql" : "SPT--CreateDefaultSchema.sql";

            # run commands to set up DB tables and schema from scratch
            $EverythingOkay = ExecuteSQLCommands($DB, "Axis--Session--CreateTables.sql");
            if ($EverythingOkay) {  $EverythingOkay = ExecuteSQLCommands($DB, "Axis--User--CreateTables.sql");  }
            if ($EverythingOkay) {  $EverythingOkay = ExecuteSQLCommands($DB, "Scout--SearchEngine--CreateTables.sql");  }
            if ($EverythingOkay) {  $EverythingOkay = ExecuteSQLCommands($DB, "SPT--CreateTables.sql");  }
            if ($EverythingOkay) {  $EverythingOkay = ExecuteSQLCommands($DB, $DefaultSchemaFileName);  }

            # load sample records into database
            if ($EverythingOkay) 
            {  
                if (Debug()) {  print("IC: Importing sample data...<br>\n");  }
                $SampleDataFileName = ($AppName == "CWIS") ? "CWIS--SampleData.sql" : "SPT--SampleData.sql";
                $EverythingOkay = ExecuteSQLCommands($DB, $SampleDataFileName);  
                if (Debug()) {  print("IC: ...sample data import completed.<br>\n");  }
            }
            
            # set active UI depending on install type
            $DB->Query("UPDATE SystemConfiguration SET DefaultActiveUI = '"
                       .(($AppName == "CWIS") ? "SPTUI--CWIS" : "SPTUI--Default")."'");

            # set up administrator login and e-mail
            # (overwrites values in APUsers for "sptadmin" login set up when tables are created)
            $DB->Query("UPDATE APUsers SET UserPassword = '".addslashes(crypt($AdminPassword))."' WHERE UserName = 'sptadmin'");
            $DB->Query("UPDATE APUsers SET EMail = '".addslashes($AdminEmail)."' WHERE UserName = 'sptadmin'");
            $DB->Query("UPDATE APUsers SET UserName = '".addslashes($AdminLogin)."' WHERE UserName = 'sptadmin'");
            $DB->Query("UPDATE SystemConfiguration SET AdminEmail = '".addslashes($AdminEmail)."'");
        }
    }

    # report back to caller whether everything was okay (TRUE if okay)
    if (Debug()) {  print("IC: ...database table setup completed.<br>\n");  }
    return $EverythingOkay;
}

function SetUpOaiConfiguration($HostName, $UserName, $Password, $DBName, $AppName)
{
    if (Debug()) {  print("IC: Setting up OAI configuration defaults...<br>\n");  }

    # assume that everything will be okay
    $EverythingOkay = TRUE;

    # open connection to database server and select 
    $DB =& new Database($UserName, $Password, $DBName, $HostName);

    # if OAI domain not set
    $CurrentVal = $DB->Query("SELECT OaiIdDomain FROM SystemConfiguration", "OaiIdDomain");
    if (strlen($CurrentVal) < 1)
    {
        # grab host name from our URL
        global $_SERVER;
        $HostName = $_SERVER["HTTP_HOST"];
        
        # split host name up into pieces in reverse order
        $Pieces = array_reverse(explode(".", $HostName));
        
        # if domain is .edu or two-letter
        if (($Pieces[0] == "edu") || (strlen($Pieces[0]) == 2))
        {
            # use top three levels of host name for OAI domain
            $OaiDomain = $Pieces[2].".".$Pieces[1].".".$Pieces[0];
        }
        else
        {
            # use top two levels of host name (if available) for OAI domain
            $OaiDomain = (isset($Pieces[1]) ? $Pieces[1]."." : "").$Pieces[0];
        }
        
        # set OAI domain in database
        $DB->Query("UPDATE SystemConfiguration SET OaiIdDomain = '".$OaiDomain."'");
    }
    
    # if OAI prefix not set
    $CurrentVal = $DB->Query("SELECT OaiIdPrefix FROM SystemConfiguration", "OaiIdPrefix");
    if (strlen($CurrentVal) < 1)
    {
        # use database name for OAI prefix
        $OaiPrefix = $DBName;
        
        # set OAI prefix in database
        $DB->Query("UPDATE SystemConfiguration SET OaiIdPrefix = '".$OaiPrefix."'");
    }

    # if OAI earliest date not set
    $CurrentVal = $DB->Query("SELECT OaiEarliestDate FROM SystemConfiguration", "OaiEarliestDate");
    if (strlen($CurrentVal) < 1)
    {
        # pick plausible value for OAI earliest date
        $OaiEarliestDate = "1990-01-01";
        
        # set OAI earliest date in database
        $DB->Query("UPDATE SystemConfiguration SET OaiEarliestDate = '".$OaiEarliestDate."'");
    }
    
    # if admin e-mail address is not set or looks invalid
    $CurrentVal = $DB->Query("SELECT AdminEmail FROM SystemConfiguration", "AdminEmail");
    if ((strlen($CurrentVal) < 1) || ($CurrentVal == "webmaster@localhost"))
    {
        # pick plausible value for contact e-mail address
        global $_SERVER;
        $AdminEmail = "webmaster@".$_SERVER["HTTP_HOST"];
        
        # set OAI earliest date in database
        $DB->Query("UPDATE SystemConfiguration SET AdminEmail = '".$AdminEmail."'");
    }

    # report back to caller whether everything was okay (TRUE if okay)
    if (Debug()) {  print("IC: ...OAI configuration default setup completed.<br>\n");  }
    return $EverythingOkay;
}

function SetUpRssConfiguration($HostName, $UserName, $Password, $DBName, $AppName, $AdminEmail)
{
    if (Debug()) {  print("IC: Setting up RSS configuration defaults...<br>\n");  }

    # assume that everything will be okay
    $EverythingOkay = TRUE;

    # open connection to database server and select 
    $DB =& new Database($UserName, $Password, $DBName, $HostName);

    # if we're not upgrading
    if (!file_exists("OLDVERSION"))
    {
        # set RSS contact e-mail addresses
        $DB->Query("UPDATE RSSConfiguration SET ManagingEditor = '".$AdminEmail."'");
        $DB->Query("UPDATE RSSConfiguration SET Webmaster = '".$AdminEmail."'");

        # set RSS image links
        if ($AppName == "CWIS")
        {
            $ImageLink = "SPTUI--CWIS/images/cwis_logo.gif";
            $ImageWidth = 69;
            $ImageHeight = 21;
        }
        else
        {
            $ImageLink = "SPTUI--Default/images/SPT_Logo_Tiny.gif";
            $ImageWidth = 96;
            $ImageHeight = 36;
        }
        global $_SERVER;
        $ImageLink = "http://".$_SERVER["SERVER_NAME"]
                .dirname($_SERVER["REQUEST_URI"])."/".$ImageLink;
        $DB->Query("UPDATE RSSConfiguration SET ImageLink = '".$ImageLink."'");
        $DB->Query("UPDATE RSSConfiguration SET ImageUrl = '".$ImageLink."'");
        $DB->Query("UPDATE RSSConfiguration SET ImageWidth = '".$ImageWidth."'");
        $DB->Query("UPDATE RSSConfiguration SET ImageHeight = '".$ImageHeight."'");
    }
    
    # report back to caller whether everything was okay (TRUE if okay)
    if (Debug()) {  print("IC: ...RSS configuration default setup completed.<br>\n");  }
    return $EverythingOkay;
}


# ----- MAIN -----------------------------------------------------------------

# assume install will go flawlessly
$EverythingOkay = TRUE;

# check if software already installed
$AlreadyInstalled = file_exists("include/INSTALLED");
if ($AlreadyInstalled) {  $EverythingOkay = FALSE;  }

# check if we're doing an upgrade
$DoingAnUpgrade = ($AlreadyInstalled) ? FALSE : file_exists("include/SPT--Config.php");
if (Debug() && $DoingAnUpgrade) {  print("IC: Doing an upgrade.<br>\n");  }

# grab version we're installing
$InputFile = fopen("VERSION", "r");
$AppVersion = trim(fgets($InputFile, 256));
fclose($InputFile);
if (Debug()) {  print("IC: Installing version $AppVersion (SPT) / ".$SPT_SPTCWISVersionMap[$AppVersion]." (CWIS).<br>\n");  }

# check environment
if ($EverythingOkay)
{
    $EverythingOkay = CheckEnvironment();
}

# check incoming values
if ($EverythingOkay)
{
    if (file_exists("include/SPT--Config.php"))
    {
        if (Debug()) {  print("IC: Existing config file found.<br>\n");  }
        include("include/SPT--Config.php");
        $F_DBHost     = isset($SPT_DBHost) ? $SPT_DBHost : $DBHost;
        $F_DBUserName = isset($SPT_DBUserName) ? $SPT_DBUserName : $DBUserName;
        $F_DBPassword = isset($SPT_DBPassword) ? $SPT_DBPassword : $DBPassword;
        $F_DBName     = isset($SPT_DBName) ? $SPT_DBName : $DBName;
        $F_AppName    = isset($_POST["F_AppName"]) ? trim($_POST["F_AppName"]) : "";
        $F_AdminLogin    = NULL;
        $F_AdminPassword = NULL;
        $F_AdminEmail    = NULL;
        if (strlen($F_AppName) < 1)
        {
            $F_AppName    = isset($_GET["F_AppName"]) ? trim($_GET["F_AppName"]) : "";
        }
    }
    else
    {
        if (Debug()) {  print("IC: No existing config file found -- checking config values.<br>\n");  }
        $F_DBHost        = isset($_POST["F_DBHost"]) ? trim($_POST["F_DBHost"]) : NULL;
        $F_DBUserName    = isset($_POST["F_DBUserName"]) ? trim($_POST["F_DBUserName"]) : NULL;
        $F_DBPassword    = isset($_POST["F_DBPassword"]) ? trim($_POST["F_DBPassword"]) : NULL;
        $F_DBName        = isset($_POST["F_DBName"]) ? trim($_POST["F_DBName"]) : NULL;
        $F_AppName       = isset($_POST["F_AppName"]) ? trim($_POST["F_AppName"]) : NULL;
        $F_AdminLogin    = isset($_POST["F_AdminLogin"]) ? trim($_POST["F_AdminLogin"]) : NULL;
        $F_AdminPassword = isset($_POST["F_AdminPassword"]) ? trim($_POST["F_AdminPassword"]) : NULL;
        $F_AdminEmail    = isset($_POST["F_AdminEmail"]) ? trim($_POST["F_AdminEmail"]) : NULL;
        $EverythingOkay = CheckValues(
                $F_DBHost, $F_DBUserName, $F_DBPassword, $F_DBName,
                $F_AdminLogin, $F_AdminPassword, $F_AdminEmail, $F_AppName);
        $DBHost     = $F_DBHost;
        $DBUserName = $F_DBUserName;
        $DBPassword = $F_DBPassword;
        $DBName     = $F_DBName;
    }
}

# assume SPT install if app name not set to CWIS
if (!isset($F_AppName) || ($F_AppName != "CWIS"))
{
    $F_AppName = "SPT";
}
if (Debug()) {  print("IC: Application name is \"$F_AppName\".<br>\n");  }

# set config file values (if not already set)
if ($EverythingOkay) 
{  
    $EverythingOkay = SetConfigFileValues(
            $F_DBHost, $F_DBUserName, $F_DBPassword, $F_DBName, $F_AppName);
}

# set up database
if ($EverythingOkay) 
{  
    $EverythingOkay = SetUpDatabase(
            $F_DBHost, $F_DBUserName, $F_DBPassword, $F_DBName);
}

# set up database tables
if ($EverythingOkay) 
{  
    $EverythingOkay = SetUpDatabaseTables(
            $F_DBHost, $F_DBUserName, $F_DBPassword, $F_DBName, 
            $F_AppName, $F_AdminLogin, $F_AdminPassword, $F_AdminEmail);
}

# set up OAI configuration defaults (if needed)
if ($EverythingOkay) 
{  
    $EverythingOkay = SetUpOaiConfiguration(
            $F_DBHost, $F_DBUserName, $F_DBPassword, $F_DBName, $F_AppName);
}

# set up RSS configuration defaults (if needed)
if ($EverythingOkay) 
{  
    $EverythingOkay = SetUpRssConfiguration(
            $F_DBHost, $F_DBUserName, $F_DBPassword, $F_DBName, $F_AppName, $F_AdminEmail);
}

# if install succeeded
if ($EverythingOkay) 
{  
    if (Debug()) {  print("IC: Install completed successfully!<br>\n");  }
    # mark as complete
    touch("include/INSTALLED");
}

if ($F_AppName == "CWIS")
{
    include("SPTUI--CWIS/SPT--InstallComplete.html");
}
else
{
    include("SPTUI--Default/SPT--InstallComplete.html");
}


?>
