﻿//
// HideElement: Sets the display of an element to "none". If
//     Id is set to true, it will get the element with that id.
//     Also sets the previous display attribute
//
// ShowElement: Sets the display of an element to "block" or the
//     previous display attribute. If Id is set to true, it will
//     get the element with that id.
//
// ToggleDisplay: Sets the display attribute of an element based
//     on its current display attribute. Sets it either to 'none'
//     or its past display attribute value.
//
// SelectInputText: Selects the text of an input box from Start
//     to End
//
// RemovePadding: Removes whitespace padding from the beginning
//     and end of a string.
//
// GetElement: Gets an element by its id
//
// GetElementsByClass: Gets all of the elements in a certain
//     class, node, and tag.
//
// GetElementsByTag: Gets all of the elements with a certain
//     tag.
//
// GetNextSibling: Gets the next non-whitespace sibling of an
//     element.
//
// GetKeyCode: Gets the Unicode value of the key pressed
//
// GetMousePosition: Gets the X/Y position of the mouse
//
// GetElementPosition: Gets the X/Y position of an element
//
// GetElementSize: Gets the height and width of an element
//
// SetElementPosition: Sets the X/Y position of an element
//
// SetElementClass: Sets the class of an element
//
//
// (c) 2006 and onward. Tim Baumgard
//

function HideElement(Element, Id){
    // Sets Id to false if not set
    Id = Id || false;

    // If an Id was passed instead of an element
    if (Id === true || !Element.nodeName){
        Element = new GetElement(Element);
    }

    // If the element isn't shown, exit
    if (Element.style.display == 'none'){
        return;
    }

    // Gets the current style attribute and sets it as an attribute
    Element.setAttribute("display", Element.style.display);

	// Hide the element
	Element.style.display = "none";
}

function ShowElement(Element, Id){
    // Sets Id to false if not set
    Id = Id || false;

    // If an Id was passed instead of an element
    if (Id === true || !Element.nodeName){
        Element = new GetElement(Element);
    }
	
	// If the element isn't hidden, exit
	if (Element.style.display != 'none'){
	    return;
	}
	
	// Gets any possible past display settings
	var Display = Element.getAttribute("display") ? 
	    Element.getAttribute("display") : 'block';
	
	// Show the element, using past display attribute if necessary
	if (Display === undefined || Display === ''){
	    Element.style.display = 'block';
	} else{
	    Element.style.display = Display;
	}
}

function ToggleDisplay(Element, Id){
    // Sets Id to false if not set
    Id = Id || false;

    // If an Id was passed instead of an element
    if (Id === true || !Element.nodeName){
        Element = new GetElement(Element);
    }

	// Gets any possible past display settings
	var Display = Element.getAttribute("display") ? 
	    Element.getAttribute("display") : 'block';

    // Gets the current style attribute and sets it as an attribute
    if (Element.style.display != 'none'){
        Element.setAttribute("display", Element.style.display);
    }

	// Hide or show the element
	if (Display != 'none' && Element.style.display != 'none'){
	    Element.style.display = 'none';
	} else{
	    Element.style.display = Display;
	}
}

function SelectInputText(Element, Start, End, Id){
    // Sets Id to false if not set
    Id = Id || false;

    // If an Id was passed instead of an element
    if (Id === true || typeof(Element) == "string"){
        Element = new GetElement(Element);
    }
    
    // Select the text. Depends on browser ability
    if (Element.createTextRange){
        var Range = Range.createTextRange();
        Range.moveStart('character', Element.value.length);
        Range.select();
    } else if (Element.setSelectionRange){
        Element.setSelectionRange(Start, End);
    }
    
    // Focus the element, which focuses the selection
    Element.focus();
}

function RemovePadding(Text){
    // Remove spaces before and after the text
    return Text.replace(/^\s*|\s*$/g, '');
}

function GetElement(ElementName){
    // Depending on the broswer, return the element
    if (document.getElementById){
        return document.getElementById(ElementName);
    } else if (document.all){
        return document.all[ElementName];
    } else if (document.layers){
        return documet.layers[ElementName];
    }
}

function GetElementsByClass(Class, Node, Tag){
    // Set values if they're not set
    if (Node === null || Node === undefined){
        Node = document;
    }
    if (Tag === null || Tag === undefined){
        Tag = '*';
    }
    
    // Initialize variables
    var ClassElements = [];
    var Elements      = Node.getElementsByTagName(Tag);
    var Pattern       = new RegExp('(^|\\s)'+ Class +'(\\s|$)');
    
    // Loop through the elements. Save the ones wanted
    for (i = 0, j = 0; i < Elements.length; i++){
        if (Pattern.test(Elements[i].className) &&
            Elements[i].nodeType == 1 && Elements[i].innerHTML !== null){
                ClassElements[j] = Elements[i];
                j++;
        }
    }
    return ClassElements;
}

function GetElementsByTag(Tag, Node){
    // Set values if they're not set
    if (Node === null){
        Node = document;
    }
    
    // Get the elements
    var Elements = Node.getElementsByTagName(Tag);
    
    return Elements;
}

function GetNextSibling(Element, Id){
    // Sets Id to false if not set
    Id = Id || false;

    // If an Id was passed instead of an element
    if (Id === true || !Element.nodeName){
        Element = new GetElement(Element);
    }
    
    // Loops through siblings, skipping whitespace
    while (Element.nodeType != 1){
        Element = Element.nextSibling;
    }
    
    // Return the sibling
    return Element;
}

function GetKeyCode(Event){
    // Get the proper event
    Event = Event || window.event;
    
    // Return the proper key code
	if (Event.keyCode){
	    KeyCode = Event.keyCode;
        return Event.keyCode;
	}
	else if (Event.which){
	    KeyCode = Event.which;
        return Event.which;
	}
}

function GetMousePosition(Event){
    // Get the proper event
	Event = Event || window.event;

    // Return the proper position, or (0,0)
	if (Event.pageX || Event.pageY){
		return {X:Event.pageX, Y:Event.pageY};
	} else if (Event.clientX || Event.clientY){
		return {X:Event.clientX + document.body.scrollLeft
		        + document.documentElement.scrollLeft,
		        Y:Event.clientY + document.body.scrollTop
			    + document.documentElement.scrollTop};
	} else{
	    return {X:0, Y:0};
	}
}

function GetElementPosition(Element, Id){
    // Sets Id to false if not set
    Id = Id || false;

    // If an Id was passed instead of an element
    if (Id === true || !Element.nodeName){
        Element = new GetElement(Element);
    }

    // Return the position, if available
    if (Element.offsetParent){
        var Top  = Element.offsetTop;
        var Left = Element.offsetLeft;
        while (Element = Element.offsetParent){
            Top  += Element.offsetTop;
            Left += Element.offsetLeft;
        }
        return {Top:Top, Left:Left};
    }
    
    // Else return (0,0)
    return {Top:0, Left:0};
}

function GetElementSize(Element, Id){
    // Sets Id to false if not set
    Id = Id || false;

    // If an Id was passed instead of an element
    if (Id === true || !Element.nodeName){
        Element = new GetElement(Element);
    }

    return {Height:Element.offsetHeight,
            Width:Element.offsetWidth};
}

function SetElementClass(Element, ClassName, Id){
    // Sets Id to false if not set
    Id = Id || false;

    // If an Id was passed instead of an element
    if (Id === true || !Element.nodeName){
        Element = new GetElement(Element);
    }
    
    // Set the class name. Depends on browser ability
    if (Element.className){
        Element.className = ClassName;
    } else{
        Element.setAttribute('class', ClassName);
    }
}

function SetElementPosition(Element, X, Y, Id){
    // Sets Id to false if not set
    Id = Id || false;

    // If an Id was passed instead of an element
    if (Id === true || !Element.nodeName){
        Element = new GetElement(Element);
    }

    // Set the position of the element    
    Element.style.left = (X * 1) + "px";
    Element.style.top  = (Y * 1) + "px";
}

// Set an event by id. Only need for compatibility. Events on the bottom
// are added on a "need" basis
function SetEventById(Element, Event, Function, Phase)
{
    // In case an element name was passed instead of an element
    if (!Element.nodeName)
    {
        Element = new GetElemById(Element);
    }
 
    if (!Phase) // In most cases, phase should be false anyway
    {
        Phase = false;   
    }
    
    if (addEventListener)
    {
        Element.addEventListener(Event, Function, Phase);
    }
    // Mouse events
    else if (Event == "click")
    {
        Element.onclick = Function;
    }
    else if (Event == "dblclick")
    {
        Element.ondblclick = Function;
    }
    else if (Event == "mouseup")
    {
        Element.onmouseup = Function;
    }
    else if (Event == "mousedown")
    {
        Element.onmousedown = Function;
    }
    else if (Event == "mousemove")
    {
        Element.onmousemove = Function;
    }
    else if (Event == "mouseover")
    {
        Element.onmouseover = Function;
    }
    // Key events
    else if (Event == "keydown")
    {
        Element.onkeydown = Function;
    }
    else if (Event == "keypress")
    {
        Element.onkeypress = Function;
    }
    else if (Event == "keyup")
    {
        Element.onkeyup = Function;
    }
}

    
    //-- Get the current size of the window
    function GetWindowSize()
    {
        if (self.innerWidth)
        {
            return {Height:self.innerHeight, Width:self.innerWidth};
        }
        else
        {
            return {Height:document.body.clientHeight, Width:document.body.clientWidth};
        }
    }
    
    //-- Get the window offsets
    function GetWindowOffsets()
    {
        if (window.innerHeight)
        {
            return {X:window.pageXOffset, Y:window.pageYOffset};
        }
        else if (document.documentElement && document.documentElement.scrollTop)
        {
            return {X:document.documentElement.scrollLeft, Y:document.documentElement.scrollTop};
        }
        else (document.body)
        {
            return {X:document.body.scrollLeft, Y:document.body.scrollTop};
        }
    }
    
    //-- Trim space from the beginning and end of a string
    function Trim(Text)
    {
        return Text.replace(/^\s*|\s*$/g, "");
    }
    
    //-- Convert "<" and ">" to their HTML equivalents
    function ConvertLTGT(Text)
    {
        return Text.replace(/</gi, "&lt;").replace(/>/gi, "&gt;");
    }
    
    //-- Convert HTML equivalents to "<" and ">"
    function ReconvertLTGT(Text)
    {
        return Text.replace(/&lt;/gi, "<").replace(/&gt;/gi, ">");
    }
    
    //-- Convert breaks to new lines
    function BreakToNewline(Text)
    {
        // Return converted text
        return Text.replace(/<br \/>/gi, "\n").replace(/<br\/>/gi, "\n").replace(/<br>/gi, "\n");
    }
    
    //-- Convert new lines to breaks
    function NewlineToBreak(Text)
    {
        // Return converted text
        return Text.replace(/\n/gi, "<br />");
    }
    
    // Get an element's custom attributes
    function GetAttribute(Element, Attr)
    {
        return ReconvertLTGT(GetElement(Element.id + "_Attr").innerHTML);
    }
 
    // Get an element's custom attributes
    function GetAttributes(Element)
    {        
        // Initialize the Attr variable to an empty array
        var Attr = [];
        
        // Get the attribute layer using the prefix and place in AttrLayer
        var AttrLayer = new GetElement(Element.id + "_Attr");

        // For every sibling, take the class name, use it as a key in the Attr array and use its HTML as the value
        for (i = 0; i < AttrLayer.childNodes.length; i++)
        {
            Attr[AttrLayer.childNodes[i].id.replace(Element.id+"_", "")] = ReconvertLTGT(AttrLayer.childNodes[i].innerHTML);
        }

        // Return the Attr array
        return Attr;
    }
    
    // Set an element's custom attributes
    function SetAttribute(Element, Attr, NewValue)
    {
        GetElement(Element.id + "_" + Attr).innerHTML = ConvertLTGT(NewValue);
    }

function SetClassEvent(Class, Event, Function, Node, Tag, Phase)
{
    var ClassElements = [];
    if (Node === undefined)
    {
        Node = document;
    }
    if (Tag === undefined)
    {
        Tag = '*';
    }
    var Elements = Node.getElementsByTagName(Tag);
    var Length = Elements.length;
    var Pattern = new RegExp('(^|\\s)' + Class + '(\\s|$)');
    for (i = 0, j = 0; i < Length; i++) 
    {
        if (Pattern.test(Elements[i].className) && Elements[i].nodeType == 1 && Elements[i].innerHTML !== null)
        {
            if (!Phase) // In most cases, phase should be false anyway
            {
                Phase = false;   
            }
    
            if (addEventListener)
            {
                Elements[i].addEventListener(Event, Function, Phase);
            }
            // Mouse events
            if (Event == "click")
            {
                Elements[i].onclick = Function;
            }
            else if (Event == "dblclick")
            {
                Elements[i].ondblclick = Function;
            }
            else if (Event == "mouseenter")
            {
                Elements[i].onmouseenter = Function;
            }
            else if (Event == "mouseleave")
            {
                Elements[i].onmouseleave = Function;
            }
            else if (Event == "mousemove")
            {
                Elements[i].onmousemove = Function;
            }
            else if (Event == "mouseover")
            {
                Elements[i].onmouseover = Function;
            }
            else if (Event == "keyup")
            {
                Element.onkeyup = Function;
            }
        }
    }
}